<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;

class StockLedgerValidationException extends StockLedgerException
{
    /**
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param BinLocation $binLocation
     * @return self
     */
    public static function binLocationNotMapped(
        ArticleDetail $articleDetail,
        Warehouse $warehouse,
        BinLocation $binLocation
    ) {
        $messageArguments = [
            $articleDetail->getNumber(),
            $binLocation->getCode(),
            $warehouse->getDisplayName(),
        ];
        $exception = new self(vsprintf(
            'Cannot relocate variant #%1$s from bin location "%2$s" in warehouse "%3$s", because the bin location is currently not mapped to the variant.',
            $messageArguments
        ));
        $exception->setSnippetName('stock_manager_validation_exception/bin_location_not_mapped');
        $exception->setSnippetArguments($messageArguments);

        return $exception;
    }

    /**
     * @param ArticleDetail $articleDetail
     * @param Warehouse $warehouse
     * @param BinLocation $binLocation
     * @param int $stockChange
     * @param int $currentStock
     * @return self
     */
    public static function stockChangeExceedsCurrentStock(
        ArticleDetail $articleDetail,
        Warehouse $warehouse,
        BinLocation $binLocation,
        $stockChange,
        $currentStock
    ) {
        $messageArguments = [
            abs($stockChange),
            $binLocation->getCode(),
            $currentStock,
            $articleDetail->getNumber(),
            $warehouse->getDisplayName(),
        ];
        $exception = new self(vsprintf(
            'The absolute stock change %1$d of source bin location "%2$s" exceeds the current physical stock %3$d of variant #%4$s on that bin location in warehouse "%5$s".',
            $messageArguments
        ));
        $exception->setSnippetName('stock_manager_validation_exception/stock_change_exceeds_current_stock');
        $exception->setSnippetArguments($messageArguments);

        return $exception;
    }

    /**
     * @param BinLocation $binLocation
     * @param Warehouse $warehouse
     * @return self
     */
    public static function warehouseMismatch(
        BinLocation $binLocation,
        Warehouse $warehouse
    ) {
        $messageArguments = [
            $binLocation->getCode(),
            $warehouse->getDisplayName(),
        ];
        $exception = new self(vsprintf(
            'The given bin location "%1$s" is not part of warehouse "%$2s" that is managed by the active stock manager.',
            $messageArguments
        ));
        $exception->setSnippetName('stock_manager_validation_exception/warehouse_mismatch');
        $exception->setSnippetArguments($messageArguments);

        return $exception;
    }
}
