<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockLedger;

use Enlight_Hook;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\NegativeStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\PositiveStockChangeList;

class StockUpdaterService implements StockUpdater, Enlight_Hook
{
    /**
     * @var StockLedgerService
     */
    private $stockLedgerService;

    /**
     * @param StockLedgerService $stockLedgerService
     */
    public function __construct(StockLedgerService $stockLedgerService)
    {
        $this->stockLedgerService = $stockLedgerService;
    }

    /**
     * @inheritdoc
     */
    public function recordOrderDetailShippedChange(
        ArticleDetail $articleDetail,
        OrderDetail $orderDetail,
        $stockChanges,
        User $user = null
    ) {
        if ($stockChanges instanceof NegativeStockChangeList) {
            return $this->stockLedgerService->recordSoldStock(
                $articleDetail,
                $orderDetail,
                $stockChanges,
                '',
                $user
            );
        }
        if ($stockChanges instanceof PositiveStockChangeList) {
            return $this->stockLedgerService->recordSoldStockCorrection(
                $articleDetail,
                $orderDetail,
                $stockChanges,
                '',
                $user
            );
        }

        throw self::createUnsupportedInstanceOfChangeListException(__METHOD__, $stockChanges);
    }

    /**
     * @inheritdoc
     */
    public function recordReturnShipmentItemReturnedChange(
        ArticleDetail $articleDetail,
        ReturnShipmentItem $returnShipmentItem,
        $stockChanges,
        User $user = null
    ) {
        if ($stockChanges instanceof PositiveStockChangeList) {
            return $this->stockLedgerService->recordReturnedStock(
                $articleDetail,
                $returnShipmentItem,
                $stockChanges,
                '',
                $user
            );
        }
        if ($stockChanges instanceof NegativeStockChangeList) {
            return $this->stockLedgerService->recordReturnedStockCorrection(
                $articleDetail,
                $returnShipmentItem,
                $stockChanges,
                '',
                $user
            );
        }

        throw self::createUnsupportedInstanceOfChangeListException(__METHOD__, $stockChanges);
    }

    /**
     * @inheritdoc
     */
    public function recordReturnShipmentItemWriteOffChange(
        ArticleDetail $articleDetail,
        ReturnShipmentItem $returnShipmentItem,
        $stockChanges,
        User $user = null
    ) {
        if ($stockChanges instanceof NegativeStockChangeList) {
            return $this->stockLedgerService->recordWriteOff(
                $articleDetail,
                $stockChanges,
                $returnShipmentItem,
                '',
                $user
            );
        }
        if ($stockChanges instanceof PositiveStockChangeList) {
            return $this->stockLedgerService->recordWriteOffCorrection(
                $articleDetail,
                $stockChanges,
                $returnShipmentItem,
                '',
                $user
            );
        }

        throw self::createUnsupportedInstanceOfChangeListException(__METHOD__, $stockChanges);
    }

    /**
     * @inheritdoc
     */
    public function recordArticleDetailStockChange(
        ArticleDetail $articleDetail,
        $stockChanges,
        User $user = null
    ) {
        if ($stockChanges instanceof NegativeStockChangeList) {
            return $this->stockLedgerService->recordOutgoingStock(
                $articleDetail,
                $stockChanges,
                '',
                $user
            );
        }
        if ($stockChanges instanceof PositiveStockChangeList) {
            return $this->stockLedgerService->recordIncomingStock(
                $articleDetail,
                $stockChanges,
                null,
                '',
                $user
            );
        }

        throw self::createUnsupportedInstanceOfChangeListException(__METHOD__, $stockChanges);
    }

    /**
     * @param string $method
     * @param mixed $passedObject
     * @return \InvalidArgumentException
     */
    private static function createUnsupportedInstanceOfChangeListException($method, $passedObject)
    {
        return new \InvalidArgumentException(sprintf(
            'Method %s only supports instances of %s and %s as third parameter, but %s was passed.',
            $method,
            PositiveStockChangeList::class,
            NegativeStockChangeList::class,
            is_object($passedObject) ? ('instance of ' . get_class($passedObject)) : gettype($passedObject)
        ));
    }
}
