<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockMovement;

use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonCommon\Classes\Util\OrderDetailUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\NegativeStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\PositiveStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockUpdater;

abstract class AbstractStockLocation
{
    /**
     * @var StockLedgerService
     */
    protected $stockLedgerService;

    /**
     * @var StockUpdater
     */
    protected $stockUpdater;

    /**
     * @param StockLedgerService $stockLedgerService
     * @param StockUpdater $stockUpdater
     */
    public function __construct(StockLedgerService $stockLedgerService, StockUpdater $stockUpdater)
    {
        $this->stockLedgerService = $stockLedgerService;
        $this->stockUpdater = $stockUpdater;
    }

    /**
     * @param ArticleDetail $articleDetail
     * @param int $quantity
     * @return PositiveStockChangeList|NegativeStockChangeList
     * @throws StockMovementException
     */
    abstract protected function createStockChangeListForArticleDetail(ArticleDetail $articleDetail, $quantity);

    /**
     * Returns the total stock of the ArticleDetail on this AbstractLocation.
     *
     * @param ArticleDetail $articleDetail
     * @return int
     * @throws StockMovementException
     */
    abstract protected function calculateTotalStockOfArticleDetail(ArticleDetail $articleDetail);

    /**
     * Moves the entire stock of the $articleDetail on this AbstractLocation to another location given by $destination.
     *
     * @param ArticleDetail $articleDetail
     * @param self $destination
     */
    public function moveEntireStockOfArticleDetailToLocation(ArticleDetail $articleDetail, self $destination)
    {
        $this->stockLedgerService->performWithLockForArticleDetail(
            $articleDetail,
            function () use ($articleDetail, $destination) {
                $movedQuantity = $this->calculateTotalStockOfArticleDetail($articleDetail);
                if ($movedQuantity === 0) {
                    return;
                }
                $this->applyStockChangeToArticleDetail($articleDetail, -1 * $movedQuantity);
                $destination->applyStockChangeToArticleDetail($articleDetail, 1 * $movedQuantity);
            }
        );
    }

    /**
     * Moves a give $quantity of stock of the $articleDetail on this AbstractLocation to another location given by
     * $destination.
     *
     * @param ArticleDetail $articleDetail
     * @param self $destination
     * @param int $quantity
     */
    public function moveStockOfArticleDetailToLocation(ArticleDetail $articleDetail, self $destination, $quantity)
    {
        if ($quantity === 0) {
            return;
        }
        $this->stockLedgerService->performWithLockForArticleDetail(
            $articleDetail,
            function () use ($articleDetail, $destination, $quantity) {
                $this->applyStockChangeToArticleDetail($articleDetail, -1 * $quantity);
                $destination->applyStockChangeToArticleDetail($articleDetail, $quantity);
            }
        );
    }

    /**
     * Changes the stock of the $articleDetail on this AbstractLocation to the absolute value of $quantity.
     *
     * @param ArticleDetail $articleDetail
     * @param int $quantity
     */
    public function setStockOfArticleDetail(ArticleDetail $articleDetail, $quantity)
    {
        $this->stockLedgerService->performWithLockForArticleDetail(
            $articleDetail,
            function () use ($articleDetail, $quantity) {
                $quantityChange = $quantity - $this->calculateTotalStockOfArticleDetail($articleDetail);
                if ($quantityChange !== 0) {
                    $this->applyStockChangeToArticleDetail($articleDetail, $quantityChange);
                }
            }
        );
    }

    /**
     * Changes the stock of the $articleDetail on this AbstractLocation by the relative value of $quantity.
     *
     * @param ArticleDetail $articleDetail
     * @param int $quantity
     */
    public function changeStockOfArticleDetail(ArticleDetail $articleDetail, $quantity)
    {
        if ($quantity === 0) {
            return;
        }
        $this->stockLedgerService->performWithLockForArticleDetail(
            $articleDetail,
            function () use ($articleDetail, $quantity) {
                $this->applyStockChangeToArticleDetail($articleDetail, $quantity);
            }
        );
    }

    /**
     * Changes the stock of the $articleDetail on this AbstractLocation as a result of a shippedQuantity change of the
     * $orderDetail.
     *
     * @param OrderDetail $orderDetail
     * @param int $quantity
     */
    public function shipStockForOrderDetail(OrderDetail $orderDetail, $quantity)
    {
        if ($quantity === 0) {
            return;
        }

        $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
        if (!$articleDetail) {
            return;
        }

        $this->stockLedgerService->performWithLockForArticleDetail(
            $articleDetail,
            function () use ($orderDetail, $quantity, $articleDetail) {
                $changeListSourceLocation = $this->createStockChangeListForArticleDetail(
                    $articleDetail,
                    -1 * $quantity
                );
                $this->stockUpdater->recordOrderDetailShippedChange(
                    $articleDetail,
                    $orderDetail,
                    $changeListSourceLocation
                );
            }
        );
    }

    /**
     * @param ArticleDetail $articleDetail
     * @param int $quantity
     */
    private function applyStockChangeToArticleDetail(ArticleDetail $articleDetail, $quantity)
    {
        $changeList = $this->createStockChangeListForArticleDetail($articleDetail, $quantity);
        $this->stockUpdater->recordArticleDetailStockChange($articleDetail, $changeList);
    }
}
