<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockMovement;

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockUpdater;

class BinLocationStockLocation extends AbstractStockLocation
{
    /**
     * @var StockChangeListFactory
     */
    protected $stockChangeListFactory;

    /**
     * @var BinLocation
     */
    protected $binLocation;

    /**
     * @param StockLedgerService $stockLedgerService
     * @param StockUpdater $stockUpdater
     * @param StockChangeListFactory $stockChangeListFactory
     * @param BinLocation $binLocation
     */
    public function __construct(
        StockLedgerService $stockLedgerService,
        StockUpdater $stockUpdater,
        StockChangeListFactory $stockChangeListFactory,
        BinLocation $binLocation
    ) {
        parent::__construct($stockLedgerService, $stockUpdater);
        $this->stockChangeListFactory = $stockChangeListFactory;
        $this->stockUpdater = $stockUpdater;
        $this->binLocation = $binLocation;
    }

    /**
     * @inheritdoc
     */
    protected function createStockChangeListForArticleDetail(ArticleDetail $articleDetail, $quantity)
    {
        if (!$this->binLocation->isNullBinLocation() && $quantity < 0) {
            $availableStock = $this->calculateTotalStockOfArticleDetail($articleDetail);
            if ($availableStock + $quantity < 0) {
                throw StockMovementException::notEnoughStockOnBinLocation();
            }
        }

        return $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
            $this->binLocation,
            $quantity
        );
    }

    /**
     * @inheritdoc
     */
    protected function calculateTotalStockOfArticleDetail(ArticleDetail $articleDetail)
    {
        $mappingsForArticleDetail = $this->binLocation->getArticleDetailBinLocationMappings()->filter(
            function (ArticleDetailBinLocationMapping $mapping) use ($articleDetail) {
                return $mapping->getArticleDetail()->getId() === $articleDetail->getId();
            }
        );

        // There is by definition only one or no mapping for an article detail
        $mappingForArticleDetail = $mappingsForArticleDetail->first();
        if ($mappingForArticleDetail) {
            return $mappingForArticleDetail->getStock();
        }

        return 0;
    }
}
