<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockMovement;

use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\AbstractValidationException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\ArticleDetailStockInfoProvider;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockUpdater;

final class StockLocationFactoryService
{
    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var StockChangeListFactory
     */
    private $stockChangeListFactory;

    /**
     * @var StockUpdater
     */
    private $stockUpdater;

    /**
     * @var StockLedgerService
     */
    private $stockLedgerService;

    /**
     * @var ArticleDetailStockInfoProvider
     */
    private $articleDetailStockInfoProvider;

    /**
     * @param ModelManager $entityManager
     * @param StockChangeListFactory $stockChangeListFactory
     * @param StockUpdater $stockUpdater
     * @param StockLedgerService $stockLedgerService
     * @param ArticleDetailStockInfoProvider $articleDetailStockInfoProvider
     */
    public function __construct(
        ModelManager $entityManager,
        StockChangeListFactory $stockChangeListFactory,
        StockUpdater $stockUpdater,
        StockLedgerService $stockLedgerService,
        ArticleDetailStockInfoProvider $articleDetailStockInfoProvider
    ) {
        $this->entityManager = $entityManager;
        $this->stockChangeListFactory = $stockChangeListFactory;
        $this->stockUpdater = $stockUpdater;
        $this->stockLedgerService = $stockLedgerService;
        $this->articleDetailStockInfoProvider = $articleDetailStockInfoProvider;
    }

    /**
     * @param array $stockLocationArray
     * @param string $path
     * @throws AbstractValidationException
     */
    public static function validateStockLocationArray(array $stockLocationArray, $path = '')
    {
        self::validateKeyCombination(
            $stockLocationArray,
            [
                ['warehouse'],
                ['binLocation'],
            ],
            $path
        );

        if (isset($stockLocationArray['warehouse'])) {
            self::validateWarehouseStockLocationArray($stockLocationArray, $path);
        }

        if (isset($stockLocationArray['binLocation'])) {
            $binLocation = $stockLocationArray['binLocation'];
            $subPath = $path . '.binLocation';
            ParameterValidator::assertIsArray($binLocation, $subPath);
            self::validateKeyCombination(
                $binLocation,
                [
                    [
                        'id',
                    ],
                    [
                        'warehouseId',
                        'code',
                    ],
                    [
                        'warehouseCode',
                        'code',
                    ],
                ],
                $subPath
            );

            if (isset($binLocation['id'])) {
                ParameterValidator::assertIsInteger($binLocation['id'], $subPath . '.id');
            }
            if (isset($binLocation['warehouseId'])) {
                ParameterValidator::assertIsInteger($binLocation['warehouseId'], $subPath . '.warehouseId');
            }
        }
    }

    /**
     * @param array $stockLocationArray
     * @param string $path
     * @throws AbstractValidationException
     */
    public static function validateWarehouseStockLocationArray(array $stockLocationArray, $path = '')
    {
        $warehouse = $stockLocationArray['warehouse'];
        $subPath = $path . '.warehouse';
        ParameterValidator::assertIsArray($warehouse, $subPath);
        self::validateKeyCombination(
            $warehouse,
            [
                ['id'],
                ['code'],
            ],
            $subPath
        );
        if (isset($warehouse['id'])) {
            ParameterValidator::assertIsInteger($warehouse['id'], $subPath . '.id');
        }
    }

    /**
     * Checks whether the keys of the $array are one of the allowed combinations in $allowedKeyCombinations.
     *
     * Keys that do not appear in any array of $allowedKeyCombinations are ignored.
     *
     * @param array $array assoc array
     * @param string[][] $allowedKeyCombinations
     * @param string $parameterName
     * @throws CustomValidationException
     */
    private static function validateKeyCombination(array $array, array $allowedKeyCombinations, $parameterName)
    {
        // Find out whether the keys of $array are valid combination given by $allowedKeyCombinations
        $relevantKeys = array_unique(array_merge(...$allowedKeyCombinations));
        $allowedKeyCombinations = array_map(
            function (array $allowedKeyCombination) {
                sort($allowedKeyCombination);

                return $allowedKeyCombination;
            },
            $allowedKeyCombinations
        );
        $usedCombination = array_intersect(array_keys($array), $relevantKeys);
        sort($usedCombination);
        if (in_array($usedCombination, $allowedKeyCombinations)) {
            return;
        }

        // Build an understandable error message
        $allowedCombinations = array_map(
            function (array $keyCombination) {
                return implode(' and ', $keyCombination);
            },
            $allowedKeyCombinations
        );
        $msg = sprintf(
            'The parameters "%s" can only appear in the following combinations: Either (%s)',
            implode('", "', $relevantKeys),
            implode(') or (', $allowedCombinations)
        );
        throw new CustomValidationException(
            $parameterName,
            '[object]',
            $msg
        );
    }

    /**
     * @param array $stockLocationArray
     * @return AbstractStockLocation
     * @throws StockMovementException
     */
    public function createStockLocationFromArray(array $stockLocationArray)
    {
        if (isset($stockLocationArray['binLocation'])) {
            if (isset($stockLocationArray['binLocation']['warehouseCode'])) {
                /** @var Warehouse|null $warehouse */
                $warehouse = $this->entityManager->getRepository(Warehouse::class)->findOneBy([
                    'code' => $stockLocationArray['binLocation']['warehouseCode'],
                ]);
                if (!$warehouse) {
                    throw StockMovementException::warehouseNotFound();
                }
                unset($stockLocationArray['binLocation']['warehouseCode']);
                $stockLocationArray['binLocation']['warehouse'] = $warehouse;
            }
            /** @var null|BinLocation $binLocation */
            $binLocation = $this->entityManager->getRepository(BinLocation::class)->findOneBy($stockLocationArray['binLocation']);
            if (!$binLocation) {
                throw StockMovementException::binLocationNotFound();
            }

            return new BinLocationStockLocation(
                $this->stockLedgerService,
                $this->stockUpdater,
                $this->stockChangeListFactory,
                $binLocation
            );
        }
        if (isset($stockLocationArray['warehouse'])) {
            /** @var Warehouse|null $warehouse */
            $warehouse = $this->entityManager->getRepository(Warehouse::class)->findOneBy($stockLocationArray['warehouse']);
            if (!$warehouse) {
                throw StockMovementException::warehouseNotFound();
            }

            return new WarehouseStockLocation(
                $this->stockLedgerService,
                $this->stockUpdater,
                $this->stockChangeListFactory,
                $this->articleDetailStockInfoProvider,
                $warehouse
            );
        }

        throw new \InvalidArgumentException(sprintf(
            (
                'The array passed to %s does not represent a valid stock location because it neither has the key '
                . '"binLocation" nor "warehouse" set. Please apply the method %s to the corresponding array before '
                . 'passing it to this method.'
            ),
            __METHOD__,
            __CLASS__ . '::validateStockLocationArray'
        ));
    }
}
