<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockReservation;

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\OrderStockReservation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\BinLocationStockChange;

interface StockReservation
{
    /**
     * First removes any stock reservations of the given $orderDetail in the given $warehouse. If the remaining/open
     * quantity (quantity - shipped) is greater zero, new stock reservations for that remaining quantity are created.
     * Finally, the cached reserved stock of the respective article detail is updated and the created stock reservations
     * are returned.
     *
     * @param Warehouse $warehouse
     * @param OrderDetail $orderDetail
     * @return OrderStockReservation[]
     */
    public function reserveOpenQuantityForOrderDetail(Warehouse $warehouse, OrderDetail $orderDetail);

    /**
     * Converts the given $stockChanges into stock reservations for the given $articleDetail and flushes them into the
     * database.
     *
     * @param ArticleDetail $articleDetail
     * @param OrderDetail $orderDetail
     * @param BinLocationStockChange[] $stockChanges
     * @return OrderStockReservation[]
     */
    public function reserveStockForStockChanges(
        ArticleDetail $articleDetail,
        OrderDetail $orderDetail,
        array $stockChanges
    );

    /**
     * Clears reserved stock by deleting the reservation from the database, before updating (reducing) the cached
     * reserved stock of the warehouse, which the given $reservation belongs. If the given $reservedStock entity is NOT
     * managed, only its object association to the bin location article detail is removed.
     *
     * @param OrderStockReservation $reservation
     */
    public function clearStockReservation(OrderStockReservation $reservation);

    /**
     * Finds any stock reservations for the given $orderDetail in the given $warehouse and removes them from the
     * database. Finally, the cached reserved stock of the respective article detail is updated.
     *
     * @param Warehouse $warehouse
     * @param OrderDetail $orderDetail
     */
    public function clearStockReservationsForOrderDetail(Warehouse $warehouse, OrderDetail $orderDetail);
}
