<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockReservation;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\OrderStockReservation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonPickwareERP\Components\DerivedPropertyUpdater\DerivedPropertyUpdater;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\BinLocationStockChange;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\ZeroStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;

class StockReservationService implements StockReservation, Enlight_Hook
{
    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var StockChangeListFactory
     */
    protected $stockChangeListFactory;

    /**
     * @var DerivedPropertyUpdater
     */
    protected $derivedPropertyUpdater;

    /**
     * @var StockLedgerService
     */
    protected $stockLedgerService;

    /**
     * @var OrderDetailQuantityCalculator
     */
    protected $orderDetailQuantityCalculator;

    /**
     * @param ModelManager $entityManager
     * @param StockChangeListFactory $stockChangeListFactory
     * @param DerivedPropertyUpdater $derivedPropertyUpdater
     * @param StockLedgerService $stockLedgerService
     * @param OrderDetailQuantityCalculator $orderDetailQuantityCalculator
     */
    public function __construct(
        $entityManager,
        StockChangeListFactory $stockChangeListFactory,
        DerivedPropertyUpdater $derivedPropertyUpdater,
        StockLedgerService $stockLedgerService,
        OrderDetailQuantityCalculator $orderDetailQuantityCalculator
    ) {
        $this->entityManager = $entityManager;
        $this->stockChangeListFactory = $stockChangeListFactory;
        $this->derivedPropertyUpdater = $derivedPropertyUpdater;
        $this->stockLedgerService = $stockLedgerService;
        $this->orderDetailQuantityCalculator = $orderDetailQuantityCalculator;
    }

    /**
     * @inheritdoc
     */
    public function reserveOpenQuantityForOrderDetail(Warehouse $warehouse, OrderDetail $orderDetail)
    {
        // Check the order detail for a real article detail whose stock is managed
        /** @var ArticleDetail|null $articleDetail */
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $orderDetail->getArticleNumber(),
        ]);
        if (!$articleDetail || ($articleDetail->getAttribute() && $articleDetail->getAttribute()->getPickwareStockManagementDisabled())) {
            return [];
        }

        // Remove all existing reservations for the order detail
        $this->clearStockReservationsForOrderDetail($warehouse, $orderDetail);

        // Reserve any remaining quantity of the order detail
        $remainingQuantity = $this->orderDetailQuantityCalculator->calculateRemainingQuantityToShip($orderDetail);
        if ($remainingQuantity > 0) {
            $stockChangeList = $this->stockChangeListFactory->createStockChangeList(
                $warehouse,
                $articleDetail,
                -1 * $remainingQuantity
            );
            $binLocationStockChanges = $stockChangeList->getBinLocationStockChanges();
        } else {
            $binLocationStockChanges = [];
        }
        $stockReservations = $this->reserveStockForStockChanges(
            $articleDetail,
            $orderDetail,
            $binLocationStockChanges
        );

        // Save stock reservations and update the cache
        $this->entityManager->flush($stockReservations);
        $this->derivedPropertyUpdater->recalculateReservedStockForArticleDetailInWarehouse($articleDetail, $warehouse);

        return $stockReservations;
    }

    /**
     * @inheritdoc
     */
    public function reserveStockForStockChanges(
        ArticleDetail $articleDetail,
        OrderDetail $orderDetail,
        array $stockChanges
    ) {
        $stockReservations = array_map(
            function (BinLocationStockChange $stockChange) use ($articleDetail, $orderDetail) {
                $binLocationMapping = $this->entityManager->getRepository(ArticleDetailBinLocationMapping::class)->findOneBy([
                    'articleDetail' => $articleDetail,
                    'binLocation' => $stockChange->getBinLocation(),
                ]);

                $stockReservation = new OrderStockReservation(
                    $binLocationMapping,
                    $orderDetail,
                    abs($stockChange->getStockChange())
                );
                $this->entityManager->persist($stockReservation);

                return $stockReservation;
            },
            $stockChanges
        );
        $this->entityManager->flush($stockReservations);

        return $stockReservations;
    }

    /**
     * @inheritdoc
     */
    public function clearStockReservation(OrderStockReservation $reservation)
    {
        // Always remove the stock reservation from the associated bin
        $reservation->getArticleDetailBinLocationMapping()->getStockReservations()->removeElement($reservation);
        if (!$this->entityManager->contains($reservation)) {
            return;
        }

        $warehouse = $reservation->getArticleDetailBinLocationMapping()->getBinLocation()->getWarehouse();
        $articleDetail = $reservation->getArticleDetailBinLocationMapping()->getArticleDetail();

        // Remove entity from the database and update the cache
        $this->entityManager->remove($reservation);
        $this->entityManager->flush($reservation);
        $this->derivedPropertyUpdater->recalculateReservedStockForArticleDetailInWarehouse(
            $articleDetail,
            $warehouse
        );

        // Create Zero Stock Change
        $stockChanges = ZeroStockChangeList::fromBinLocation($reservation->getArticleDetailBinLocationMapping()->getBinLocation());

        $this->stockLedgerService->updateBinLocationMappings($articleDetail, $stockChanges);
    }

    /**
     * @inheritdoc
     */
    public function clearStockReservationsForOrderDetail(Warehouse $warehouse, OrderDetail $orderDetail)
    {
        // Check the order detail for a real article detail
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $orderDetail->getArticleNumber(),
        ]);
        if (!$articleDetail) {
            return;
        }

        // Find all stock reservations for the order detail
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select('reservedStock')
            ->from(OrderStockReservation::class, 'reservedStock')
            ->innerJoin('reservedStock.articleDetailBinLocationMapping', 'articleDetailBinLocationMapping')
            ->innerJoin('articleDetailBinLocationMapping.binLocation', 'binLocation')
            ->where('reservedStock.orderDetail = :orderDetail')
            ->andWhere('binLocation.warehouse = :warehouse')
            ->setParameters([
                'orderDetail' => $orderDetail,
                'warehouse' => $warehouse,
            ]);
        $stockReservations = $builder->getQuery()->getResult();
        if (count($stockReservations) === 0) {
            return;
        }

        // Remove all found stock reservations and update the cache
        foreach ($stockReservations as $stockReservation) {
            $stockReservation->getArticleDetailBinLocationMapping()->getStockReservations()->removeElement($stockReservation);
            $this->entityManager->remove($stockReservation);
        }
        $this->entityManager->flush($stockReservations);
        $this->derivedPropertyUpdater->recalculateReservedStockForArticleDetailInWarehouse($articleDetail, $warehouse);
    }
}
