<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\StockValuation;

use DateTime;
use Doctrine\DBAL\Connection;
use Enlight_Config;
use LogicException;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\StockValuation\Report;
use Shopware\CustomModels\ViisonPickwareERP\StockValuation\ReportPurchase;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;

class StockValuationService
{
    /**
     * @var Connection
     */
    private $database;

    /**
     * @var ModelManager
     */
    private $entityManager;

    /**
     * @var Enlight_Config
     */
    private $pluginConfig;

    public function __construct(ModelManager $entityManager, Connection $database, Enlight_Config $pluginConfig)
    {
        $this->database = $database;
        $this->entityManager = $entityManager;
        $this->pluginConfig = $pluginConfig;
    }

    /**
     * @param Report $report
     * @throws StockValuationException
     */
    public function performNextReportGenerationStep(Report $report)
    {
        if ($report->getId() === null) {
            throw new LogicException(sprintf(
                'Please persist and flush the Entity %s before generating a report.',
                get_class($report)
            ));
        }

        if ($report->isGenerated()) {
            throw new \LogicException('Cannot regenerate a report');
        }

        switch ($report->getGenerationStep()) {
            case Report::GENERATION_STEP_REPORT_CREATED:
                $report->updateUntilDateBasedOnGenerationDate(new DateTime('now'));
                $this->removeFailedAndPreviewReports($report);
                $this->ensureNoYoungerReportExists($report);
                break;
            case Report::GENERATION_STEP_REPORT_PREPARED:
                $this->calculateStocks($report);
                break;
            case Report::GENERATION_STEP_STOCKS_CALCULATED:
                $this->calculatePurchases($report);
                break;
            case Report::GENERATION_STEP_PURCHASES_CALCULATED:
                $this->calculateAveragePurchasePrice($report);
                break;
            case Report::GENERATION_STEP_AVERAGE_PURCHASE_PRICE_CALCULATED:
                $this->rateStock($report);
                break;
            case Report::GENERATION_STEP_STOCK_RATED:
                $this->persistReport($report);
                break;
        }

        $report->markCurrentGenerationStepAsFinished();

        $this->entityManager->flush($report);
    }

    /**
     * @param Report $report
     * @throws StockValuationException
     */
    private function ensureNoYoungerReportExists(Report $report)
    {
        $youngerReports = $this->entityManager
            ->createQuery(
                'SELECT report
                FROM Shopware\\CustomModels\\ViisonPickwareERP\\StockValuation\\Report AS report
                JOIN report.warehouse AS warehouse
                WHERE
                    report.preview = 0
                    AND report.generated = 1
                    AND report.untilDate >= :untilDate
                    AND warehouse.id = :warehouseId'
            )
            ->setParameters(
                [
                    'untilDate' => $report->getUntilDate(),
                    'warehouseId' => $report->getWarehouse()->getId(),
                ]
            )
            ->setMaxResults(1)
            ->getResult();
        if (count($youngerReports)) {
            throw StockValuationException::youngerReportExists();
        }
    }

    /**
     * Removes all existing previews and failed reports but the given one.
     *
     * @param Report $report
     */
    private function removeFailedAndPreviewReports(Report $report)
    {
        $this->database->executeUpdate('TRUNCATE TABLE `pickware_erp_stock_valuation_temp_stocks`');
        $this->database->executeUpdate('TRUNCATE TABLE `pickware_erp_stock_valuation_temp_purchases`');
        $this->database->executeUpdate(
            'DELETE FROM `pickware_erp_stock_valuation_reports`
            WHERE (`preview` = 1 OR `generated` = 0)
                AND `id` != :reportId',
            ['reportId' => $report->getId()]
        );
    }

    /**
     * Calculates the stocks of all article details at the report's untilDate
     *
     * @param Report $report
     */
    private function calculateStocks(Report $report)
    {
        $this->database->executeUpdate(
            'INSERT INTO `pickware_erp_stock_valuation_temp_stocks`
                (`reportId`, `articleDetailId`, `stock`)
            SELECT
                :reportId AS `reportId`,
                `articleDetail`.`id` AS `articleDetailId`,
                COALESCE(`latestStockLedgerEntry`.`newStock`, 0) AS `stock`
            FROM `s_articles_details` AS `articleDetail`
            LEFT JOIN (
                SELECT
                    MAX(`stockLedgerEntry`.`id`) AS `id`,
                    `stockLedgerEntry`.`articleDetailId` AS `articleDetailId`
                FROM `pickware_erp_stock_ledger_entries` AS `stockLedgerEntry`
                WHERE `stockLedgerEntry`.`warehouseId` = :warehouseId
                    AND `stockLedgerEntry`.`created` < :untilDate
                GROUP BY `stockLedgerEntry`.`articleDetailId`
            ) AS `latestStockLedgerEntryId` ON `latestStockLedgerEntryId`.`articleDetailId` = `articleDetail`.`id`
            LEFT JOIN `pickware_erp_stock_ledger_entries` `latestStockLedgerEntry`
                ON `latestStockLedgerEntry`.`id` = `latestStockLedgerEntryId`.`id`
            LEFT JOIN `s_articles_attributes` `articleDetailAttribute`
                ON `articleDetail`.`id` = `articleDetailAttribute`.`articledetailsID`
            WHERE `articleDetailAttribute`.`pickware_stock_management_disabled` != 1',
            [
                'reportId' => $report->getId(),
                'warehouseId' => $report->getWarehouse()->getId(),
                'untilDate' => $report->getUntilDate()->format('c'),
            ]
        );
    }

    /**
     * Calculates all purchases processed within a report's period. This period is defined as the time span between the
     * last report's and the given report's until dates.
     *
     * @param Report $report
     */
    private function calculatePurchases(Report $report)
    {
        $this->database->executeUpdate(
            'INSERT INTO `pickware_erp_stock_valuation_temp_purchases` (
                `reportId`,
                `articleDetailId`,
                `quantity`,
                `purchasePriceNet`,
                `date`,
                `type`,
                `averagePurchasePriceNet`,
                `purchaseStockLedgerEntryId`,
                `carryOverReportRowId`
            ) SELECT
                :reportId AS `reportId`,
                -- This SELECT finds all stock movements that are actually purchases processed within the report period
                -- periode
                `articleDetail`.`id` AS `articleDetailId`,
                `stockLedgerEntry`.`changeAmount` AS `quantity`,
                IF (
                    :purchasePriceMode = "net",
                    COALESCE(`stockLedgerEntry`.`purchasePrice`, `articleDetail`.`purchaseprice`),
                    -- In case the purchase price is stored as GROSS, the purchase price is converted to net. Currently
                    -- the purchase price is rounded to 2 digits. We may loose precision here, but if the user wants to
                    -- have more precise values he should store its purchase prices in net.
                    ROUND(COALESCE(
                        `stockLedgerEntry`.`purchasePrice`,
                        `articleDetail`.`purchaseprice`
                    ) / (1 + `tax`.`tax` / 100), 2)
                ) AS `purchasePriceNet`,
                `stockLedgerEntry`.`created` AS `date`,
                :purchaseTypePurchase AS `type`,
                NULL AS `averagePurchasePriceNet`,
                `stockLedgerEntry`.`id` AS `purchaseStockLedgerEntryId`,
                NULL AS `carryOverReportRowId`
            FROM `pickware_erp_stock_ledger_entries` AS `stockLedgerEntry`
            INNER JOIN `s_articles_details` AS `articleDetail` ON `articleDetail`.`id` = `stockLedgerEntry`.`articleDetailId`
            INNER JOIN `s_articles` AS `article` ON `article`.`id` = `articleDetail`.`articleID`
            LEFT JOIN `s_core_tax` AS `tax` ON `article`.`taxID` = `tax`.`id`
            LEFT JOIN (
                SELECT
                    MAX(`report`.`untilDate`) AS `untilDate`,
                    `reportRow`.`articleDetailId` AS `articleDetailId`
                FROM `pickware_erp_stock_valuation_report_rows` AS `reportRow`
                INNER JOIN `pickware_erp_stock_valuation_reports` AS `report` ON `report`.`id` = `reportRow`.`reportId`
                WHERE `report`.`preview` = 0
                    AND `report`.`generated` = 1
                    AND `report`.`warehouseId` = :warehouseId
                GROUP BY `reportRow`.`articleDetailId`
            ) AS `latestReport` ON `latestReport`.`articleDetailId` = `articleDetail`.`id`
            WHERE
                `stockLedgerEntry`.`type` IN (
                    :stockLedgerTypePurchase,
                    :stockLedgerTypeInitialization,
                    :stockLedgerTypeIncoming,
                    :stockLedgerTypeStocktake
                )
                AND `stockLedgerEntry`.`changeAmount` > 0
                AND `stockLedgerEntry`.`created` < :untilDate
                AND `stockLedgerEntry`.`warehouseId` = :warehouseId
                AND (
                     `stockLedgerEntry`.`created` >= `latestReport`.`untilDate` OR `latestReport`.`untilDate` IS NULL
                )
            UNION ALL SELECT
                :reportId AS `reportId`,
                -- This select finds the last report for each article detail an adds its result as a "carry-over"
                -- purchase, which works the same as a normal purchase.
                `reportRow`.`articleDetailId` AS `articleDetailId`,
                GREATEST(`reportRow`.`stock`, 0) AS `quantity`,
                IF(
                    `reportRow`.`stock` <= 0,
                    0,
                    `reportRow`.`valuationNet` / `reportRow`.`stock`
                ) AS `purchasePrice`,
                -- For the date of the carry-over purchase a second is subtracted for usability reasons
                DATE_ADD(`report`.`untilDate`, INTERVAL -1 SECOND) AS `date`,
                :purchaseTypeCarryOver AS `type`,
                `reportRow`.`averagePurchasePriceNet` AS `averagePurchasePriceNet`,
                NULL AS purchaseStockLedgerEntryId,
                `reportRow`.`id` AS `carryOverReportRowId`
            FROM `pickware_erp_stock_valuation_report_rows` AS `reportRow`
            INNER JOIN `pickware_erp_stock_valuation_reports` AS `report` On `report`.`id` = `reportRow`.`reportId`
            WHERE `reportRow`.`id` IN (
                SELECT
                    MAX(`reportRow`.`id`)
                FROM `pickware_erp_stock_valuation_report_rows` AS `reportRow`
                INNER JOIN `pickware_erp_stock_valuation_reports` AS `report` On `report`.`id` = `reportRow`.`reportId`
                WHERE `report`.`preview` = 0
                    AND `report`.`generated` = 1
                    AND `report`.`warehouseId` = :warehouseId
                GROUP BY `articleDetailId`
            );',
            [
                'reportId' => $report->getId(),
                'warehouseId' => $report->getWarehouse()->getId(),
                'untilDate' => $report->getUntilDate()->format('c'),
                'purchasePriceMode' => $this->pluginConfig->get('purchasePriceMode'),
                'stockLedgerTypePurchase' => StockLedgerEntry::TYPE_PURCHASE,
                'stockLedgerTypeInitialization' => StockLedgerEntry::TYPE_INITIALIZATION,
                'stockLedgerTypeIncoming' => StockLedgerEntry::TYPE_INCOMING,
                'stockLedgerTypeStocktake' => StockLedgerEntry::TYPE_STOCKTAKE,
                'purchaseTypePurchase' => ReportPurchase::TYPE_PURCHASE,
                'purchaseTypeCarryOver' => ReportPurchase::TYPE_CARRY_OVER,
            ]
        );
    }

    /**
     * Calculates for all articles their respective average purchase price within the report period.
     *
     * This value is necessary for FiFo, LiFo and Average
     * @param Report $report
     */
    private function calculateAveragePurchasePrice(Report $report)
    {
        $this->database->executeUpdate(
            'UPDATE pickware_erp_stock_valuation_temp_stocks stock
            LEFT JOIN (
                SELECT
                    `averagePurchasePriceNet`,
                    `articleDetailId`
                FROM `pickware_erp_stock_valuation_report_rows`
                WHERE `id` IN (
                    SELECT
                        MAX(`reportRow`.`id`)
                    FROM `pickware_erp_stock_valuation_report_rows` AS `reportRow`
                    INNER JOIN `pickware_erp_stock_valuation_reports` AS `report` On `report`.`id` = `reportRow`.`reportId`
                    WHERE `report`.`preview` = 0
                        AND `report`.`generated` = 1
                )
            ) AS `latestReportRow` ON `latestReportRow`.`articleDetailId` = `stock`.`articleDetailId`
            LEFT JOIN (
                SELECT
                    ROUND(SUM(quantity * purchasePriceNet) / SUM(quantity), 2) AS averagePurchasePrice,
                    articleDetailId
                FROM pickware_erp_stock_valuation_temp_purchases
                WHERE `reportId` = :reportId
                GROUP BY articleDetailId
            ) AS purchaseAverage ON purchaseAverage.articleDetailId = stock.articleDetailId
            INNER JOIN `s_articles_details` AS `articleDetail` ON `articleDetail`.`id` = `stock`.`articleDetailId`
            INNER JOIN `s_articles` AS `article` ON `article`.`id` = `articleDetail`.`articleID`
            LEFT JOIN `s_core_tax` AS `tax` ON `article`.`taxID` = `tax`.`id`
            SET stock.`averagePurchasePriceNet` = COALESCE(
                purchaseAverage.averagePurchasePrice,
                `latestReportRow`.`averagePurchasePriceNet`,
                ROUND(IF(
                    :purchasePriceMode = "net",
                    `articleDetail`.`purchaseprice`,
                    `articleDetail`.`purchaseprice` / (1 + `tax`.`tax` / 100)
                ), 2)
            )
            WHERE `stock`.`reportId` = :reportId',
            [
                'reportId' => $report->getId(),
                'purchasePriceMode' => $this->pluginConfig->get('purchasePriceMode'),
            ]
        );
    }

    /**
     * Rates the previously calculated stock according to the report`s valuation method.
     *
     * @param Report $report
     */
    private function rateStock(Report $report)
    {
        if ($report->getMethod() === Report::METHOD_AVERAGE) {
            $this->rateStockByAverage($report);
        } else {
            $this->rateStockByFifoOrLifo($report);
        }
    }

    /**
     * @param Report $report
     */
    private function rateStockByAverage(Report $report)
    {
        $this->database->executeUpdate(
            'UPDATE pickware_erp_stock_valuation_temp_stocks stock
            SET stock.valuationNet = stock.stock * stock.averagePurchasePriceNet
            WHERE `reportId` = :reportId',
            [
                'reportId' => $report->getId(),
            ]
        );
        $this->database->executeUpdate(
            'UPDATE pickware_erp_stock_valuation_temp_purchases purchase
            SET purchase.quantityUsedForValuation = purchase.quantity
            WHERE `reportId` = :reportId',
            [
                'reportId' => $report->getId(),
            ]
        );
        $this->database->executeUpdate(
            'UPDATE `pickware_erp_stock_valuation_temp_stocks` AS `stock`
            LEFT JOIN (
                SELECT
                    SUM(quantity) AS totalQuantity,
                    articleDetailId
                FROM pickware_erp_stock_valuation_temp_purchases
                WHERE `reportId` = :reportId
                GROUP BY articleDetailId
            ) AS purchaseSum ON purchaseSum.articleDetailId = stock.articleDetailId
            SET stock.surplusStock = stock.stock - IFNULL(purchaseSum.totalQuantity, 0),
                stock.surplusPurchasePriceNet = stock.averagePurchasePriceNet
            WHERE
                IFNULL(purchaseSum.totalQuantity, 0) < stock.stock
                AND `reportId` = :reportId',
            [
                'reportId' => $report->getId(),
            ]
        );
    }

    /**
     * @param Report $report
     */
    private function rateStockByFifoOrLifo(Report $report)
    {
        $this->database->executeUpdate(
        /** @lang MySQL */
            'DROP PROCEDURE IF EXISTS `rateStock`;'
        );

        // Skip the rating of all products that do not have stock. This has a great performance impact for shops with
        // many articles that have a stock of 0. Products with a stock of 0 are skipped in the next step.
        $this->database->executeUpdate(
            'UPDATE `pickware_erp_stock_valuation_temp_stocks`
            SET `valuationNet` = 0.0
            WHERE
                `stock` = 0
                AND `reportId` = :reportId',
            [
                'reportId' => $report->getId(),
            ]
        );

        $purchaseSorting = ($report->getMethod() === Report::METHOD_LIFO) ? 'ASC' : 'DESC';
        $this->database->executeUpdate(
            sprintf(
                /** @lang MySQL */
                'CREATE PROCEDURE `rateStock`()
                BEGIN
                    DECLARE `articleDetailId` INTEGER DEFAULT NULL;
                    DECLARE `stock` INTEGER DEFAULT 0;
                    DECLARE `purchaseQuantity` INT;
                    DECLARE `purchasePriceNet` DECIMAL(10,2);
                    DECLARE `purchaseId` INT;

                    DECLARE `valuationNet` DECIMAL(10,2);
                    DECLARE `currentArticleDetailId` INTEGER DEFAULT NULL;
                    DECLARE `currentStock` INTEGER DEFAULT NULL;

                    DECLARE `finished` INTEGER DEFAULT 0;
                    DECLARE `dataCursor` CURSOR FOR (
                        SELECT
                            `stock`.`articleDetailId`,
                            `stock`.`stock`,
                            `purchase`.`id`,
                            `purchase`.`quantity`,
                            `purchase`.`purchasePriceNet`
                        FROM `pickware_erp_stock_valuation_temp_stocks` AS `stock`
                        LEFT JOIN `pickware_erp_stock_valuation_temp_purchases` `purchase`
                            ON
                                `purchase`.`articleDetailId` = `stock`.`articleDetailId`
                                AND `purchase`.`reportId` = `stock`.`reportId`
                        WHERE
                            `stock`.`stock` > 0
                            AND `stock`.`reportId` = :reportId
                        ORDER BY
                            `stock`.`articleDetailId`,
                            `purchase`.`date` %s,
                            -- This is an additional sorter to make the stock valuation reports reproducible
                            `purchase`.`purchasePriceNet`
                    );
                    DECLARE CONTINUE HANDLER
                        FOR NOT FOUND SET `finished` = 1;

                    OPEN `dataCursor`;

                    -- Loop through all article details joined together (left) with their purchases
                    `purchasesLoop`: WHILE `stock` IS NOT NULL DO
                        FETCH `dataCursor` INTO `articleDetailId`, `stock`, `purchaseId`, `purchaseQuantity`, `purchasePriceNet`;

                        IF (NOT(`articleDetailId` <=> `currentArticleDetailId`) OR `finished` = 1) THEN
                            -- Next articleDetail or no articleDetails left
                            IF (`currentArticleDetailId` IS NOT NULL AND `currentStock` >= 0) THEN
                                -- We can only determine a valuation for a stock >= 0. If the stock is < 0 the valuation
                                -- will stay null what means that the valuation is not determinable.
                                UPDATE `pickware_erp_stock_valuation_temp_stocks` `stock`
                                SET
                                    `stock`.`valuationNet` = `valuationNet`,
                                    -- Save the `currentStock` in case it is not 0. That means not all stock is covered by
                                    -- purchases. The surplus stock will be valued after this loop
                                    `stock`.`surplusStock` = `currentStock`
                                WHERE `stock`.`articleDetailId` = `currentArticleDetailId`;
                            END IF;

                            IF (`finished` = 1) THEN
                                -- No further record found
                                LEAVE `purchasesLoop`;
                            END IF;

                            -- Reset variables for new articleDetail
                            SET `currentArticleDetailId` = `articleDetailId`;
                            SET `currentStock` = `stock`;
                            SET `valuationNet` = 0.0;
                        END IF;

                        -- Negative stocks cannot be valuated. Just loop to the next article.
                        IF (`stock` < 0) THEN
                            ITERATE `purchasesLoop`;
                        END IF;

                        IF (`purchaseId` IS NULL) THEN
                            -- No purchases for article detail: Set the valuation to 0.0. (If there is stock, the whole stock
                            -- will be counted and valued as surplus stock.)
                            SET `valuationNet` = 0.0;
                            ITERATE `purchasesLoop`;
                        END IF;

                        UPDATE `pickware_erp_stock_valuation_temp_purchases` AS `purchase`
                        SET `purchase`.`quantityUsedForValuation` = LEAST(`currentStock`, `purchaseQuantity`)
                        WHERE `purchase`.`id` = `purchaseId`;

                        SET `valuationNet` = `valuationNet` + LEAST(`currentStock`, `purchaseQuantity`) * `purchasePriceNet`;
                        SET `currentStock` = GREATEST(`currentStock` - `purchaseQuantity`, 0);
                    END WHILE;

                    CLOSE `dataCursor`;

                    -- Adjust the stock valuation for articleDetails with surplus stock (means: more stock than the sum of
                    -- purchases)
                    UPDATE `pickware_erp_stock_valuation_temp_stocks` AS `stock`
                    LEFT JOIN `pickware_erp_stock_valuation_temp_purchases` AS `lastCarryOver`
                        ON `lastCarryOver`.`articleDetailId` = `stock`.`articleDetailId`
                            AND `lastCarryOver`.`type` = :purchaseTypeCarryOver
                            AND `lastCarryOver`.`reportId` = `stock`.`reportId`
                    INNER JOIN `s_articles_details` AS `articleDetail` ON `articleDetail`.id = `stock`.`articleDetailId`
                    INNER JOIN `s_articles` AS `article` ON `article`.`id` = `articleDetail`.`articleID`
                    LEFT JOIN `s_core_tax` AS `tax` ON `article`.`taxID` = `tax`.`id`
                    SET
                        `stock`.`surplusPurchasePriceNet` = COALESCE(
                            `lastCarryOver`.`averagePurchasePriceNet`,
                            ROUND(IF(
                                :purchasePriceMode = "net",
                                `articleDetail`.`purchaseprice`,
                                `articleDetail`.`purchaseprice` / (1 + `tax`.`tax` / 100)
                            ), 2)
                        ),
                        `stock`.`valuationNet` = `stock`.`valuationNet` + `stock`.`surplusStock` * COALESCE(
                            `lastCarryOver`.`averagePurchasePriceNet`,
                            ROUND(IF(
                                :purchasePriceMode = "net",
                                `articleDetail`.`purchaseprice`,
                                `articleDetail`.`purchaseprice` / (1 + `tax`.`tax` / 100)
                            ), 2)
                        )
                    WHERE
                        `stock`.`surplusStock` != 0
                        AND `stock`.`reportId` = :reportId;
                END;',
                $purchaseSorting
            ),
            [
                'purchasePriceMode' => $this->pluginConfig->get('purchasePriceMode'),
                'purchaseTypeCarryOver' => ReportPurchase::TYPE_CARRY_OVER,
                'reportId' => $report->getId(),
            ]
        );

        $this->database->executeUpdate('CALL rateStock()');
    }

    /**
     * This persists the report that was calculated in temporary tables into persistent tables.
     *
     * @param Report $report
     */
    private function persistReport(Report $report)
    {
        $this->database->executeUpdate(
            'INSERT INTO `pickware_erp_stock_valuation_report_rows` (
                `reportId`,
                `articleDetailId`,
                `articleDetailNumber`,
                `articleDetailName`,
                `stock`,
                `valuationNet`,
                `valuationGross`,
                `taxRate`,
                `averagePurchasePriceNet`,
                `surplusStock`,
                `surplusPurchasePriceNet`
            ) SELECT
                `stock`.`reportId` AS `reportId`,
                `stock`.`articleDetailId` AS `articleDetailId`,
                `articleDetail`.`ordernumber` AS `articleDetailNumber`,
                IF(
                    `variantDescription`.`description` IS NOT NULL,
                    CONCAT(`article`.`name`, " - ", `variantDescription`.`description`),
                    `article`.`name`
                ) AS `articleDetailName`,
                `stock`.`stock` AS `stock`,
                `stock`.`valuationNet` AS `valuationNet`,
                ROUND(`stock`.`valuationNet` * (1 + `tax`.`tax` / 100), 2) AS `valuationGross`,
                `tax`.`tax` AS `taxRate`,
                `stock`.`averagePurchasePriceNet` AS `averagePurchasePriceNet`,
                `stock`.`surplusStock` AS `surplusStock`,
                `stock`.`surplusPurchasePriceNet` AS `surplusPurchasePriceNet`
            FROM `pickware_erp_stock_valuation_temp_stocks` `stock`
            INNER JOIN `s_articles_details` AS `articleDetail` ON `articleDetail`.`id` = `stock`.`articleDetailId`
            INNER JOIN `s_articles` AS `article` ON `article`.`id` = `articleDetail`.`articleID`
            LEFT JOIN `s_core_tax` AS `tax` ON `article`.`taxID` = `tax`.`id`
            LEFT JOIN (
                SELECT
                    articleDetail.id AS articleDetailId,
                    GROUP_CONCAT(
                        `option`.name
                        ORDER BY `group`.position
                        SEPARATOR " "
                    ) AS `description`
                FROM s_articles_details articleDetail
                INNER JOIN s_article_configurator_option_relations optionRelation ON articleDetail.id = optionRelation.article_id
                INNER JOIN s_article_configurator_options `option` ON `option`.id = optionRelation.option_id
                INNER JOIN s_article_configurator_groups `group` ON `group`.id = `option`.group_id
                GROUP BY articleDetail.id
            ) AS variantDescription ON variantDescription.articleDetailId = `articleDetail`.id
            WHERE `stock`.`reportId` = :reportId
            GROUP BY `stock`.`articleDetailId`;',
            [
                'reportId' => $report->getId(),
                'purchasePriceMode' => $this->pluginConfig->get('purchasePriceMode'),
            ]
        );

        $this->database->executeUpdate(
            'INSERT INTO `pickware_erp_stock_valuation_report_purchases` (
                `reportRowId`,
                `date`,
                `purchasePriceNet`,
                `quantity`,
                `quantityUsedForValuation`,
                `type`,
                `purchaseStockLedgerEntryId`,
                `carryOverReportRowId`
            ) SELECT
                `stockValuationReportRow`.`id`,
                `purchase`.`date`,
                `purchase`.`purchasePriceNet`,
                `purchase`.`quantity`,
                `purchase`.`quantityUsedForValuation`,
                `purchase`.`type`,
                `purchase`.`purchaseStockLedgerEntryId`,
                `purchase`.`carryOverReportRowId`
            FROM `pickware_erp_stock_valuation_temp_purchases` AS `purchase`
            INNER JOIN `pickware_erp_stock_valuation_report_rows` AS `stockValuationReportRow`
                ON `stockValuationReportRow`.`articleDetailId` = `purchase`.`articleDetailId`
                AND `stockValuationReportRow`.`reportId` = :reportId
            WHERE `purchase`.`reportId` = :reportId',
            [
                'reportId' => $report->getId(),
            ]
        );

        $this->database->executeUpdate('TRUNCATE TABLE `pickware_erp_stock_valuation_temp_stocks`');
        $this->database->executeUpdate('TRUNCATE TABLE `pickware_erp_stock_valuation_temp_purchases`');
    }
}
