<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Supplier;

use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier;
use Shopware\Models\Order\Number as NumberRange;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\NumberRangeNotFoundException;
use Shopware_Components_Snippet_Manager as SnippetManager;

class SupplierNumberGeneratorService implements SupplierNumberGenerator, Enlight_Hook
{
    const START_NUMBER = 10000;
    const NUMBER_RANGE_DESCRIPTION = 'Lieferantennummer';
    const NUMBER_RANGE_NAME = 'supplier_number';

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var SnippetManager
     */
    protected $snippetManager;

    /**
     * @param ModelManager $entityManager
     * @param SnippetManager $snippetManager
     */
    public function __construct($entityManager, $snippetManager)
    {
        $this->entityManager = $entityManager;
        $this->snippetManager = $snippetManager;
    }

    /**
     * @inheritdoc
     */
    public function isSupplierNumberAvailable($supplierNumber)
    {
        $supplier = $this->entityManager->getRepository(Supplier::class)->findOneBy([
            'number' => $supplierNumber,
        ]);

        return $supplier === null;
    }

    /**
     * @inheritdoc
     */
    public function generateSupplierNumber()
    {
        $supplierNumberRange = $this->entityManager->getRepository(NumberRange::class)->findOneBy([
            'name' => self::NUMBER_RANGE_NAME,
        ]);
        if (!$supplierNumberRange) {
            throw new NumberRangeNotFoundException(self::NUMBER_RANGE_NAME);
        }

        // Fetch a number of unused supplier numbers that are unused and starting from the current baseline
        $baseLineNumber = $supplierNumberRange->getNumber();
        while (true) {
            $baseLineNumber += 1;
            if ($this->isSupplierNumberAvailable($baseLineNumber)) {
                // Persist the new baseline to the number range and return result
                $supplierNumberRange->setNumber($baseLineNumber);
                $this->entityManager->flush($supplierNumberRange);

                return $baseLineNumber;
            }
        }
    }
}
