<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters;

use Doctrine\ORM\AbstractQuery;
use Enlight_Components_Snippet_Namespace;
use Shopware\Components\Model\ModelManager;
use Shopware\Components\SwagImportExport\Exception\AdapterException;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\AbstractDbAdapter;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\DbAdapterValidator;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware_Components_Config as Config;
use Shopware_Components_Snippet_Manager as SnippetManager;

class ViisonPickwareERPArticleStockLimitsDbAdapter extends AbstractDbAdapter
{
    /**
     * @var DbAdapterValidator
     */
    protected $validator;

    /**
     * @param Config $config
     * @param ModelManager $entityManager
     * @param SnippetManager $snippetManager
     */
    public function __construct(Config $config, ModelManager $entityManager, SnippetManager $snippetManager)
    {
        parent::__construct($config, $entityManager, $snippetManager);

        $fieldTypes = [
            'string' => [
                'articleNumber',
                'articleName',
                'warehouse',
            ],
            'unsignedInt' => [
                'minimumStock',
                'targetStock',
            ],
        ];
        $requiredFields = [
            'articleNumber',
            'warehouse',
        ];
        $this->validator = new DbAdapterValidator($snippetManager, $fieldTypes, $requiredFields);
    }

    /**
     * @inheritdoc
     */
    public function getSections()
    {
        return [
            [
                'id' => 'default',
                'name' => 'default',
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function getDefaultColumns()
    {
        return [
            'articleDetail.id AS articleDetailId',
            'article.id AS articleId',
            'articleDetail.number AS articleNumber',
            'article.name AS articleName',
            '_warehouse.code AS warehouse',
            'articleDetailConfig.minimumStock AS minimumStock',
            'articleDetailConfig.targetStock AS targetStock',
        ];
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function read($ids, $columns)
    {
        if (!$ids) {
            throw new \Exception($this->getSnippetNamespace()->get(
                'db_adapter/error/read/no_ids',
                'No IDs of article details provided.'
            ));
        }

        // Build the main query
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select($columns)
            ->from(WarehouseArticleDetailConfiguration::class, 'articleDetailConfig')
            ->leftJoin('articleDetailConfig.warehouse', '_warehouse')
            ->leftJoin('articleDetailConfig.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->where('articleDetail.id IN (:ids)')
            ->setParameter('ids', $ids);

        // Order the results by articleNumber, warehouse
        $builder
            ->orderBy('articleNumber')
            ->addOrderBy('warehouse');

        // Fetch results
        $query = $builder->getQuery();
        $query->setHydrationMode(AbstractQuery::HYDRATE_ARRAY);
        $paginator = $this->entityManager->createPaginator($query);
        $result = $paginator->getIterator()->getArrayCopy();

        // Fix article name
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($result, 'articleDetailId'));
        foreach ($result as &$record) {
            $additionalText = $additionalTexts[$record['articleDetailId']];
            $record['articleName'] .= $additionalText ? ' - ' . $additionalText : '';
        }

        return [
            'default' => $result,
        ];
    }

    /**
     * @inheritdoc
     */
    public function readRecordIds($start, $limit, $filter)
    {
        // Prepare query for fetching all warehouse stock ids
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->select('articleDetail.id')
            ->from(ArticleDetail::class, 'articleDetail');

        if (!empty($filter)) {
            $builder->addFilter($filter);
        }
        if ($start) {
            $builder->setFirstResult($start);
        }
        if ($limit) {
            $builder->setMaxResults($limit);
        }

        // Fetch results
        $records = $builder->getQuery()->getResult();

        // Select only the IDs
        $result = array_map(function ($record) {
            return $record['id'];
        }, $records);

        return $result;
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function write($records)
    {
        if (!$records['default']) {
            throw new \Exception($this->getSnippetNamespace()->get(
                'db_adapter/error/write/no_records',
                'No Pickware stock limits records found.'
            ));
        }

        $updatedWarehouseStocks = [];
        foreach ($records['default'] as $record) {
            try {
                $updatedWarehouseStocks[] = $this->writeRecord($record);
            } catch (AdapterException $e) {
                $this->saveMessage($e->getMessage());
            }

            // Flush changes in batches to improve performance
            if (count($updatedWarehouseStocks) === 50) {
                $this->entityManager->flush($updatedWarehouseStocks);
                $updatedWarehouseStocks = [];
            }
        }

        if (count($updatedWarehouseStocks) > 0) {
            // Flush the remaining changes
            $this->entityManager->flush($updatedWarehouseStocks);
        }
    }

    /**
     * Validates and parses the given $record data. If the respective ArticleDetail, Warehouse
     * or WarhouseStock could not be found, an exception is thrown. Otherwise the WarehouseArticleDetailStockCount
     * is updated with the stock limits of the given $record. Also throws exception if the
     * new stock limits are invalid (minimum stock > target stock).
     *
     * @param array $record
     * @return WarehouseArticleDetailConfiguration $warehouseStock
     * @throws AdapterException
     */
    protected function writeRecord(array $record)
    {
        // Clean and validate the record
        $record = $this->validator->filterEmptyString($record);
        $this->validator->checkRequiredFields($record);
        $this->validator->validateFieldTypes($record);

        // Check article detail
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $record['articleNumber'],
        ]);
        if (!$articleDetail) {
            throw new AdapterException(
                sprintf(
                    $this->getSnippetNamespace()->get(
                        'db_adapter/error/write/invalid_article_number',
                        'Item with number "%s" does not exists.'
                    ),
                    $record['articleNumber']
                )
            );
        }

        // Check warehouse
        $warehouse = $this->entityManager->getRepository(Warehouse::class)->findOneBy([
            'code' => $record['warehouse'],
        ]);
        if (!$warehouse) {
            throw new AdapterException(
                sprintf(
                    $this->getSnippetNamespace()->get(
                        'db_adapter/error/write/invalid_warehouse_code',
                        'Warehouse with code "%s" does not exists.'
                    ),
                    $record['warehouse']
                )
            );
        }

        // Check warehouse stock
        /** @var WarehouseArticleDetailConfiguration $warehouseConfig */
        $warehouseConfig = $this->entityManager->getRepository(WarehouseArticleDetailConfiguration::class)->findOneBy([
            'articleDetailId' => $articleDetail->getId(),
            'warehouseId' => $warehouse->getId(),
        ]);
        if (!$warehouseConfig) {
            throw new AdapterException(
                sprintf(
                    $this->getSnippetNamespace()->get(
                        'db_adapter/error/write/invalid_warehouse_stocks',
                        'Warehouse config could not be found for item "%s" and warehouse "%s"'
                    ),
                    $record['articleNumber'],
                    $record['warehouse']
                )
            );
        }

        $newMinimumStock = $record['minimumStock'] ?: 0;
        $newTargetStock = $record['targetStock'] ?: 0;

        if ($newMinimumStock > $newTargetStock) {
            throw new AdapterException(
                $this->getSnippetNamespace()->get(
                    'db_adapter/error/write/invalid_stock_limit_values',
                    'Target stock must at least be as high as the minimum stock.'
                )
            );
        }

        $warehouseConfig->setMinimumStock($newMinimumStock);
        $warehouseConfig->setTargetStock($newTargetStock);

        // Return unflushed entity, since it will be flushed batchwise in caller method.
        return $warehouseConfig;
    }

    /**
     * @return Enlight_Components_Snippet_Namespace
     */
    private function getSnippetNamespace()
    {
        return $this->snippetManager->getNamespace(
            'backend/plugins/swag_import_export/viison_pickware_erp_article_stock_limits'
        );
    }
}
