<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query\Expr;
use Enlight_Components_Snippet_Namespace;
use Shopware\Components\Model\ModelManager;
use Shopware\Components\SwagImportExport\Exception\AdapterException;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\AbstractDbAdapter;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\DbAdapterValidator;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;
use Shopware_Components_Config as Config;
use Shopware_Components_Snippet_Manager as SnippetManager;

class ViisonPickwareERPArticleStocksDbAdapter extends AbstractDbAdapter
{
    const IMPORT_MODE_RELATIVE = 1;
    const IMPORT_MODE_ABSOLUTE = 2;

    /**
     * The code used for a warehouse's null bin location.
     */
    const UNKNOWN_LOCATION_CODE = 'pickware_unknown_location';

    /**
     * @var StockLedgerService
     */
    protected $stockLedgerService;

    /**
     * @var StockChangeListFactory
     */
    protected $stockChangeListFactory;

    /**
     * @var DbAdapterValidator
     */
    protected $validator;

    /**
     * @var Warehouse
     */
    protected $defaultWarehouse;

    /**
     * @var
     */
    protected $importMode;

    /**
     * @param Config $config
     * @param ModelManager $entityManager
     * @param SnippetManager $snippetManager
     * @param StockLedgerService $stockLedgerService
     * @param StockChangeListFactory $stockChangeListFactory
     * @param int $importMode
     */
    public function __construct(
        Config $config,
        ModelManager $entityManager,
        SnippetManager $snippetManager,
        StockLedgerService $stockLedgerService,
        StockChangeListFactory $stockChangeListFactory,
        $importMode
    ) {
        parent::__construct($config, $entityManager, $snippetManager);
        $this->stockLedgerService = $stockLedgerService;
        $this->stockChangeListFactory = $stockChangeListFactory;
        $this->defaultWarehouse = $this->entityManager->getRepository(Warehouse::class)->getDefaultWarehouse();
        $this->importMode = $importMode;

        if (!in_array($this->importMode, [self::IMPORT_MODE_ABSOLUTE, self::IMPORT_MODE_RELATIVE])) {
            throw new \InvalidArgumentException(
                '5th argument ($importMode) of ViisonPickwareERPArticleStocksDbAdapter::__construct() must be ' .
                'either ViisonPickwareERPArticleStocksDbAdapter::IMPORT_MODE_RELATIVE or '.
                'ViisonPickwareERPArticleStocksDbAdapter::IMPORT_MODE_ABSOLUTE'
            );
        }

        $this->validator = new DbAdapterValidator(
            $snippetManager,
            $this->getFieldTypes(),
            [
                'articleNumber',
            ]
        );
    }

    /**
     * Returns all necessary fields and types which differ if the import mode is relative or absolute.
     *
     * @return array
     */
    private function getFieldTypes()
    {
        $fieldTypes = [
            'float' => [
                'purchasePrice',
            ],
            'string' => [
                'binLocation',
                'articleNumber',
                'warehouse',
            ],
        ];
        if ($this->importMode === ViisonPickwareERPArticleStocksDbAdapter::IMPORT_MODE_RELATIVE) {
            $fieldTypes['int'] = [
                'physicalStock',
            ];
        } else {
            $fieldTypes['int'] = [
                'availableStock',
            ];
            $fieldTypes['unsignedInt'] = [
                'physicalStock',
            ];
        }

        return $fieldTypes;
    }

    /**
     * @inheritdoc
     */
    public function getSections()
    {
        return [
            [
                'id' => 'default',
                'name' => 'default',
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function getDefaultColumns()
    {
        $defaultColumns = [
            'ad.id AS articleDetailId',
            'a.id AS articleId',
            'a.name AS articleName',
            'ad.number AS articleNumber',
            'w.code AS warehouse',
            'CASE WHEN (wbl.code = \'' . BinLocation::NULL_BIN_LOCATION_CODE . '\') THEN \'\' ELSE wbl.code END AS binLocation',
            'CASE WHEN wblad.defaultMapping = 1 THEN \'yes\' ELSE \'no\' END AS isDefaultBinLocation',
            ($this->importMode === self::IMPORT_MODE_RELATIVE) ? '0 AS physicalStock' : 'wblad.stock AS physicalStock',
            'ad.purchasePrice AS purchasePrice',
        ];

        if ($this->importMode === self::IMPORT_MODE_ABSOLUTE) {
            // Only add available stock column when handling absolute import/export mode
            array_splice($defaultColumns, 8, 0, [
                'ad.inStock AS availableStock',
            ]);
        }

        return $defaultColumns;
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function read($ids, $columns)
    {
        if (!$ids) {
            throw new \Exception($this->getSnippetNamespace()->get('db_adapter/error/read/no_ids', 'Cannot read Pickware stocks without valid item IDs.'));
        }

        // Build the main query
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select($columns)
                ->from(ArticleDetail::class, 'ad')
                ->leftJoin(ArticleDetailBinLocationMapping::class, 'wblad', Expr\Join::WITH, 'wblad.articleDetailId = ad.id')
                ->join('wblad.binLocation', 'wbl')
                ->join('wbl.warehouse', 'w')
                ->join('ad.article', 'a')
                ->where('ad.id IN (:ids)')
                ->setParameter('ids', $ids);

        // Order the results by articleNumber, warehouse, binLocation
        $builder->orderBy('articleNumber')
                ->addOrderBy('warehouse')
                ->addOrderBy('binLocation');

        // Fetch results
        $query = $builder->getQuery();
        $query->setHydrationMode(AbstractQuery::HYDRATE_ARRAY);
        $paginator = $this->entityManager->createPaginator($query);
        $result = $paginator->getIterator()->getArrayCopy();

        // Set default values if necessary
        foreach ($result as &$record) {
            if (!$record['warehouse']) {
                $record['warehouse'] = $this->defaultWarehouse->getCode();
            }
            if ($record['binLocation'] === '' && $record['physicalStock'] !== 0) {
                $record['binLocation'] = self::UNKNOWN_LOCATION_CODE;
            }
        }

        /** @var integer[] $articleDetailIds */
        $articleDetailIds = array_map(function ($article) {
            return $article['articleDetailId'];
        }, $result);

        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts($articleDetailIds);

        // fix article name
        foreach ($result as &$record) {
            $record['articleName'] .= ($additionalTexts[$record['articleDetailId']]) ? (' - ' . $additionalTexts[$record['articleDetailId']]) : '';
        }

        return [
            'default' => $result,
        ];
    }

    /**
     * @inheritdoc
     */
    public function readRecordIds($start, $limit, $filter)
    {
        if (isset($filter['stockFilter'])) {
            unset($filter['stockFilter']);
        }

        // Prepare query for fetching article detail IDs
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select('ad.id')->from(ArticleDetail::class, 'ad');
        if (!empty($filter)) {
            $builder->addFilter($filter);
        }
        if ($start) {
            $builder->setFirstResult($start);
        }
        if ($limit) {
            $builder->setMaxResults($limit);
        }

        // Fetch results
        $records = $builder->getQuery()->getResult();

        // Select only the IDs
        $result = array_map(function ($record) {
            return $record['id'];
        }, $records);

        return $result;
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function write($records)
    {
        if (!$records['default']) {
            throw new \Exception($this->getSnippetNamespace()->get('db_adapter/error/write/no_records', 'No Pickware stock records found.'));
        }

        foreach ($records['default'] as $record) {
            try {
                $this->writeRecord($record);
            } catch (AdapterException $e) {
                $this->saveMessage($e->getMessage());
            }
        }
    }

    /**
     * Validates and parses the given $record data. Since all fields but the 'articleNumber'
     * are optional, 'purchasePrice' and 'availableStock' are only saved if set in the record.
     * If no 'warehouse' is set, the default warehouse is used and the fallback for the
     * 'binLocation' is the current bin location of the article detail. If a 'binLocation' is
     * set, but the code does not correspond to any bin location on the selected warehouse,
     * a new bin location is created before any stock changes are logged. If the bin location
     * of the article detail changes, first a 'relocation' stock entry is logged, before the
     * stock changes are logged in a 'manual' entry.
     *
     * @param array $record
     * @throws AdapterException
     */
    protected function writeRecord(array $record)
    {
        // Clean and validate the record
        $record = $this->validator->filterEmptyString($record);
        $this->validator->checkRequiredFields($record);
        $this->validator->validateFieldTypes($record);

        // Check article detail
        /** @var ArticleDetail $articleDetail */
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $record['articleNumber'],
        ]);
        if (!$articleDetail) {
            throw new AdapterException(
                sprintf($this->getSnippetNamespace()->get('db_adapter/error/write/invalid_article_number', 'Item with number "%s" does not exists.'), $record['articleNumber'])
            );
        }

        // Save changed purchase price, if set
        if (isset($record['purchasePrice'])) {
            $articleDetail->setPurchasePrice(floatval($record['purchasePrice']));
            $this->entityManager->flush($articleDetail);
        }

        /** @var \Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository $warehouseRepository */
        $warehouseRepository = $this->entityManager->getRepository(Warehouse::class);

        // Check for warehouse
        /** @var Warehouse $warehouse */
        if (isset($record['warehouse'])) {
            $warehouse = $warehouseRepository->findOneBy([
                'code' => $record['warehouse'],
            ]);
            if (!$warehouse) {
                throw new AdapterException(
                    sprintf($this->getSnippetNamespace()->get('db_adapter/error/write/invalid_warehouse', 'Warehouse with name "%s" does not exist.'), $record['warehouse'])
                );
            }
        } else {
            $warehouse = $this->defaultWarehouse;
        }

        // Check for bin location and create one if it does not exist
        if (isset($record['binLocation']) && $record['binLocation'] !== self::UNKNOWN_LOCATION_CODE) {
            $binLocation = $this->entityManager->getRepository(BinLocation::class)->findOneBy([
                'warehouse' => $warehouse,
                'code' => $record['binLocation'],
            ]);
            if (!$binLocation) {
                // Create a new bin location with the given code
                $binLocation = new BinLocation($warehouse, $record['binLocation']);
                $this->entityManager->persist($binLocation);
                $this->entityManager->flush($binLocation);
            }
        } else {
            $binLocation = $warehouse->getNullBinLocation();
        }

        $articleDetailBinLocationMapping = $warehouseRepository->findArticleDetailBinLocationMapping($articleDetail, $binLocation);
        if ($articleDetailBinLocationMapping) {
            // If the article detail bin location mapping existed beforehand we need to refresh it here, so the entity
            // is up to date when the stock updates are processed line by line in this import.
            $this->entityManager->refresh($articleDetailBinLocationMapping);
        }

        // Update the default bin location flag if necessary (activate default bin location or remove the flag)
        if (isset($record['isDefaultBinLocation'])
            && !$binLocation->isNullBinLocation()
            && self::yesNoStringToBoolean($record['isDefaultBinLocation'])
        ) {
            $this->stockLedgerService->selectDefaultBinLocation($articleDetail, $warehouse, $binLocation);
        } elseif ($articleDetailBinLocationMapping && $articleDetailBinLocationMapping->isDefaultMapping() && !self::yesNoStringToBoolean($record['isDefaultBinLocation'])) {
            // We can safely assume that if this article detail bin location mapping is the default mapping, no other
            // mapping can be the default and we can remove 'all' default flags in this warehouse.
            $this->stockLedgerService->selectDefaultBinLocation($articleDetail, $warehouse);
        }

        if (isset($record['physicalStock'])) {
            $stockChange = (int) $record['physicalStock'];
            if ($this->importMode === self::IMPORT_MODE_ABSOLUTE && $articleDetailBinLocationMapping) {
                // When handling the absolute stock import and an article detail bin location mapping already exists, we
                // need to calculate the stock change accordingly.
                $stockChange -= $articleDetailBinLocationMapping->getStock();
            }

            if ($stockChange !== 0) {
                // Write the stock change depending on the amount (increase or decrease) of the stock change. Creates any
                // missing article detail bin location mappings, or removes them if necessary.
                $stockChanges = $this->stockChangeListFactory->createSingleBinLocationStockChangeList(
                    $binLocation,
                    $stockChange
                );
                $stockChangeComment = 'SwagImportExport: "physicalStock" Import';
                if ($stockChange > 0) {
                    $this->stockLedgerService->recordIncomingStock(
                        $articleDetail,
                        $stockChanges,
                        $articleDetail->getPurchasePrice(),
                        $stockChangeComment
                    );
                } else {
                    $this->stockLedgerService->recordOutgoingStock(
                        $articleDetail,
                        $stockChanges,
                        $stockChangeComment
                    );
                }
            }
        }

        // Save changed available stock, if set. This can only be the case for absolute stock import (that is why the
        // new stock value is equal to the given available stock value.
        if (isset($record['availableStock'])) {
            $articleDetail->setInStock(intval($record['availableStock']));
            $this->entityManager->flush($articleDetail);
        }
    }

    /**
     * @return Enlight_Components_Snippet_Namespace
     */
    private function getSnippetNamespace()
    {
        return $this->snippetManager->getNamespace(
            'backend/plugins/swag_import_export/viison_pickware_erp_article_stocks_common'
        );
    }

    /**
     * Convert a human readable $yesNoString that represents a boolean to a boolean
     *
     * @param $yesNoString
     * @return bool
     */
    protected static function yesNoStringToBoolean($yesNoString)
    {
        $yesNoString = trim(mb_strtolower((string) $yesNoString));

        switch ($yesNoString) {
            case 'yes':
            case 'ja':
            case '1':
            case 'on':
            case 'an':
            case 'true':
            case 'j':
            case 'y':
            case 't':
                return true;
            case 'no':
            case 'nein':
            case '0':
            case 'off':
            case 'aus':
            case 'false':
            case 'n':
            case 'f':
            case '':
            case 'null':
                return false;
            default:
                return (bool) $yesNoString;
        }
    }
}
