<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters;

use Doctrine\ORM\AbstractQuery;
use Enlight_Components_Snippet_Namespace;
use Shopware\Components\Model\ModelManager;
use Shopware\Components\SwagImportExport\Exception\AdapterException;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\AbstractDbAdapter;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\DbAdapterValidator;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware_Components_Config as Config;
use Shopware_Components_Snippet_Manager as SnippetManager;

class ViisonPickwareERPArticleSuppliersDbAdapter extends AbstractDbAdapter
{
    /**
     * @var DbAdapterValidator
     */
    protected $validator;

    /**
     * @param Config $config
     * @param ModelManager $entityManager
     * @param SnippetManager $snippetManager
     */
    public function __construct(Config $config, ModelManager $entityManager, SnippetManager $snippetManager)
    {
        parent::__construct($config, $entityManager, $snippetManager);

        $fieldTypes = [
            'float' => [
                'purchasePrice',
            ],
            'string' => [
                'supplierNumber',
                'defaultSupplier',
                'articleNumber',
                'supplierArticleNumber',
                'currency',
            ],
            'unsignedInt' => [
                'minimumOrderAmount',
                'packingUnit',
            ],
        ];
        $requiredFields = [
            'articleNumber',
            'supplierNumber',
        ];
        $this->validator = new DbAdapterValidator($snippetManager, $fieldTypes, $requiredFields);
    }

    /**
     * @inheritdoc
     */
    public function getSections()
    {
        return [
            [
                'id' => 'default',
                'name' => 'default',
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function getDefaultColumns()
    {
        return [
            'variant.id AS articleDetailId',
            'a.id AS articleId',
            'a.name AS articleName',
            'variant.number AS articleNumber',
            'supplier.number AS supplierNumber',
            'CASE WHEN (mapping.defaultSupplier = 1) THEN \'y\' ELSE \'n\' END AS defaultSupplier',
            'mapping.purchasePrice AS purchasePrice',
            'supplierCurrency.currency AS currency',
            'mapping.minimumOrderAmount AS minimumOrderAmount',
            'mapping.packingUnit AS packingUnit',
            'mapping.supplierArticleNumber AS supplierArticleNumber',
        ];
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function read($ids, $columns)
    {
        if (!$ids) {
            throw new \Exception($this->getSnippetNamespace()->get('db_adapter/error/read/no_ids', 'Cannot read Pickware supplier mappings without valid IDs.'));
        }

        // Build the main query
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select($columns)
                ->from(ArticleDetailSupplierMapping::class, 'mapping')
                ->join('mapping.articleDetail', 'variant')
                ->join('mapping.supplier', 'supplier')
                ->join('supplier.currency', 'supplierCurrency')
                ->join('variant.article', 'a')
                ->where('mapping.id IN (:ids)')
                ->setParameter('ids', $ids);

        // Order the results by articleNumber, supplierNumber
        $builder->orderBy('articleNumber')
                ->orderBy('supplierNumber');

        // Fetch results
        $query = $builder->getQuery();
        $query->setHydrationMode(AbstractQuery::HYDRATE_ARRAY);
        $paginator = $this->entityManager->createPaginator($query);
        $result = $paginator->getIterator()->getArrayCopy();

        // Fix article name
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($result, 'articleDetailId'));
        foreach ($result as &$record) {
            $additionalText = $additionalTexts[$record['articleDetailId']];
            $record['articleName'] .= $additionalText ? ' - ' . $additionalText : '';
        }

        return [
            'default' => $result,
        ];
    }

    /**
     * @inheritdoc
     */
    public function readRecordIds($start, $limit, $filter)
    {
        if (isset($filter['stockFilter'])) {
            unset($filter['stockFilter']);
        }

        // Prepare query for fetching all supplier article detail IDs
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select('mapping.id')->from(ArticleDetailSupplierMapping::class, 'mapping');
        if (!empty($filter)) {
            $builder->addFilter($filter);
        }
        if ($start) {
            $builder->setFirstResult($start);
        }
        if ($limit) {
            $builder->setMaxResults($limit);
        }

        // Fetch results
        $records = $builder->getQuery()->getResult();

        // Select only the IDs
        $result = array_map(function ($record) {
            return $record['id'];
        }, $records);

        return $result;
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function write($records)
    {
        if (!$records['default']) {
            throw new \Exception($this->getSnippetNamespace()->get('db_adapter/error/write/no_records', 'No Pickware supplier mapping records found.'));
        }

        $updatedMappings = [];
        foreach ($records['default'] as $record) {
            try {
                $updatedMappings[] = $this->writeRecord($record);
            } catch (AdapterException $e) {
                $this->saveMessage($e->getMessage());
            }

            // Flush changes in batches to improve performance
            if (count($updatedMappings) === 50) {
                $this->entityManager->flush($updatedMappings);
                $updatedMappings = [];
            }
        }

        if (count($updatedMappings) > 0) {
            // Flush the remaining changes
            $this->entityManager->flush($updatedMappings);
        }
    }

    /**
     * Validates and parses the given $record data. The values 'articleNumber' and 'supplierNumber' are mandatory to
     * identify a SupplierArticle-mapping. If one of them is missing or missed to identify an existing entity, an error
     * is thrown. Otherweise (mapping could be found), the mapping will be updated.
     *
     * @param array $record
     * @return Supplier
     * @throws AdapterException
     */
    protected function writeRecord(array $record)
    {
        // Clean and validate the record
        $record = $this->validator->filterEmptyString($record);
        $this->validator->checkRequiredFields($record);
        $this->validator->validateFieldTypes($record);

        // Check article detail
        $articleDetail = $this->entityManager->getRepository(ArticleDetail::class)->findOneBy([
            'number' => $record['articleNumber'],
        ]);
        if (!$articleDetail) {
            throw new AdapterException(
                sprintf($this->getSnippetNamespace()->get('db_adapter/error/write/invalid_article_number', 'Item with number "%s" does not exists.'), $record['articleNumber'])
            );
        }

        // Check supplier
        $supplier = $this->entityManager->getRepository(Supplier::class)->findOneBy([
            'number' => $record['supplierNumber'],
        ]);
        if (!$supplier) {
            throw new AdapterException(
                sprintf($this->getSnippetNamespace()->get('db_adapter/error/write/invalid_supplier_number', 'Supplier with number "%s" does not exists.'), $record['supplierNumber'])
            );
        }

        // Try to find existing mapping
        $mapping = $this->entityManager->getRepository(ArticleDetailSupplierMapping::class)->findOneBy([
            'articleDetail' => $articleDetail,
            'supplier' => $supplier,
        ]);
        if (!$mapping) {
            // Create new mapping
            $mapping = new ArticleDetailSupplierMapping($articleDetail, $supplier);
            $this->entityManager->persist($mapping);
        }

        // Update the mapping
        if (isset($record['purchasePrice'])) {
            $mapping->setPurchasePrice($record['purchasePrice']);
        }
        if (isset($record['supplierArticleNumber'])) {
            $mapping->setSupplierArticleNumber($record['supplierArticleNumber']);
        }
        if (isset($record['minimumOrderAmount'])) {
            $mapping->setMinimumOrderAmount($record['minimumOrderAmount']);
        }
        if (isset($record['packingUnit'])) {
            $mapping->setPackingUnit($record['packingUnit']);
        }
        if (isset($record['defaultSupplier'])) {
            $shouldBeDefaultSupplier = $record['defaultSupplier'] == '1' || $record['defaultSupplier'] == 'y';
            if ($shouldBeDefaultSupplier && !$mapping->isDefaultSupplier()) {
                // Make sure no other supplier mapping of the article detail is set as default supplier
                $defaultMapping = $this->entityManager->getRepository(ArticleDetailSupplierMapping::class)->findOneBy([
                    'articleDetail' => $articleDetail,
                    'defaultSupplier' => true,
                ]);
                if ($defaultMapping) {
                    // Flush the changes right away
                    $defaultMapping->setDefaultSupplier(false);
                    $this->entityManager->flush($defaultMapping);
                }
            }
            $mapping->setDefaultSupplier($shouldBeDefaultSupplier);
            if ($mapping->isDefaultSupplier()) {
                // Flush the changes on the mapping right away, to make sure it can be found as a default mapping
                // in following calls to 'writeRecord'
                $this->entityManager->flush($mapping);
            }
        }

        return $mapping;
    }

    /**
     * @return Enlight_Components_Snippet_Namespace
     */
    private function getSnippetNamespace()
    {
        return $this->snippetManager->getNamespace('backend/plugins/swag_import_export/viison_pickware_erp_article_suppliers');
    }
}
