<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters;

use Doctrine\ORM\AbstractQuery;
use Enlight_Components_Snippet_Namespace;
use Shopware\Components\Model\ModelManager;
use Shopware\Components\SwagImportExport\Exception\AdapterException;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier;
use Shopware\Models\Shop\Currency;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\AbstractDbAdapter;
use Shopware\Plugins\ViisonCommon\Classes\Plugins\SwagImportExport\DbAdapterValidator;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\Supplier\SupplierNumberGenerator;
use Shopware_Components_Config as Config;
use Shopware_Components_Snippet_Manager as SnippetManager;

class ViisonPickwareERPSuppliersDbAdapter extends AbstractDbAdapter
{
    /**
     * @var DbAdapterValidator
     */
    protected $validator;

    /**
     * @var SupplierNumberGenerator
     */
    protected $supplierNumberGenerator;

    /**
     * @param Config $config
     * @param ModelManager $entityManager
     * @param SnippetManager $snippetManager
     * @param SupplierNumberGenerator $supplierNumberGenerator
     */
    public function __construct(
        Config $config,
        ModelManager $entityManager,
        SnippetManager $snippetManager,
        SupplierNumberGenerator $supplierNumberGenerator
    ) {
        parent::__construct($config, $entityManager, $snippetManager);
        $this->supplierNumberGenerator = $supplierNumberGenerator;

        $fieldTypes = [
            'email' => [
                'email',
            ],
            'string' => [
                'supplierNumber',
                'address',
                'comment',
                'contact',
                'customerNumber',
                'fax',
                'name',
                'phone',
                'salutation',
                'currency',
            ],
            'unsignedInt' => [
                'deliveryTime',
                'templateShopId',
            ],
        ];
        $requiredFields = [
            'name',
        ];
        $this->validator = new DbAdapterValidator($snippetManager, $fieldTypes, $requiredFields);
    }

    /**
     * @inheritdoc
     */
    public function getSections()
    {
        return [
            [
                'id' => 'default',
                'name' => 'default',
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function getDefaultColumns()
    {
        return [
            'supplier.number AS supplierNumber',
            'supplier.name AS name',
            'supplier.salutation AS salutation',
            'supplier.contact AS contact',
            'supplier.address AS address',
            'supplier.email AS email',
            'supplier.phone AS phone',
            'supplier.fax AS fax',
            'supplier.comment AS comment',
            'supplier.customerNumber AS customerNumber',
            'supplier.deliveryTime AS deliveryTime',
            'supplier.documentLocalizationSubShopId as templateShopId',
            'supplierCurrency.currency as currency',
        ];
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function read($ids, $columns)
    {
        if (!$ids) {
            throw new \Exception($this->getSnippetNamespace()->get('db_adapter/error/read/no_ids', 'Cannot read Pickware suppliers without valid supplier IDs.'));
        }

        // Build the main query
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select($columns)
                ->from(Supplier::class, 'supplier')
                ->leftJoin('supplier.currency', 'supplierCurrency')
                ->where('supplier.id IN (:ids)')
                ->setParameter('ids', $ids);

        // Order the results by name
        $builder->orderBy('name');

        // Fetch results
        $query = $builder->getQuery();
        $query->setHydrationMode(AbstractQuery::HYDRATE_ARRAY);
        $paginator = $this->entityManager->createPaginator($query);
        $result = $paginator->getIterator()->getArrayCopy();

        // Set default values if necessary
        foreach ($result as &$record) {
            if ($record['deliveryTime'] === null) {
                $record['deliveryTime'] = 0;
            }
        }

        return [
            'default' => $result,
        ];
    }

    /**
     * @inheritdoc
     */
    public function readRecordIds($start, $limit, $filter)
    {
        if (isset($filter['stockFilter'])) {
            unset($filter['stockFilter']);
        }

        // Prepare query for fetching supplier IDs
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select('supplier.id')->from(Supplier::class, 'supplier');
        if (!empty($filter)) {
            $builder->addFilter($filter);
        }
        if ($start) {
            $builder->setFirstResult($start);
        }
        if ($limit) {
            $builder->setMaxResults($limit);
        }

        // Fetch results
        $records = $builder->getQuery()->getResult();

        // Select only the IDs
        $result = array_map(function ($record) {
            return $record['id'];
        }, $records);

        return $result;
    }

    /**
     * @inheritdoc
     * @throws \Exception
     */
    public function write($records)
    {
        if (!$records['default']) {
            throw new \Exception($this->getSnippetNamespace()->get('db_adapter/error/write/no_records', 'No Pickware supplier records found.'));
        }

        $updatedSuppliers = [];
        foreach ($records['default'] as $record) {
            try {
                $updatedSuppliers[] = $this->writeRecord($record);
            } catch (AdapterException $e) {
                $this->saveMessage($e->getMessage());
            }

            // Flush changes in batches to improve performance
            if (count($updatedSuppliers) === 50) {
                $this->entityManager->flush($updatedSuppliers);
                $updatedSuppliers = [];
            }
        }

        if (count($updatedSuppliers) > 0) {
            // Flush the remaining changes
            $this->entityManager->flush($updatedSuppliers);
        }
    }

    /**
     * Validates and parses the given $record data. If a 'supplierNumber' is set, the respective supplier is updated.
     * Otherwise a new supplier with the default shop's language/locale is created, if at least the 'name' is set in the
     * $record.
     *
     * @param array $record
     * @return Supplier
     * @throws AdapterException
     */
    protected function writeRecord(array $record)
    {
        // Clean and validate the record
        $record = $this->validator->filterEmptyString($record);
        $this->validator->checkRequiredFields($record);
        $this->validator->validateFieldTypes($record);

        $supplier = null;
        $isNewSupplier = false;
        if (isset($record['supplierNumber'])) {
            $supplier = $this->entityManager->getRepository(Supplier::class)->findOneBy([
                'number' => $record['supplierNumber'],
            ]);
        }

        if (!$supplier) {
            // Create a new supplier
            $supplier = new Supplier();
            $isNewSupplier = true;
        }

        // Update the supplier
        if (isset($record['supplierNumber']) && $record['supplierNumber'] !== '') {
            $supplier->setNumber($record['supplierNumber']);
        } else {
            // If no or empty supplier number is given, generate new one
            $unusedSupplierNumber = $this->supplierNumberGenerator->generateSupplierNumber();
            $supplier->setNumber($unusedSupplierNumber);
        }
        if (isset($record['name'])) {
            $supplier->setName($record['name']);
        }
        if (isset($record['salutation'])) {
            $supplier->setSalutation($record['salutation']);
        }
        if (isset($record['contact'])) {
            $supplier->setContact($record['contact']);
        }
        if (isset($record['address'])) {
            $supplier->setAddress($record['address']);
        }
        if (isset($record['email'])) {
            $supplier->setEmail($record['email']);
        }
        if (isset($record['phone'])) {
            $supplier->setPhone($record['phone']);
        }
        if (isset($record['fax'])) {
            $supplier->setFax($record['fax']);
        }
        if (isset($record['comment'])) {
            $supplier->setComment($record['comment']);
        }
        if (isset($record['customerNumber'])) {
            $supplier->setCustomerNumber($record['customerNumber']);
        }
        if (isset($record['deliveryTime'])) {
            $supplier->setDeliveryTime($record['deliveryTime']);
        }
        if (isset($record['templateShopId'])) {
            $shop = $this->entityManager->find(Shop::class, $record['templateShopId']);
            if (!$shop) {
                throw new AdapterException(
                    sprintf(
                        $this->getSnippetNamespace()->get(
                            'db_adapter/error/write/invalid_shop_id',
                            'Sub shop for document localization with ID "%d" does not exist.'
                        ),
                        $record['templateShopId']
                    )
                );
            }
            $supplier->setDocumentLocalizationSubShop($shop);
        }
        if (isset($record['currency'])) {
            $currency = $this->entityManager->getRepository(Currency::class)->findOneBy([
                'currency' => $record['currency'],
            ]);
            if (!$currency) {
                throw new AdapterException(
                    sprintf(
                        $this->getSnippetNamespace()->get(
                            'db_adapter/error/write/invalid_currency',
                            'Currency with ISO currency code "%s" does not exist.'
                        ),
                        $record['currency']
                    )
                );
            }
            $supplier->setCurrency($currency);
        } elseif ($isNewSupplier) {
            $supplier->setCurrency(CurrencyUtil::getDefaultCurrency());
        }

        // Doctrine always flushes all newly persisted object into the database even if only one element should be flushed.
        // Since the supplier number service flushes an object into the database and a supplier can only be saved
        // once its email, name and number are set, we have to persist the entity here.
        if ($isNewSupplier === true) {
            $this->entityManager->persist($supplier);
        }

        return $supplier;
    }

    /**
     * @return Enlight_Components_Snippet_Namespace
     */
    private function getSnippetNamespace()
    {
        return $this->snippetManager->getNamespace('backend/plugins/swag_import_export/viison_pickware_erp_suppliers');
    }
}
