<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Warehouse;

/**
 * @internal Use the service `pickware.erp.bin_location_code_generator_service` to generate bin location code
 *           components instead.
 */
class BinLocationCodeComponent
{
    /**
     * @var boolean
     */
    private $fixed;

    /**
     * @var string|int
     */
    private $currentValue;

    /**
     * @var BinLocationCodeComponent
     */
    private $successor;

    /* Dynamic components only */

    /**
     * @var boolean
     */
    private $numeric;

    /**
     * @var int
     */
    private $length;

    /**
     * @var string
     */
    private $start;

    /**
     * @var string
     */
    private $end;

    /**
     * @var boolean
     */
    private $leadingZeros;

    /**
     * Takes the first component from the given array of components and parses it.
     * If that component is a dynamic, letters component, the letteres are split up into
     * several succeeding sub-components, where the first of these sub-components is used
     * to create this linked component, while the other sub-components are prepended to
     * the array of components.
     * In any case, if any components remain in the array, a succeeding component is created
     * using the array.
     *
     * @param array $components
     */
    public function __construct(array $components)
    {
        // Remove the first component from the array and use it to initialise this instance
        $component = array_shift($components);
        $this->fixed = $component['type'] === 'fixed';
        if ($this->fixed) {
            $this->currentValue = ($component['value']) ?: '';
        } else {
            $this->numeric = $component['type'] === 'digits';
            if ($this->numeric) {
                $this->length = intval($component['length']);
                $this->start = intval($component['start']);
                $this->end = intval($component['end']);
                $this->leadingZeros = $component['leadingZeros'] === true;
            } else {
                // Split this component in as many single components as its start/end values have characters
                // to make to easy to iterate, reset etc.
                $startChars = str_split($component['start']);
                $endChars = str_split($component['end']);
                $this->length = 1;
                $this->start = array_shift($startChars);
                $this->end = array_shift($endChars);

                // Use remaining characters to create sub components
                $subComponents = [];
                for ($i = 0; $i < count($startChars); $i++) {
                    $subComponents[] = [
                        'type' => 'letters',
                        'start' => $startChars[$i],
                        'end' => $endChars[$i],
                    ];
                }
                $components = array_merge($subComponents, $components);
            }

            // Use start value as current value for dynamic components
            $this->currentValue = $this->start;
        }

        if (count($components) > 0) {
            $this->successor = new BinLocationCodeComponent($components);
        }
    }

    /**
     * Creates all remaining bin locations using this component and any succeeding components. If a 'limit'
     * greater zero is given, at most 'limit' locations are created. Note: Calling this method does not
     * reset the state of this component. If you need to create all bin locations twice using the same
     * BinLocationCodeComponent instance, you must call 'reset()' yourself before calling 'createBinLocationCodes'
     * for a second time. This also means that calling 'createBinLocationCodes()' repeatedly on the same component
     * never creates the same bin locations.
     *
     * @param int $limit (optional)
     * @return array
     */
    public function createCodes($limit = 0)
    {
        $binLocations = [];
        do {
            $binLocations[] = $this->getCurrentValue();
            $this->next();
        } while ($this->hasMoreCodes() && ($limit <= 0 || count($binLocations) < $limit));

        return $binLocations;
    }

    /**
     * Calculates the number of values that can be generated by this component incl.
     * all succeeding components.
     *
     * @return int
     */
    public function getNumberOfPossibleCodes()
    {
        // Calculate the number of possible values of this component
        $possibilities = 1;
        if (!$this->fixed) {
            if ($this->numeric) {
                $possibilities = 1 + $this->end - $this->start;
            } else {
                $possibilities = 1 + ord($this->end) - ord($this->start);
            }
        }

        // Multiply with number of possibilities of successor, if available
        if ($this->successor) {
            $possibilities *= $this->successor->getNumberOfPossibleCodes();
        }

        return $possibilities;
    }

    /**
     * @return boolean
     */
    public function hasMoreCodes()
    {
        return $this->isValid() && !$this->isFixed();
    }

    /**
     * @return boolean
     */
    private function isFixed()
    {
        return $this->fixed && (!$this->successor || $this->successor->isFixed());
    }

    /**
     * @return boolean
     */
    private function isValid()
    {
        return ($this->fixed || $this->currentValue <= $this->end) && (!$this->successor || $this->successor->isValid());
    }

    /**
     * Returns the current, formatted value of this component prepended to value any succeeding components.
     *
     * @return string
     */
    private function getCurrentValue()
    {
        // Pad value with zeros if necessary
        $value = ($this->numeric && $this->leadingZeros) ? str_pad($this->currentValue, $this->length, '0', STR_PAD_LEFT) : $this->currentValue;

        // Append current value of successor, if available
        if ($this->successor) {
            $value .= $this->successor->getCurrentValue();
        }

        return $value;
    }

    /**
     * Computes the next value of this component incl. all succeeding components. That is,
     * as long as the succeeding components are still valid (meaning: can be 'increased'),
     * the 'next()' call is passed onward to the successor and the value of this component
     * is not changed. If however the succeeding components are not valid anymore, they are
     * reset and the next value of this component is computet instead.
     */
    private function next()
    {
        // Forward the call to the successor, if it is available and dynamic
        if ($this->successor && !$this->successor->isFixed()) {
            $this->successor->next();
            if ($this->successor->isValid()) {
                // Successor is still valid after increasing it, hence it's not necessary
                // to do any further work in this component
                return;
            }
        }
        if ($this->fixed) {
            return;
        }

        // Create next value of this component
        if ($this->numeric) {
            $this->currentValue += 1;
        } else {
            $this->currentValue = chr(ord($this->currentValue) + 1);
        }

        // Reset successor, if available
        if ($this->successor) {
            $this->successor->reset();
        }
    }

    /**
     * Resets this component and all succeeding components. That is, for dynamic components
     * the 'currentValue' is reset to the 'start' value.
     */
    private function reset()
    {
        if (!$this->fixed) {
            $this->currentValue = $this->start;
        }

        // Reset successor, if available
        if ($this->successor) {
            $this->successor->reset();
        }
    }
}
