<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Warehouse;

use Enlight_Hook;

class BinLocationCodeGeneratorService implements BinLocationCodeGenerator, Enlight_Hook
{
    /**
     * @inheritdoc
     */
    public function createLinkedCodeComponent(array $rawComponents)
    {
        if (!$this->areRawCodeComponentsValid($rawComponents)) {
            throw BinLocationCodeGeneratorException::invalidCodeComponents();
        }

        return new BinLocationCodeComponent($rawComponents);
    }

    /**
     * @inheritdoc
     */
    public function areRawCodeComponentsValid(array $rawComponents)
    {
        foreach ($rawComponents as $rawComponent) {
            if (!$this->isRawCodeComponentValid($rawComponent)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param array $rawComponent
     * @return boolean
     */
    protected function isRawCodeComponentValid(array $rawComponent)
    {
        if ($rawComponent['type'] === 'fixed') {
            // Fixed component
            return is_string($rawComponent['value']);
        }

        // Validate general dynamic fields
        if (mb_strlen('' . $rawComponent['start']) === 0 || mb_strlen('' . $rawComponent['end']) === 0) {
            return false;
        }

        // Validate type specific, dynamic fields
        if ($rawComponent['type'] === 'letters') {
            // Check that 'start' and 'end' are both uppercase letters only and have the same length
            $valid = preg_match('/^[A-Z]+$/', $rawComponent['start'])
                && preg_match('/^[A-Z]+$/', $rawComponent['end'])
                && mb_strlen($rawComponent['start']) === mb_strlen($rawComponent['end']);
            if (!$valid) {
                return false;
            }

            // Check that 'start' <= 'end'
            $startChars = str_split($rawComponent['start']);
            $endChars = str_split($rawComponent['end']);
            for ($i = 0; $i < count($startChars); $i++) {
                if ($startChars[$i] > $endChars[$i]) {
                    return false;
                }
            }

            return true;
        } elseif ($rawComponent['type'] === 'digits') {
            // Check that 'start' and 'end' are both digits only, both positive, 'start' <= 'end'
            // and that the length of the 'end' string value is <= the given length
            return (is_int($rawComponent['start']) || preg_match('/^[\\d]+$/', $rawComponent['start']))
                && (is_int($rawComponent['end']) || preg_match('/^[\\d]+$/', $rawComponent['end']))
                && intval($rawComponent['start']) >= 0
                && intval($rawComponent['start']) <= intval($rawComponent['end'])
                && intval($rawComponent['length']) > 0
                && intval($rawComponent['length']) >= mb_strlen('' . $rawComponent['end']);
        }

        // Else: unknown or missing dynamic type
        return false;
    }
}
