<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Components\Warehouse;

use Doctrine\Common\Collections\ArrayCollection;
use Enlight_Event_EventManager;
use Enlight_Hook;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\WarehouseArticleDetailStockCount;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware_Components_Snippet_Manager;
use Zend_Db_Adapter_Pdo_Abstract;

class WarehouseIntegrityService implements WarehouseIntegrity, Enlight_Hook
{
    /**
     * The name of the collect event that is fired when determining whether a warehouse can be deleted.
     */
    const EVENT_NAME_COLLECT_WAREHOUSE_DELETION_CONSTRAINT_VIOLATIONS = 'ViisonPickwareERP_WarehouseIntegrityService_CollectWarehouseDeletionConstraintViolations';

    /**
     * @var ModelManager
     */
    protected $entityManager;

    /**
     * @var Zend_Db_Adapter_Pdo_Abstract
     */
    protected $database;

    /**
     * @var Enlight_Event_EventManager
     */
    protected $eventManager;

    /**
     * @var Shopware_Components_Snippet_Manager
     */
    protected $snippetManager;

    /**
     * @param ModelManager $entityManager
     * @param Zend_Db_Adapter_Pdo_Abstract $database
     * @param Enlight_Event_EventManager $eventManager
     * @param Shopware_Components_Snippet_Manager $snippetManager
     */
    public function __construct($entityManager, $database, $eventManager, $snippetManager)
    {
        $this->entityManager = $entityManager;
        $this->database = $database;
        $this->eventManager = $eventManager;
        $this->snippetManager = $snippetManager;
    }

    /**
     * @inheritdoc
     */
    public function ensureSingleDefaultWarehouse()
    {
        /** @var Warehouse[] $allWarehouses */
        $allWarehouses = $this->entityManager->getRepository(Warehouse::class)->findAll();
        if (count($allWarehouses) < 1) {
            return;
        }

        // Sort warehouses by ID so when we have to decide a new default warehouse, we pick the oldest by default
        usort(
            $allWarehouses,
            function (Warehouse $lhs, Warehouse $rhs) {
                return $lhs->getId() - $rhs->getId();
            }
        );

        $defaultWarehouses = array_filter(
            $allWarehouses,
            function (Warehouse $warehouse) {
                return $warehouse->isDefaultWarehouse();
            }
        );

        if (count($defaultWarehouses) < 1) {
            // No default warehouse set - make the one with the lowest ID our new default warehouse
            $allWarehouses[0]->setDefaultWarehouse(true);
        } elseif (count($defaultWarehouses) > 1) {
            // More than one default warehouse set - set the one with the lowest ID as the only default warehouse
            /** @var Warehouse $shouldNotBeDefaultWarehouse */
            foreach (array_slice($defaultWarehouses, 1) as $shouldNotBeDefaultWarehouse) {
                $shouldNotBeDefaultWarehouse->setDefaultWarehouse(false);
            }
        }

        // Save any changes we may have made to the warehosues
        $this->entityManager->flush($allWarehouses);
    }

    /**
     * @inheritDoc
     */
    public function ensureSingleDefaultReturnShipmentWarehouse()
    {
        /** @var Warehouse[] $allWarehouses */
        $allWarehouses = $this->entityManager->getRepository(Warehouse::class)->findAll();
        if (count($allWarehouses) < 1) {
            return;
        }

        // Sort warehouses by ID so when we have to decide a new default warehouse, we pick the oldest by default
        usort(
            $allWarehouses,
            function (Warehouse $lhs, Warehouse $rhs) {
                return $lhs->getId() - $rhs->getId();
            }
        );

        $defaultReturnShipmentWarehouses = array_filter(
            $allWarehouses,
            function (Warehouse $warehouse) {
                return $warehouse->isDefaultReturnShipmentWarehouse();
            }
        );

        if (count($defaultReturnShipmentWarehouses) < 1) {
            // No default warehouse set - make the one with the lowest ID our new default warehouse
            $allWarehouses[0]->setDefaultReturnShipmentWarehouse(true);
        } elseif (count($defaultReturnShipmentWarehouses) > 1) {
            // More than one default warehouse set - set the one with the lowest ID as the only default warehouse
            /** @var Warehouse $shouldNotBeDefaultWarehouse */
            foreach (array_slice($defaultReturnShipmentWarehouses, 1) as $shouldNotBeDefaultWarehouse) {
                $shouldNotBeDefaultWarehouse->setDefaultReturnShipmentWarehouse(false);
            }
        }

        // Save any changes we may have made to the warehouses
        $this->entityManager->flush($allWarehouses);
    }

    /**
     * @inheritdoc
     */
    public function ensureMappingsForAllArticleDetailsInWarehouse(Warehouse $warehouse)
    {
        $this->database->query(
            'INSERT IGNORE INTO `pickware_erp_article_detail_bin_location_mappings` (`binLocationId`, `articleDetailId`)
                SELECT :nullBinLocationId, `s_articles_details`.`id`
                FROM `s_articles_details`',
            [
                'nullBinLocationId' => $warehouse->getNullBinLocation()->getId(),
            ]
        );
        $this->database->query(
            'INSERT IGNORE INTO `pickware_erp_warehouse_article_detail_stock_counts` (`warehouseId`, `articleDetailId`)
                SELECT :warehouseId, `s_articles_details`.`id`
                FROM `s_articles_details`',
            [
                'warehouseId' => $warehouse->getId(),
            ]
        );
        $this->database->query(
            'INSERT IGNORE INTO `pickware_erp_warehouse_article_detail_configurations` (`warehouseId`, `articleDetailId`)
                SELECT :warehouseId, `s_articles_details`.`id`
                FROM `s_articles_details`',
            [
                'warehouseId' => $warehouse->getId(),
            ]
        );
    }

    /**
     * @inheritdoc
     */
    public function ensureAllWarehouseMappingsForArticleDetail(ArticleDetail $articleDetail)
    {
        /** @var Warehouse[] $warehouses */
        $warehouses = $this->entityManager->getRepository(Warehouse::class)->findAll();
        foreach ($warehouses as $warehouse) {
            // Check for existing mappings in the warehouse
            $builder = $this->entityManager->createQueryBuilder();
            $builder->select('binLocationMapping')
                    ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
                    ->join('binLocationMapping.binLocation', 'binLocation')
                    ->where('binLocationMapping.articleDetailId = :articleDetailId')
                    ->andWhere('binLocation.warehouseId = :warehouseId')
                    ->setParameter('articleDetailId', $articleDetail->getId())
                    ->setParameter('warehouseId', $warehouse->getId());
            $existingMappings = $builder->getQuery()->getResult();
            if (count($existingMappings) == 0) {
                // Create a mapping to the null bin location
                $binLocationMapping = new ArticleDetailBinLocationMapping(
                    $warehouse->getNullBinLocation(),
                    $articleDetail
                );
                $this->entityManager->persist($binLocationMapping);
                $this->entityManager->flush($binLocationMapping);
            }

            // Make sure a warehouse stock cache exists
            $warehouseStock = $this->entityManager->getRepository(WarehouseArticleDetailStockCount::class)->findOneBy([
                'warehouse' => $warehouse,
                'articleDetail' => $articleDetail,
            ]);
            if (!$warehouseStock) {
                $warehouseStock = new WarehouseArticleDetailStockCount($warehouse, $articleDetail);
                $this->entityManager->persist($warehouseStock);
                $this->entityManager->flush($warehouseStock);
            }

            // Make sure a warehouse configuration exists
            $warehouseConfig = $this->entityManager->getRepository(WarehouseArticleDetailConfiguration::class)->findOneBy([
                'warehouse' => $warehouse,
                'articleDetail' => $articleDetail,
            ]);
            if (!$warehouseConfig) {
                $warehouseConfig = new WarehouseArticleDetailConfiguration($warehouse, $articleDetail);
                $this->entityManager->persist($warehouseConfig);
                $this->entityManager->flush($warehouseConfig);
            }
        }
    }

    /**
     * @inheritdoc
     */
    public function evaluateWarehouseDeletionConstraints(Warehouse $warehouse)
    {
        $constraintViolations = [];
        $snippetNamespace = $this->snippetManager->getNamespace(
            'viison_pickware_erp/components/warehouse/warehouse_integrity'
        );

        if ($warehouse->isDefaultWarehouse()) {
            $constraintViolations[] = $snippetNamespace->get(
                'warehouse_deletion_constraint_violation/is_default_warehouse'
            );
        }

        if ($warehouse->isDefaultReturnShipmentWarehouse()) {
            $constraintViolations[] = $snippetNamespace->get(
                'warehouse_deletion_constraint_violation/is_default_return_shipment_warehouse'
            );
        }

        // Check for any stock in the warehouse
        $builder = $this->entityManager->createQueryBuilder();
        $builder->select(
            'COUNT(binLocationMapping.id)'
        )->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->join('binLocationMapping.binLocation', 'binLocation')
            ->where('binLocationMapping.stock > 0')
            ->andWhere('binLocation.warehouseId = :warehouseId')
            ->setParameter('warehouseId', $warehouse->getId());
        $binLocationsWithStock = $builder->getQuery()->getSingleScalarResult();
        if (intval($binLocationsWithStock) > 0) {
            $constraintViolations[] = $snippetNamespace->get(
                'warehouse_deletion_constraint_violation/warehouse_contains_stock'
            );
        }

        // Allow other plugins to validate the warehouse deletion
        $eventResult = $this->eventManager->collect(
            self::EVENT_NAME_COLLECT_WAREHOUSE_DELETION_CONSTRAINT_VIOLATIONS,
            new ArrayCollection(),
            [
                'subject' => $this,
                'warehouse' => $warehouse,
            ]
        );
        $constraintViolations = array_merge(
            $constraintViolations,
            $eventResult->toArray()
        );

        return $constraintViolations;
    }
}
