<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonCommon\Classes\ApiException;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\ApiExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Controllers\ControllerEvents;

class Shopware_Controllers_Api_ViisonPickwareERPBinLocations extends Shopware_Controllers_Api_Rest
{
    use ApiExceptionHandling;

    /**
     * Fetches all bin locations with the given sort, filter and pagination limits (offset and limit).
     *
     * GET /api/binLocations
     */
    public function indexAction()
    {
        $filter = $this->Request()->getParam('filter', []);
        $sort = $this->Request()->getParam('sort', []);
        $offset = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 1000);

        $queryBuilder = $this->get('models')->createQueryBuilder();
        $queryBuilder
            ->select('binLocation')
            ->from(BinLocation::class, 'binLocation')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        $query = $queryBuilder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $binLocations = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $binLocations,
            'total' => $totalResult,
        ]);
    }

    /**
     * GET /api/binLocations/{id}
     */
    public function getAction()
    {
        try {
            $binLocation = $this->getBinLocationFromRequest();

            $queryBuilder = $this->get('models')->createQueryBuilder();
            $queryBuilder
                ->select(
                    'binLocation',
                    'articleDetailBinLocationMappings'
                )
                ->from(BinLocation::class, 'binLocation')
                ->leftJoin('binLocation.articleDetailBinLocationMappings', 'articleDetailBinLocationMappings')
                ->where('binLocation = :binLocation')
                ->setParameter('binLocation', $binLocation);
            $binLocations = $queryBuilder->getQuery()->getArrayResult();
            $binLocationArray = $binLocations[0];

            foreach ($binLocationArray['articleDetailBinLocationMappings'] as &$binLocationMapping) {
                $binLocationMapping['lastStocktake'] = $binLocationMapping['lastStocktake'] ? $binLocationMapping['lastStocktake']->format(DateTime::ISO8601) : null;
            }
            unset($binLocationMapping);

            $this->View()->assign([
                'success' => true,
                'data' => $binLocationArray,
            ]);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * POST /api/binLocations
     */
    public function postAction()
    {
        try {
            $binLocationPostData = $this->Request()->getPost();
            $warehouseId = $binLocationPostData['warehouseId'];
            ParameterValidator::assertIsNotNull($warehouseId, 'warehouseId');
            $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
            ParameterValidator::assertEntityFound(
                $warehouse,
                Warehouse::class,
                $warehouseId,
                'warehouseId'
            );
            $this->validateBinLocationCodeIsAvailable($binLocationPostData['code'], $warehouse);

            // Create a new bin location in the warehouse using the given code
            $binLocation = new BinLocation($warehouse, $binLocationPostData['code']);

            // Save changes
            $this->get('models')->persist($binLocation);
            $this->get('models')->flush($binLocation);

            $this->Response()->setHttpResponseCode(201);
            $this->View()->assign([
                'success' => true,
                'data' => [
                    'id' => $binLocation->getId(),
                    'location' => sprintf('%s/binLocations/%d', $this->apiBaseUrl, $binLocation->getId()),
                ],
            ]);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * Updates a bin location by updating the code only.
     *
     * PUT /api/binLocations/{id}
     */
    public function putAction()
    {
        try {
            $binLocation = $this->getBinLocationFromRequest();
            $binLocationPostData = $this->Request()->getPost();
            $this->validateBinLocationCodeIsAvailable($binLocationPostData['code'], $binLocation->getWarehouse());

            $binLocation->setCode($binLocationPostData['code']);
            $this->get('models')->flush($binLocation);

            $this->View()->assign('success', true);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * Deletes a bin location. This is not possible for the null bin location or bin locations that have article details
     * still assigned to them.
     *
     * DELETE /api/binLocations/{id}
     */
    public function deleteAction()
    {
        try {
            $binLocation = $this->getBinLocationFromRequest();
            if ($binLocation->isNullBinLocation()) {
                throw new CustomValidationException(
                    'id',
                    $binLocation->getId(),
                    sprintf(
                        (
                            'Cannot remove bin location because it is the "unknown" bin location of the warehouse "%s".'
                            . ' "unknown" bin locations are automatically removed when removing their respective '
                            . 'warehouse.'
                        ),
                        $binLocation->getWarehouse()->getCode()
                    )
                );
            }
            if (!$binLocation->getArticleDetailBinLocationMappings()->isEmpty()) {
                throw new CustomValidationException(
                    'id',
                    $binLocation->getId(),
                    sprintf(
                        (
                            'Bin location "%s" cannot be removed because there are still some article details mapped '
                            . 'to it. This can be caused by stock of an article detail residing on the bin location or '
                            . 'the bin location being selected as an article detail\'s default bin location.'
                        ),
                        $binLocation->getCode()
                    )
                );
            }

            // Fire event to let other plugins prevent the deletion of the bin location (e.g. Shopware WMS, if there
            // exist any picked quantities referncing the bin location)
            $eventResult = $this->get('events')->notifyUntil(
                ControllerEvents::NOTIFY_UNTIL_PREVENT_BIN_LOCATION_DELETION,
                [
                    'binLocation' => $binLocation,
                ]
            );
            if ($eventResult && $eventResult->getReturn() !== null) {
                throw new ApiException(
                    sprintf(
                        'Bin location "%s" cannot be deleted. %s',
                        $binLocation->getCode(),
                        $eventResult->getReturn()
                    ),
                    406
                );
            }

            $this->get('models')->remove($binLocation);
            $this->get('models')->flush($binLocation);

            $this->View()->assign([
                'success' => true,
            ]);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * Fetches and returns the `BinLocation` having the id of the processed request. Throws an exception if no such bin
     * location was found.
     *
     * @return BinLocation
     */
    private function getBinLocationFromRequest()
    {
        $binLocationId = $this->Request()->getParam('id');
        ParameterValidator::assertIsNotNull($binLocationId, 'id');
        $binLocation = $this->get('models')->find(BinLocation::class, $binLocationId);
        if (!$binLocation) {
            // Since the bin location is the target entity of this API request, use custom ApiException with error code
            // 404 instead of ParameterValidator::assertEntityFound which would cause an error code 400.
            throw new ApiException(
                sprintf('The bin location identified by parameter id=%d was not found.', $binLocationId),
                404
            );
        }

        return $binLocation;
    }

    /**
     * Validates the given bin location code for the given warehouse.
     *
     * Throws an exception if the code is missing, is equal to the warehouse's null bin location code (which is
     * forbidden), or if the code already exists in the given warehouse.
     *
     * @param string $binLocationCode
     * @param Warehouse $warehouse
     */
    private function validateBinLocationCodeIsAvailable($binLocationCode, Warehouse $warehouse)
    {
        ParameterValidator::assertIsNotEmptyString($binLocationCode, 'code');

        if (mb_strtolower($binLocationCode) === mb_strtolower(BinLocation::NULL_BIN_LOCATION_CODE)) {
            throw new CustomValidationException(
                'code',
                $binLocationCode,
                sprintf('The bin location code "%s" is invalid, because it is a reserved value.', $binLocationCode)
            );
        }
        $binLocation = $this->get('models')->getRepository(BinLocation::class)->findOneBy([
            'code' => $binLocationCode,
            'warehouse' => $warehouse,
        ]);
        if ($binLocation) {
            throw new CustomValidationException(
                'code',
                $binLocationCode,
                sprintf('The bin location with code "%s" already exists in this warehouse.', $binLocationCode)
            );
        }
    }
}
