<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\Api\Manager as ResourceManager;
use Shopware\Components\Model\ModelManager;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityValidator\OrderDetailQuantityValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\AbstractRestApiBatchController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockMovement\StockLocationFactoryService;

class Shopware_Controllers_Api_ViisonPickwareERPOrderDetails extends AbstractRestApiBatchController
{
    /**
     * PUT /api/orderDetails/ship/{idempotencyKey}
     */
    public function putShipAction()
    {
        ResourceManager::getResource('order')->checkPrivilege('update');

        $this->executeBatchAction(
            function ($dataset, $datasetKey) {
                self::validateShipOperationsData($dataset, $datasetKey);
            },
            function (array $dataset) {
                return $this->executeShipOperation($dataset);
            }
        );
    }

    /**
     * @param mixed $dataset
     * @param int $datasetKey
     */
    private static function validateShipOperationsData($dataset, $datasetKey)
    {
        $path = '[' . $datasetKey . ']';

        ParameterValidator::assertIsArray($dataset, $path);
        ParameterValidator::assertIsInteger($dataset['orderDetailId'], $path . '.orderDetailId');
        ParameterValidator::assertIsInteger($dataset['quantity'], $path . '.quantity');
        StockLocationFactoryService::validateStockLocationArray($dataset, $path);
    }

    /**
     * @param array $dataset
     * @return array
     */
    private function executeShipOperation(array $dataset)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var StockLocationFactoryService $stockLocationFactoryService */
        $stockLocationFactoryService = $this->get('pickware.erp.stock_location_factory_service');
        /** @var OrderDetailQuantityValidator $orderDetailQuantityValidator */
        $orderDetailQuantityValidator = $this->get('pickware.erp.order_detail_quantity_validator_service');

        /** @var OrderDetail|null $orderDetail */
        $orderDetail = $entityManager->find(OrderDetail::class, $dataset['orderDetailId']);
        if (!$orderDetail) {
            return [
                'success' => false,
                'message' => sprintf('The passed order detail with id=%d does not exist.', $dataset['orderDetailId']),
            ];
        }

        try {
            $newShippedQuantity = $orderDetail->getShipped() + $dataset['quantity'];
            $orderDetailQuantityValidator->validateQuantityAndShippedQuantityCombination(
                $orderDetail,
                $orderDetail->getQuantity(),
                $newShippedQuantity
            );

            $orderDetail->setShipped($newShippedQuantity);
            $entityManager->flush($orderDetail);

            $location = $stockLocationFactoryService->createStockLocationFromArray($dataset);
            $location->shipStockForOrderDetail($orderDetail, $dataset['quantity']);
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }

        return [
            'success' => true,
        ];
    }
}
