<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\Api\Manager as ResourceManager;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\DetailStatus as OrderDetailStatus;
use Shopware\Models\Order\Order;
use Shopware\Models\Order\Status as OrderStatus;
use Shopware\Plugins\ViisonCommon\Classes\ApiException;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\ApiExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Classes\Util\OrderDetailUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockUpdaterService;
use Shopware\Plugins\ViisonPickwareERP\Components\StockMovement\StockLocationFactoryService;

class Shopware_Controllers_Api_ViisonPickwareERPOrders extends Shopware_Controllers_Api_Rest
{
    use ApiExceptionHandling;

    /**
     * PUT /api/orders/{id}/ship
     *
     * Marks an order as completely shipped and clears the stock from the specified warehouse.
     *
     * The parameter "suppressStatusMail" can be supplied to suppress the dispatch of a status mail.
     *
     * @throws ApiException
     */
    public function putShipAction()
    {
        try {
            ResourceManager::getResource('order')->checkPrivilege('update');

            $this->validatePutShipAction();
            $this->executePutShipAction();
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    private function validatePutShipAction()
    {
        $postData = $this->Request()->getPost();

        ParameterValidator::assertIsArray($postData, '');
        ParameterValidator::assertArrayElementIsArray($postData, 'warehouse');
        StockLocationFactoryService::validateWarehouseStockLocationArray($postData);
    }

    private function executePutShipAction()
    {
        /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculator */
        $orderDetailQuantityCalculator = $this->get('pickware.erp.order_detail_quantity_calculator_service');
        /** @var StockChangeListFactory $stockChangeListFactory */
        $stockChangeListFactory = $this->get('pickware.erp.stock_change_list_factory_service');
        /** @var StockUpdaterService $stockUpdater */
        $stockUpdater = $this->get('pickware.erp.stock_updater_service');
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $order = $this->getOrderForRequest();
        $postData = $this->Request()->getPost();

        /** @var Warehouse $warehouse */
        $warehouse = $entityManager->getRepository(Warehouse::class)->findOneBy($postData['warehouse']);
        ParameterValidator::assertEntityFound($warehouse, Warehouse::class, '[object]', 'warehouse');

        /** @var OrderDetailStatus $detailStatusCompleted */
        $detailStatusCompleted = $entityManager->find(
            OrderDetailStatus::class,
            OrderDetailUtil::ORDER_DETAIL_STATUS_ID_COMPLETED
        );
        /** @var OrderDetail $orderDetail */
        foreach ($order->getDetails() as $orderDetail) {
            if (!OrderDetailUtil::isOrderDetailForArticleDetail($orderDetail)) {
                continue;
            }

            $remainingQuantityToShip = $orderDetailQuantityCalculator->calculateRemainingQuantityToShip($orderDetail);
            $orderDetail->setStatus($detailStatusCompleted);
            $orderDetail->setShipped($orderDetail->getShipped() + $remainingQuantityToShip);

            $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
            if ($articleDetail && $remainingQuantityToShip !== 0) {
                $stockChanges = $stockChangeListFactory->createStockChangeList(
                    $warehouse,
                    $articleDetail,
                    -1 * $remainingQuantityToShip
                );
                $stockUpdater->recordOrderDetailShippedChange(
                    $articleDetail,
                    $orderDetail,
                    $stockChanges
                );
            }
        }
        $entityManager->flush($order->getDetails()->toArray());

        if ($order->getOrderStatus()->getId() !== OrderStatus::ORDER_STATE_COMPLETELY_DELIVERED) {
            /** @var OrderStatus $orderStatusCompletelyDelivered */
            $orderStatusCompletelyDelivered = $entityManager->find(
                OrderStatus::class,
                OrderStatus::ORDER_STATE_COMPLETELY_DELIVERED
            );
            $order->setOrderStatus($orderStatusCompletelyDelivered);
            $entityManager->flush($order);

            $suppressStatusMail = isset($postData['suppressStatusMail']) ? $postData['suppressStatusMail'] : false;
            if (!$suppressStatusMail) {
                /** @var sOrder $orderModule */
                $orderModule = $this->get('modules')->getModule('Order');
                $mail = $orderModule->createStatusMail($order->getId(), OrderStatus::ORDER_STATE_COMPLETELY_DELIVERED);
                if ($mail) {
                    $orderModule->sendStatusMail($mail);
                }
            }
        }

        $this->View()->success = true;
    }

    /**
     * @return Order
     * @throws ApiException
     */
    private function getOrderForRequest()
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $idParam = $this->Request()->getParam('id');
        $identifierIsOrderNumber = boolval($this->Request()->getParam('useNumberAsId', 0));
        /** @var Order|null $order */
        $order = null;
        if ($identifierIsOrderNumber) {
            ParameterValidator::assertIsNotNull($idParam, 'id');
            $order = $entityManager->getRepository(Order::class)->findOneBy([
                'number' => $idParam,
            ]);
        } else {
            ParameterValidator::assertIsInteger($idParam, 'id');
            $order = $entityManager->find(Order::class, $idParam);
        }
        if (!$order) {
            // Explicitly throw an ApiException with error code 404 instead of an AbstractValidationException.
            // AbstractValidationException will be converted to an API exception with error code 400 in
            // \Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\ApiExceptionHandling::handleException but we
            // want it to be a 404 because the requested resource (here: order) was not found.
            throw new ApiException(
                sprintf('Entity of class "%s" identified by parameter id=%s was not found.', Order::class, $idParam),
                404
            );
        }

        return $order;
    }
}
