<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\Api\Manager as ResourceManager;
use Shopware\Components\Model\ModelManager;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\AbstractRestApiBatchController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockMovement\StockMovementException;
use Shopware\Plugins\ViisonPickwareERP\Components\StockMovement\StockLocationFactoryService;

class Shopware_Controllers_Api_ViisonPickwareERPStock extends AbstractRestApiBatchController
{
    const QUANTITY_ALL = 'all';

    /**
     * PUT /api/stock/change/{idempotencyKey}
     */
    public function putChangeAction()
    {
        ResourceManager::getResource('variant')->checkPrivilege('update');

        $this->executeBatchAction(
            function ($dataset, $datasetKey) {
                self::validateStockSetOrChangeOperationDataset($dataset, $datasetKey);

                if (intval($dataset['quantity']) === 0) {
                    throw new CustomValidationException(
                        sprintf('[%s].quantity', $datasetKey),
                        $dataset['quantity'],
                        'The quantity 0 is not allowed for a stock change operation.'
                    );
                }
            },
            function ($dataset) {
                return $this->executeChangeStockOperation($dataset);
            }
        );
    }

    /**
     * @param mixed $dataset
     * @param int $datasetKey
     */
    private static function validateStockSetOrChangeOperationDataset($dataset, $datasetKey)
    {
        $path = '[' . $datasetKey . ']';

        ParameterValidator::assertIsArray($dataset, $path);
        ParameterValidator::assertIsInteger($dataset['quantity'], $path . '.quantity');
        ParameterValidator::assertIsInteger($dataset['articleDetailId'], $path . '.articleDetailId');
        StockLocationFactoryService::validateStockLocationArray($dataset, $path);
    }

    /**
     * @param array $dataset
     * @return array
     */
    private function executeChangeStockOperation(array $dataset)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var StockLocationFactoryService $restApiStockService */
        $restApiStockService = $this->get('pickware.erp.stock_location_factory_service');

        /** @var ArticleDetail|null $articleDetail */
        $articleDetail = $entityManager->find(ArticleDetail::class, $dataset['articleDetailId']);
        if (!$articleDetail) {
            return [
                'success' => false,
                'message' => sprintf('The passed order detail with id=%d does not exist.', $dataset['articleDetailId']),
            ];
        }

        try {
            $location = $restApiStockService->createStockLocationFromArray($dataset);
            $location->changeStockOfArticleDetail($articleDetail, $dataset['quantity']);
        } catch (StockMovementException $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }

        return [
            'success' => true,
        ];
    }

    /**
     * PUT /api/stock/set/{idempotencyKey}
     */
    public function putSetAction()
    {
        ResourceManager::getResource('variant')->checkPrivilege('update');

        $this->executeBatchAction(
            function ($dataset, $datasetKey) {
                self::validateStockSetOrChangeOperationDataset($dataset, $datasetKey);

                if ($dataset['quantity'] < 0) {
                    throw new CustomValidationException(
                        sprintf('[%s].quantity', $datasetKey),
                        $dataset['quantity'],
                        'The quantity for a stock set operation must not be less than 0.'
                    );
                }
            },
            function (array $dataset) {
                return $this->executeSetStockOperation($dataset);
            }
        );
    }

    /**
     * @param array $dataset
     * @return array
     */
    private function executeSetStockOperation(array $dataset)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var StockLocationFactoryService $restApiStockService */
        $restApiStockService = $this->get('pickware.erp.stock_location_factory_service');

        /** @var ArticleDetail|null $articleDetail */
        $articleDetail = $entityManager->find(ArticleDetail::class, $dataset['articleDetailId']);
        if (!$articleDetail) {
            return [
                'success' => false,
                'message' => sprintf('The passed article detail with id=%d does not exist.', $dataset['articleDetailId']),
            ];
        }

        try {
            $location = $restApiStockService->createStockLocationFromArray($dataset);
            $location->setStockOfArticleDetail($articleDetail, $dataset['quantity']);
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }

        return [
            'success' => true,
        ];
    }

    /**
     * PUT /api/stock/relocate/{idempotencyKey}
     */
    public function putRelocateAction()
    {
        ResourceManager::getResource('variant')->checkPrivilege('update');

        $this->executeBatchAction(
            function ($dataset, $datasetKey) {
                self::validateStockRelocateOperationsDataset($dataset, $datasetKey);
            },
            function (array $dataset) {
                return $this->executeRelocateStockOperation($dataset);
            }
        );
    }

    /**
     * @param mixed $dataset
     * @param int $datasetKey
     */
    private static function validateStockRelocateOperationsDataset($dataset, $datasetKey)
    {
        $path = '[' . $datasetKey . ']';

        ParameterValidator::assertIsArray($dataset, $path);
        ParameterValidator::assertIsInteger($dataset['articleDetailId'], $path . '.articleDetailId');
        ParameterValidator::assertIsNotNull($dataset['quantity'], $path . '.quantity');
        if ($dataset['quantity'] !== self::QUANTITY_ALL) {
            ParameterValidator::assertIsInteger($dataset['quantity'], $path . '.quantity');
        }
        ParameterValidator::assertIsArray($dataset['source'], $path . '.source');
        ParameterValidator::assertIsArray($dataset['destination'], $path . '.destination');
        StockLocationFactoryService::validateStockLocationArray($dataset['source'], $path . '.source');
        StockLocationFactoryService::validateStockLocationArray($dataset['destination'], $path . '.destination');
    }

    /**
     * @param array $dataset
     * @return array
     */
    public function executeRelocateStockOperation(array $dataset)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var StockLocationFactoryService $restApiStockService */
        $restApiStockService = $this->get('pickware.erp.stock_location_factory_service');

        /** @var ArticleDetail|null $articleDetail */
        $articleDetail = $entityManager->find(ArticleDetail::class, $dataset['articleDetailId']);
        if (!$articleDetail) {
            return [
                'success' => false,
                'message' => sprintf(
                    'The passed article detail with id=%d does not exist.',
                    $dataset['articleDetailId']
                ),
            ];
        }

        try {
            $source = $restApiStockService->createStockLocationFromArray($dataset['source']);
            $destination = $restApiStockService->createStockLocationFromArray($dataset['destination']);

            if ($dataset['quantity'] === self::QUANTITY_ALL) {
                $source->moveEntireStockOfArticleDetailToLocation($articleDetail, $destination);
            } else {
                $source->moveStockOfArticleDetailToLocation($articleDetail, $destination, $dataset['quantity']);
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }

        return [
            'success' => true,
        ];
    }
}
