<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonCommon\Classes\ApiException;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\ApiExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;

class Shopware_Controllers_Api_ViisonPickwareERPWarehouses extends Shopware_Controllers_Api_Rest
{
    use ApiExceptionHandling;

    /**
     * Fetches all warehouses with the given sort, filter and pagination limits (offset and limit).
     *
     * GET /api/warehouses
     */
    public function indexAction()
    {
        $filter = $this->Request()->getParam('filter', []);
        $sort = $this->Request()->getParam('sort', []);
        $offset = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 1000);

        $queryBuilder = $this->get('models')->createQueryBuilder();
        $queryBuilder
            ->select('warehouse')
            ->from(Warehouse::class, 'warehouse')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        $query = $queryBuilder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $warehouses = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $warehouses,
            'total' => $totalResult,
        ]);
    }

    /**
     * GET /api/warehouses/{id}/
     */
    public function getAction()
    {
        $warehouse = $this->getWarehouseFromRequest();

        // Fetch the warehouse anew since we need an array result
        $queryBuilder = $this->get('models')->createQueryBuilder();
        $queryBuilder
            ->select('warehouse')
            ->from(Warehouse::class, 'warehouse')
            ->where('warehouse = :warehouse')
            ->setParameter('warehouse', $warehouse);
        $warehouses = $queryBuilder->getQuery()->getArrayResult();

        $this->View()->assign([
            'success' => true,
            'data' => $warehouses[0],
        ]);
    }

    /**
     * POST /api/warehouses
     */
    public function postAction()
    {
        try {
            $warehousePostData = $this->Request()->getPost();
            ParameterValidator::assertIsNotEmptyString($warehousePostData['code'], 'code');
            ParameterValidator::assertIsNotEmptyString($warehousePostData['name'], 'name');
            $this->validateWarehouseData($warehousePostData);

            // Create warehouse
            $warehouse = new Warehouse();
            $warehouse->fromArray($warehousePostData);
            $this->get('models')->persist($warehouse);

            // Create and add a new null bin location
            $nullBinLocation = new BinLocation($warehouse, BinLocation::NULL_BIN_LOCATION_CODE);
            $warehouse->setNullBinLocation($nullBinLocation);
            $this->get('models')->persist($nullBinLocation);

            $this->get('models')->flush([
                $warehouse,
                $nullBinLocation,
            ]);
            // Refresh warehouse entity, so it is persisted and has an id, so the remaining tasks can be performed
            // (ensureMappingsForAllArticleDetailsInWarehouse)
            $this->get('models')->refresh($warehouse);

            // Assign all existing article details to the default bin location
            $this->get('pickware.erp.warehouse_integrity_service')->ensureMappingsForAllArticleDetailsInWarehouse(
                $warehouse
            );

            $this->View()->assign([
                'success' => true,
                'data' => [
                    'id' => $warehouse->getId(),
                    'location' => sprintf('%s/warehouses/%d', $this->apiBaseUrl, $warehouse->getId()),
                ],
            ]);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * PUT /api/warehouses/{id}/
     */
    public function putAction()
    {
        try {
            $warehouse = $this->getWarehouseFromRequest();

            $warehousePostData = $this->Request()->getPost();
            if (array_key_exists('code', $warehousePostData)) {
                ParameterValidator::assertIsNotEmptyString($warehousePostData['code'], 'code');
            }
            if (array_key_exists('name', $warehousePostData)) {
                ParameterValidator::assertIsNotEmptyString($warehousePostData['name'], 'name');
            }
            $this->validateWarehouseData($warehousePostData);

            $warehouse = $warehouse->fromArray($warehousePostData);
            $this->get('models')->flush($warehouse);

            $this->View()->assign('success', true);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * DELETE /api/warehouses/{id}/
     */
    public function deleteAction()
    {
        try {
            $warehouse = $this->getWarehouseFromRequest();
            $this->validateWarehouseCanBeDeleted($warehouse);

            $this->get('models')->remove($warehouse);
            $this->get('models')->flush($warehouse);

            $this->View()->assign([
                'success' => true,
            ]);
        } catch (Exception $e) {
            throw $this->wrapExceptionIntoApiException($e);
        }
    }

    /**
     * Fetches and returns the `Warehouse` having the id of the processed request. Throws an exception if no such
     * warehouse was found.
     *
     * @return Warehouse
     */
    private function getWarehouseFromRequest()
    {
        $warehouseId = $this->Request()->getParam('id');
        ParameterValidator::assertIsNotNull($warehouseId, 'id');
        $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
        if (!$warehouse) {
            // Since the warehouse is the target entity of this API request, use custom ApiException with error code
            // 404 instead of ParameterValidator::assertEntityFound which would cause an error code 400.
            throw new ApiException(
                sprintf('The warehouse identified by parameter id=%d was not found.', $warehouseId),
                404
            );
        }

        return $warehouse;
    }

    /**
     * Validates the given warehouse post data. Asserts that no required parameter is empty and no forbidden
     * parameter is set.
     *
     * @param array $data
     */
    private function validateWarehouseData(array $data)
    {
        $this->validateWarehouseDoesNotAlreadyExist($data);

        if (array_key_exists('defaultWarehouse', $data)) {
            throw new CustomValidationException(
                'defaultWarehouse',
                $data['defaultWarehouse'],
                'The default warehouse cannot be changed using the REST API.'
            );
        }
        if (array_key_exists('defaultReturnShipmentWarehouse', $data)) {
            throw new CustomValidationException(
                'defaultReturnShipmentWarehouse',
                $data['defaultReturnShipmentWarehouse'],
                'The default return shipment warehouse cannot be changed using the REST API.'
            );
        }
        if (array_key_exists('nullBinLocationId', $data)) {
            throw new CustomValidationException(
                'nullBinLocationId',
                $data['nullBinLocationId'],
                'The "unknown"/"null" bin location of a warehouse cannot be changed.'
            );
        }
    }

    /**
     * Validates that the given warehouse is ready to be deleted.
     *
     * @param Warehouse $warehouse
     */
    private function validateWarehouseCanBeDeleted(Warehouse $warehouse)
    {
        if ($warehouse->isDefaultWarehouse()) {
            throw new ApiException('The default warehouse cannot be deleted.', 400);
        }
        if ($warehouse->isDefaultReturnShipmentWarehouse()) {
            throw new ApiException('The default return shipment warehouse cannot be deleted.', 400);
        }
        // Check that no bin location (other than the null bin location) is assigned to the warehouse
        if ($warehouse->getBinLocations()->count() > 1) {
            throw new ApiException('The warehouse cannot be deleted because it still has bin locations.', 400);
        }

        // Check that no bin location has any stock or stock reservations in the warehouse
        $nonEmptyBinLocationMappingsCount = $this->get('db')->fetchOne(
            'SELECT COUNT(*)
            FROM `pickware_erp_article_detail_bin_location_mappings`
            INNER JOIN `pickware_erp_bin_locations`
                ON `pickware_erp_bin_locations`.`id` = `pickware_erp_article_detail_bin_location_mappings`.`binLocationId`
            WHERE
                `pickware_erp_bin_locations`.`warehouseId` = :warehouseId
                AND (
                    `pickware_erp_article_detail_bin_location_mappings`.`stock` != 0
                    OR `pickware_erp_article_detail_bin_location_mappings`.`reservedStock` != 0
                )
            ',
            [
                'warehouseId' => $warehouse->getId(),
            ]
        );
        if (intval($nonEmptyBinLocationMappingsCount) > 0) {
            throw new ApiException(
                'The warehouse cannot be deleted because it still contains stock or stock reservations.',
                400
            );
        }
    }

    /**
     * Validates the given warehouse post data.
     *
     * Throws exception if the data is invalid because a warehouse with the same code or name exists.
     *
     * @param array $data
     */
    private function validateWarehouseDoesNotAlreadyExist(array $data)
    {
        $warehouse = $this->get('models')->getRepository(Warehouse::class)->findOneBy([
            'code' => $data['code'],
        ]);
        if ($warehouse) {
            throw new CustomValidationException(
                'code',
                $data['code'],
                sprintf(
                    'The code "%s" is already in use by another warehouse. Warehouse codes must be unique.',
                    $data['code']
                )
            );
        }
        $warehouse = $this->get('models')->getRepository(Warehouse::class)->findOneBy([
            'name' => $data['name'],
        ]);
        if ($warehouse) {
            throw new CustomValidationException(
                'name',
                $data['name'],
                sprintf(
                    'The name "%s" is already in use by another warehouse. The names of warehouses must be unique.',
                    $data['name']
                )
            );
        }
    }
}
