<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

/**
 * This class receives the requests from the custom analytics backend controllers.
 * It fetches based on the request several sales information and returns it to the view.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPAnalyticsArticle extends Shopware_Controllers_Backend_Analytics
{

    /**
     * Fetches all sales made in the timespan between the two given dates and calculates the
     * total sales amount per article.
     */
    public function getArticleSalesAction()
    {
        $this->Front()->Plugins()->Json()->setRenderer();

        // Extract 'from date' from the request
        $fromDate = $this->Request()->getParam('fromDate');
        if (empty($fromDate)) {
            $fromDate = new \DateTime();
            $fromDate = $fromDate->sub(new DateInterval('P1M'));
        } else {
            $fromDate = new \DateTime($fromDate);
        }
        $fromDate = $fromDate->format('Y-m-d H:i:s');

        // Extract 'to date' from the request
        $toDate = $this->Request()->getParam('toDate');
        if (empty($toDate)) {
            $toDate = new \DateTime();
        } else {
            $toDate = new \DateTime($toDate);
        }
        $toDate = $toDate->add(new DateInterval('P1D'))->sub(new DateInterval('PT1S'));
        $toDate = $toDate->format('Y-m-d H:i:s');

        // Extract paging 'start' and 'limit' from the request
        $start = $this->Request()->getParam('start');
        $limit = $this->Request()->getParam('limit');
        if ($start !== null && $limit !== null) {
            $limitSQL = 'LIMIT '.intval($start).', '.intval($limit);
        } else {
            $limitSQL = '';
        }

        /* @var $defaultShop Shop */
        $defaultShop = $this->get('models')->getRepository(Shop::class)->getDefault();

        // Fetch correct snippets for summary rows. Use the current backend users language
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_analytics/article');
        $rowSnippetSales = $namespace->get('row/sales_sum');
        $rowProfitWithShipping = $namespace->get('row/profit_with_shipping');
        $rowProfitWithoutShipping = $namespace->get('row/profit_without_shipping');
        $rowShipping = $namespace->get('row/shipping');
        $rowBundleDiscount = $namespace->get('row/bundle_discount');
        $rowDiscountOrVoucher = $namespace->get('row/discount_or_voucher');
        $rowMisc = $namespace->get('row/misc');
        $rowNonShopwareArticle = $namespace->get('row/non_shopware_articles');

        // Sanitize sorting (only one column is supported)
        $sortParam = $this->Request()->getParam('sort', []);
        $sort = array_pop($sortParam);
        if (ctype_alnum($sort['property']) && in_array($sort['direction'], ['ASC', 'DESC'])) {
            $orderSQL = 'ORDER BY '.$sort['property'].' '.$sort['direction'];
        } else {
            $orderSQL = '';
        }

        $defaultShopCategoryID = $defaultShop->getCategory()->getId();
        $categoryID = $this->Request()->getParam('selectedCategory');
        if (!is_numeric($categoryID)) {
            // Use root categories of all subshops
            $categoryIDs = Shopware()->Db()->fetchCol('SELECT category_id FROM s_core_shops');
        } else {
            $categoryIDs = [$categoryID];
        }
        // Join the selected category count per article as separate table and apply condition (categoryCount) to it.
        $categoryCountConditionJoinQuery = sprintf(
            'SELECT
                ac.articleID AS articleId,
                COUNT(*) AS categoryCount
            FROM s_viison_article_analytics_tmp_min_category_ro ac
            WHERE ac.categoryID IN (%s)
            GROUP BY ac.articleID',
            implode(',', $categoryIDs)
        );

        // Exclude reversal orders (POS returns) from main result query (not from total result, which calculates the sales sum)
        $mainResultWhereSQL = '';
        if (ViisonCommonUtil::isPluginInstalled('Core', 'ViisonPickwarePOS')) {
            $mainResultWhereSQL .= ' AND oa.pickware_pos_transaction_type != \'reversal_order\' ';
        }

        $selectedShops = trim($this->Request()->getParam('selectedShops'));
        $languageIds = explode(',', $selectedShops);
        // Validate
        if (!empty($selectedShops)) {
            foreach ($languageIds as $selectedShopId) {
                if (!is_numeric($selectedShopId)) {
                    throw new \Exception('Non-numeric shop identifiers are not allowed in selectedShops.');
                }
            }
        }
        $filterLanguageIds = !empty($selectedShops) && !empty($languageIds);
        if ($filterLanguageIds) {
            $languageIdsSelector = 'IN (' . implode(',', $languageIds) . ')';
        }

        $selectedCustomerGroups = trim($this->Request()->getParam('selectedCustomerGroups'));
        $customerGroupIds = explode(',', $selectedCustomerGroups);
        // Validate
        if (!empty($selectedCustomerGroups)) {
            foreach ($customerGroupIds as $customerGroupId) {
                if (!is_numeric($customerGroupId)) {
                    throw new \Exception('Non-numeric customer group identifiers are not allowed in selectedCustomerGroups.' . var_export($customerGroupId, true));
                }
            }
        }
        $filterCustomerGroups = !empty($selectedCustomerGroups) && !empty($customerGroupIds);
        if ($filterCustomerGroups) {
            /* Translate customer group IDs to customer group descriptions, as that is the way they are saved in
             * s_user.customergroup. */
            $customerGroupIdsSelector = 'IN (' . implode(',', $customerGroupIds) . ')';
            $customerGroupsGroupKeyRows = Shopware()->Db()->fetchAll(
                'SELECT
                    groupkey
                FROM
                    s_core_customergroups
                WHERE
                    id ' . $customerGroupIdsSelector
            );
            $customerGroupsGroupKeys = array_map(
                function ($customerGroupRow) {
                    return Shopware()->Db()->quote($customerGroupRow['groupkey']);
                },
                $customerGroupsGroupKeyRows
            );
            $customerGroupsGroupKeysSelector = 'IN (' . implode(',', $customerGroupsGroupKeys) . ')';
        }

        // We want the articles to only count for one category. To achieve this, we create a temporary helper table that only contains the first category (smallest id) of each article. Only categories belonging to the main subshop are considered here.
        Shopware()->Db()->query(
            'CREATE TEMPORARY TABLE s_viison_article_analytics_tmp_min_category (
                articleID int,
                categoryID int,
                UNIQUE INDEX (`articleID`),
                INDEX (`categoryID`)
            )
            SELECT
                ac.articleID,
                MIN(c.id) AS categoryID
            FROM
                s_articles_categories_ro ac,
                s_articles_categories_ro ac2,
                s_categories c
            WHERE
                ac.categoryID = ac.parentCategoryID AND
                ac.categoryID = ac2.parentCategoryID AND
                ac2.categoryID = ? AND
                c.id = ac.categoryID AND
                c.active=1
            GROUP BY ac.articleID',
            [
                $defaultShopCategoryID
            ]
        );

        Shopware()->Db()->query(
            'CREATE TEMPORARY TABLE s_viison_article_analytics_tmp_min_category_other_subshops (
                articleID int,
                categoryID int,
                INDEX (`articleID`),
                INDEX (`categoryID`)
            )
            SELECT
                ac.articleID,
                MIN(c.id) AS categoryID
            FROM
                s_articles_categories_ro ac,
                s_categories c
            WHERE
                ac.categoryID = ac.parentCategoryID AND
                c.id = ac.categoryID AND
                c.active=1
            GROUP BY ac.articleID'
        );

        Shopware()->Db()->query('
            INSERT IGNORE INTO s_viison_article_analytics_tmp_min_category
            SELECT * FROM s_viison_article_analytics_tmp_min_category_other_subshops
        ');

        // For articles that are only assigned to inactive categories, choose the first category of those (smallest id again).
        // We want to add the determined rows to the s_viison_article_analytics_tmp_min_category table. Unfortunately MySQL
        // does not allow to reference a temporary table two times, so we cannot use an INSERT SELECT query here..
        Shopware()->Db()->query(
            'CREATE TEMPORARY TABLE s_viison_article_analytics_tmp_min_category_inactive (
                articleID int,
                categoryID int,
                INDEX (`articleID`),
                INDEX (`categoryID`)
            )
            SELECT ac.articleID, MIN(c.id) AS categoryID
            FROM s_articles_categories_ro ac
            INNER JOIN s_articles_categories_ro ac2 ON ac.categoryID = ac2.parentCategoryID AND ac2.categoryID = ? AND ac.categoryID = ac.parentCategoryID
            INNER JOIN s_categories c ON c.id = ac.categoryID
            LEFT JOIN s_viison_article_analytics_tmp_min_category mc ON mc.articleID = ac.articleID
            WHERE
                mc.categoryID IS NULL
            GROUP BY ac.articleID',
            [
                $defaultShopCategoryID
            ]
        );

        Shopware()->Db()->query('
            INSERT INTO s_viison_article_analytics_tmp_min_category
            SELECT * FROM s_viison_article_analytics_tmp_min_category_inactive
        ');

        // Create a table containing all categories, including parent categories, an article is part of (only the "main category" determined earlier and its parents)
        Shopware()->Db()->query(
            'CREATE TEMPORARY TABLE s_viison_article_analytics_tmp_min_category_ro (
                articleID int,
                categoryID int,
                INDEX (`articleID`),
                INDEX (`categoryID`)
            )
            SELECT DISTINCT
                amc.articleID,
                ac.categoryID
            FROM
                s_viison_article_analytics_tmp_min_category amc,
                s_articles_categories_ro ac
            WHERE
                ac.parentCategoryID = amc.categoryID'
        );

        // Count articles without a category towards the default shop root category
        Shopware()->Db()->query(
            'INSERT INTO s_viison_article_analytics_tmp_min_category_ro
            SELECT a.id, ?
            FROM s_articles a
            LEFT JOIN s_articles_categories ac ON ac.articleID = a.id
            WHERE ac.categoryID IS NULL',
            [
                $defaultShopCategoryID
            ]
        );

        // Get total data
        $articleTotalRow = Shopware()->Db()->fetchRow(
            'SELECT SQL_CALC_FOUND_ROWS SUM(d.quantity) AS count, \'' . $rowSnippetSales . '\' as name, \'\' AS supplier, \'\' AS articleNumber, ROUND(SUM((d.quantity * d.price / o.currencyFactor)*IF(o.net, (1+IF(o.taxfree, 0, d.tax_rate)/100), 1)), 2) AS amount, ROUND(SUM((d.quantity * d.price / o.currencyFactor)/IF(o.net, 1, (1+IF(o.taxfree, 0, d.tax_rate)/100))), 2) AS amountNet, \'\' AS price, \'\' AS instock
            FROM s_order_details d
            INNER JOIN s_order o ON d.orderID = o.id
            '.($filterCustomerGroups ? 'INNER JOIN s_user u ON o.userID = u.id' : '').'
            LEFT OUTER JOIN s_articles a ON d.articleID = a.id
            LEFT OUTER JOIN s_articles_details ad ON d.articleordernumber = ad.ordernumber
            LEFT OUTER JOIN s_articles_supplier s ON a.supplierID = s.id
            LEFT OUTER JOIN (' . $categoryCountConditionJoinQuery . ') AS category_filter ON category_filter.articleId = a.id
            WHERE d.modus = 0
            AND d.articleID != 0
            AND o.ordertime >= ?
            AND o.ordertime <= ?
            AND o.status NOT IN (-1, 4)
            AND category_filter.categoryCount > 0
            '.($filterLanguageIds ? 'AND o.language ' . $languageIdsSelector : '').'
            '.($filterCustomerGroups ? 'AND u.customergroup ' . $customerGroupsGroupKeysSelector : '').'
            '.$orderSQL,
            [
                $fromDate,
                $toDate,
            ]
        );

        // Get main results
        $data = Shopware()->Db()->fetchAll(
            'SELECT SQL_CALC_FOUND_ROWS SUM(d.quantity) AS count, ad.id AS articleDetailId, d.articleID AS articleId, IF(ISNULL(a.name), d.name, CONCAT_WS(\' \', a.name, IF(ad.additionaltext = \'\', NULL, ad.additionaltext))) AS name, GROUP_CONCAT(DISTINCT s.name) as supplier, d.articleordernumber AS articleNumber, ROUND(SUM((d.quantity * d.price / o.currencyFactor)*IF(o.net, (1+IF(o.taxfree, 0, d.tax_rate)/100), 1)), 2) AS amount, ROUND(SUM((d.quantity * d.price / o.currencyFactor)/IF(o.net, 1, (1+IF(o.taxfree, 0, d.tax_rate)/100))), 2) as amountNet, ROUND(IF(ap.price, ap.price * ((100 + IF(t.tax, t.tax, 19)) / 100), d.price * IF(o.net, (1+IF(o.taxfree, 0, d.tax_rate)/100), 1)), 2) AS price, ad.instock as instock
            FROM s_order_details d
            INNER JOIN s_order o ON d.orderID = o.id
            INNER JOIN s_order_attributes oa ON oa.orderID = o.id
            '.($filterCustomerGroups ? 'INNER JOIN s_user u ON o.userID = u.id' : '').'
            LEFT OUTER JOIN s_articles a ON d.articleID = a.id
            LEFT OUTER JOIN s_articles_details ad ON d.articleordernumber = ad.ordernumber
            LEFT OUTER JOIN s_articles_supplier s ON a.supplierID = s.id
            LEFT OUTER JOIN s_articles_prices ap ON ap.articledetailsID = ad.id AND ap.pricegroup = \'EK\' AND ap.from = 1
            LEFT OUTER JOIN s_core_tax t ON a.taxID = t.id
            LEFT OUTER JOIN (' . $categoryCountConditionJoinQuery . ') AS category_filter ON category_filter.articleId = a.id
            WHERE d.modus = 0
            AND d.articleID != 0
            AND o.ordertime >= ?
            AND o.ordertime <= ?
            AND o.status NOT IN (-1, 4)
            AND category_filter.categoryCount > 0
            '.$mainResultWhereSQL.'
            '.($filterLanguageIds ? 'AND o.language ' . $languageIdsSelector : '').'
            '.($filterCustomerGroups ? 'AND u.customergroup ' . $customerGroupsGroupKeysSelector : '').'
            GROUP BY d.articleordernumber
            '.$orderSQL.'
            '.$limitSQL,
            [
                $fromDate,
                $toDate,
            ]
        );

        $total = intval(array_pop(Shopware()->Db()->fetchRow('SELECT FOUND_ROWS()')));

        $shopIds = Shopware()->Db()->fetchCol('SELECT id FROM s_core_shops ORDER BY id');

        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($data, 'articleDetailId'));
        foreach ($data as &$row) {
            // Convert values
            $row['count'] = (int)$row['count'];
            $row['amount'] = (float)$row['amount'];
            $row['amountNet'] = (float)$row['amountNet'];

            // Fix additional names
            $additionalText = $additionalTexts[$row['articleDetailId']];
            $row['name'] .= $additionalText ? ', ' . $additionalText : '';

            // Fetch amount shares of each shop
            $shopData = Shopware()->Db()->fetchAll(
                'SELECT o.subshopID AS shopId, ROUND(SUM((d.quantity * d.price / o.currencyFactor)*IF(o.net, (1+IF(o.taxfree, 0, d.tax_rate)/100), 1)), 2) AS amount, ROUND(SUM((d.quantity * d.price / o.currencyFactor)/IF(o.net, 1, (1+IF(o.taxfree, 0, d.tax_rate)/100))), 2) as amountNet
                 FROM s_order_details d
                 INNER JOIN s_order o ON d.orderID = o.id
                 '.($filterCustomerGroups ? 'INNER JOIN s_user u ON o.userID = u.id' : '').'
                 WHERE d.modus = 0
                 AND d.articleordernumber = ?
                 '.($filterLanguageIds ? 'AND o.language ' . $languageIdsSelector : '').'
                 '.($filterCustomerGroups ? 'AND u.customergroup ' . $customerGroupsGroupKeysSelector : '').'
                 AND o.ordertime >= ?
                 AND o.ordertime <= ?
                 AND o.status NOT IN (-1, 4)
                 GROUP BY o.subshopID',
                [
                    $row['articleNumber'],
                    $fromDate,
                    $toDate,
                ]
            );
            unset($row['articleId']);

            // Initialize turnover for individual shops with zero (the query above only contains shops with non-zero results)
            foreach ($shopIds as $shopId) {
                $row['amount' . $shopId] = 0;
                $row['amountNet' . $shopId] = 0;
            }

            // Store actual non-zero turnover values for individual shops
            foreach ($shopData as $shopResult) {
                $row['amount' . $shopResult['shopId']] = (float)$shopResult['amount'];
                $row['amountNet' . $shopResult['shopId']] = (float)$shopResult['amountNet'];
            }
        }
        // Add total row
        $data[] = $articleTotalRow;

        // Get other order positions
        $otherOrderPositions = Shopware()->Db()->fetchAll(
            'SELECT SQL_CALC_FOUND_ROWS SUM(d.quantity) AS count,
            CONCAT(IF(d.modus=10, \'' . $rowBundleDiscount . '\',
                IF(d.modus=2 OR d.price < 0, \'' . $rowDiscountOrVoucher . '\',
                    IF(d.modus NOT IN (0,2,10), \'' . $rowMisc . '\', \'' . $rowNonShopwareArticle . '\'))), \': \', d.name) AS name,
            \'\' AS supplier, d.articleordernumber AS articleNumber,
            ROUND(SUM((d.quantity * d.price / o.currencyFactor)*IF(o.net, (1+IF(o.taxfree, 0, d.tax_rate)/100), 1)), 2) AS amount, ROUND(SUM((d.quantity * d.price / o.currencyFactor)/IF(o.net, 1, (1+IF(o.taxfree, 0, d.tax_rate)/100))), 2) as amountNet, \'\' AS price, \'\' AS instock
            FROM s_order_details d
            INNER JOIN s_order o ON d.orderID = o.id
            '.($filterCustomerGroups ? 'INNER JOIN s_user u ON o.userID = u.id' : '').'
            LEFT JOIN s_articles a ON d.articleId = a.id
            WHERE NOT (d.modus = 0 AND a.id IS NOT NULL) /* exclude regular order positions */
            '.($filterLanguageIds ? 'AND o.language ' . $languageIdsSelector : '').'
            '.($filterCustomerGroups ? 'AND u.customergroup ' . $customerGroupsGroupKeysSelector : '').'
            AND o.ordertime >= ?
            AND o.ordertime <= ?
            AND o.status NOT IN (-1, 4)
            GROUP BY name, d.articleordernumber',
            [
                $fromDate,
                $toDate,
            ]
        );
        $data = array_merge($data, $otherOrderPositions);

        $totalRowWithoutShipping = self::getSumOfRows(array_merge(
            [$articleTotalRow],
            $otherOrderPositions
        ));
        $totalRowWithoutShipping['name'] = $rowProfitWithoutShipping;
        $data[] = $totalRowWithoutShipping;

        $shippingRow = Shopware()->Db()->fetchRow(
            'SELECT SUM(o.invoice_shipping > 0) AS count, \'' . $rowShipping . '\' AS name, \'\' AS supplier, \'\' AS articleNumber, ROUND(SUM(o.invoice_shipping), 2) AS amount, ROUND(SUM(o.invoice_shipping_net), 2) AS amountNet, \'\' AS price, \'\' AS instock
            FROM s_order o
            ' . ($filterCustomerGroups ? 'INNER JOIN s_user u ON o.userID = u.id' : '') . '
            WHERE o.ordertime >= ?
            AND o.ordertime <= ?
            AND o.status NOT IN (-1, 4)
            ' . ($filterLanguageIds ? 'AND o.language ' . $languageIdsSelector : '') . '
            ' . ($filterCustomerGroups ? 'AND u.customergroup ' . $customerGroupsGroupKeysSelector : ''),
            [
                $fromDate,
                $toDate,
            ]
        );
        $data[] = $shippingRow;

        $totalRowWithShipping = self::getSumOfRows([
            $totalRowWithoutShipping,
            $shippingRow,
        ]);
        $totalRowWithShipping['name'] = $rowProfitWithShipping;
        $data[] = $totalRowWithShipping;

        // Send results
        $this->send($data, $total);
    }

    private static function getSumOfRows($rows)
    {
        $totalRow = [];
        foreach ($rows[0] as $columnName => $value) {
            if (is_numeric($value)) {
                $totalRow[$columnName] = array_reduce(
                    $rows,
                    function ($sum, $row) use ($columnName) {
                        return $sum + $row[$columnName];
                    },
                    0
                );
            } else {
                $totalRow[$columnName] = '';
            }
        }

        return $totalRow;
    }
}
