<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class Shopware_Controllers_Backend_ViisonPickwareERPAnalyticsGrossProfit extends Shopware_Controllers_Backend_Analytics
{
    /**
     * Generates the gross profit analytics. Returns analytics per article and a summary.
     *
     * Remark: some of the validation checks are missing (e.g. is_numeric) since some parameters are given by the
     * backend UI and are bound to be valid IDs
     *
     * @throws Exception
     */
    public function getGrossProfitAction()
    {
        /**
         * Notify other plugins that the analytics is about to be performed
         * E.g. SetArticles needs to add a temporary table
         */
        $this->get('events')->notify(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_beforeDataAccess',
            [
                'fromDate' => $this->getFromDate(),
                'toDate' => $this->getToDate(),
                'selectedShopIds' => $this->getSelectedShopIds(),
                'selectedCustomerGroupIds' => $this->getSelectedCustomerGroupIds(),
                'purchasePriceMode' => $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->get('purchasePriceMode'),
            ]
        );

        /**
         * Build summary query builder. Use the basic article query and since we use SUM functions,
         * the rows are already grouped together to form a summary
         */
        $summaryBuilder = $this->getArticleCalculationsQueryBuilder();
        $summaryResult = $summaryBuilder->execute()->fetchAll(PDO::FETCH_ASSOC);

        /**
         * Build article query builder. Use the summary builder, except this time we group by article details
         * explicitly to create per-article calculations
         *
         * Remark: we use the same builder to avoid firing all events again, since it is the same query with
         * different group statements. (increases performance)
         */
        $perArticleBuilder = $this->getPerArticleQueryBuilder($summaryBuilder);

        /**
         * Send results
         */
        $this->sendResult($perArticleBuilder, $summaryResult);
    }

    /**
     * Finalizes the per-article results by adding article names.
     * Calls parent send() to send results.
     *
     * @param Doctrine\ORM\QueryBuilder $perArticleBuilder
     * @param array $summaryResult
     */
    protected function sendResult($perArticleBuilder, $summaryResult)
    {
        $articleResult = $perArticleBuilder->execute()->fetchAll(PDO::FETCH_ASSOC);
        $totalNumberOfArticleRows = $this->get('db')->fetchOne('SELECT FOUND_ROWS()');

        // Fix article names by using ViisonCommons additionaltext fetcher
        $articleDetailIds = array_column($articleResult, 'articleDetailId');
        $variantTexts = ViisonCommonUtil::getVariantAdditionalTexts($articleDetailIds);
        foreach ($articleResult as $key => &$article) {
            $additionalText = $variantTexts[$article['articleDetailId']];
            if ($additionalText) {
                $article['name'] .= ', ' . $additionalText;
            }
        }

        // Send to Shopwares analytics controller
        parent::send($articleResult, $totalNumberOfArticleRows);

        /**
         * Remark: since the article calculation sum still contains some single-article properties (e.g. article name)
         * we have to cherry pick the summary information
         */
        $this->View()->assign([
            'summary' => [
                'gross_profit' => $summaryResult[0]['gross_profit'],
                'gross_profit_percentage' => $summaryResult[0]['gross_profit_percentage'],
                'purchase_value' => $summaryResult[0]['purchase_value'],
                'purchase_value_net' => $summaryResult[0]['purchase_value_net'],
                'sales_value' => $summaryResult[0]['sales_value'],
                'sales_value_net' => $summaryResult[0]['sales_value_net'],
            ],
        ]);
    }

    /**
     * Creates a querybuilder that fetches the basic "per article calculations" that are needed for the
     * per-article-results as well as the summary.
     * It calculates the purchase and sales sums as well as the gross profit for each sold article
     *
     * @return Doctrine\ORM\QueryBuilder
     */
    private function getArticleCalculationsQueryBuilder()
    {
        $purchasePriceMode = $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->get('purchasePriceMode');
        $builder = $this->get('dbal_connection')->createQueryBuilder();

        /**
         * Remark Article(Detail) join with article-ordernumber:
         * It would have been much more robust if we'd join article(details) with PickwareStock.articleDetailId
         * since the ordernumber can be changed in the backend. But SetArticles don't have Stocks, so any article
         * information (articles, articledetails, articleattributes, tax) would have to be joined separately and every
         * select and calculation on these information has to be done separately as well. To ease this compatibility
         * issue, we join with orderDetail.articleordernumber which can be done with SetArticles.
         */
        $builder
            ->select(
                'SQL_CALC_FOUND_ROWS orderDetail.articleordernumber AS article_number',
                'article.name AS name',
                'article.id AS articleId',
                'articleDetail.id AS articleDetailId'
            )
            ->from('s_order_details', 'orderDetail')
            ->leftJoin('orderDetail', 'pickware_erp_stock_ledger_entries', 'PickwareStock', 'orderDetail.id = PickwareStock.orderDetailId')
            ->leftJoin('orderDetail', 's_articles_details', 'articleDetail', 'orderDetail.articleordernumber = articleDetail.ordernumber')
            ->leftJoin('orderDetail', 's_order', 'shopOrder', 'shopOrder.id = orderDetail.orderID')
            ->leftJoin('articleDetail', 's_articles', 'article', 'articleDetail.articleID = article.id')
            ->leftJoin('shopOrder', 's_user', 'customer', 'customer.id = shopOrder.userID')
            ->leftJoin('customer', 's_core_customergroups', 'customerGroup', 'customerGroup.groupkey = customer.customergroup')
            ->leftJoin('orderDetail', 's_order_details_attributes', 'orderDetailAttribute', 'orderDetailAttribute.detailID = orderDetail.id')
            ->leftJoin('articleDetail', 's_articles_attributes', 'articleAttribute', 'articleAttribute.articledetailsID = articleDetail.id')
            ->leftJoin('article', 's_core_tax', 'articleTax', 'articleTax.id = article.taxID');

        /**
         * Throw event to let other plugins add additional information to the Joins
         * (e.g. SetArticles plugin needs to add SubArticles to the query)
         * Remark: join() can take 5 parameters. So make sure other plugins format the additional Join accordingly
         */
        $additionalJoins = [];
        $additionalJoins = $this->get('events')->filter(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_AdditionalJoins',
            $additionalJoins
        );
        foreach ($additionalJoins as $join) {
            if (count($join) < 3) {
                continue;
            }
            if (count($join) < 4) {
                $join[3] = null;
            }
            $builder->leftJoin($join[0], $join[1], $join[2], $join[3]);
        }

        /**
         * Add general Stock restrictions.
         * But: throw event before each condition, because other plugins might need to manipulate it
         * (e.g. SetArticles plugin needs to not-remove SetArticles, even though they have no PickwareStock entries
         */
        $stockTypeFilter = $builder->expr()->orX(
            $builder->expr()->eq(
                'PickwareStock.type',
                ':typeSale'
            ),
            $builder->expr()->eq(
                'PickwareStock.type',
                ':typeReturn'
            )
        );
        $stockTypeFilter = $this->get('events')->filter(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_AddStockTypeFilter',
            $stockTypeFilter
        );
        $builder
            ->where($stockTypeFilter)
            ->setParameters([
                'typeSale' => StockLedgerEntry::TYPE_SALE,
                'typeReturn' => StockLedgerEntry::TYPE_RETURN,
            ]);

        // Add date restrictions the same way
        if ($this->getFromDate()) {
            $fromDateFilter = $builder->expr()->gt(
                'PickwareStock.created',
                ':fromDate'
            );
            $fromDateFilter = $this->get('events')->filter(
                'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_AddFromDateFilter',
                $fromDateFilter
            );
            $builder
                ->andWhere($fromDateFilter)
                ->setParameter('fromDate', $this->getFromDate()->format('Y-m-d H:i:s'));
        }
        if ($this->getToDate()) {
            $toDateFilter = $builder->expr()->lt(
                'PickwareStock.created',
                ':toDate'
            );
            $toDateFilter = $this->get('events')->filter(
                'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_AddToDateFilter',
                $toDateFilter
            );
            $builder
                ->andWhere($toDateFilter)
                ->setParameter('toDate', $this->getToDate()->format('Y-m-d H:i:s'));
        }

        // Add shop restriction
        if ($this->getSelectedShopIds()) {
            $builder
                ->andWhere('shopOrder.language IN (:languageIds)')
                ->setParameter('languageIds', $this->getSelectedShopIds(), \Doctrine\DBAL\Connection::PARAM_INT_ARRAY);
        }

        // Add customer group restriction
        if ($this->getSelectedCustomerGroupIds()) {
            $builder
                ->andWhere('customerGroup.id IN (:customerGroupIds)')
                ->setParameter(
                    'customerGroupIds',
                    $this->getSelectedCustomerGroupIds(),
                    \Doctrine\DBAL\Connection::PARAM_INT_ARRAY
                );
        }

        /**
         * Throw event to let other plugins filter articles as well. (e.g. SetArticles filter its SubArticles)
         * Filters must be formatted to be directly used in $builder->andWhere($filter)
         */
        $additionalFilters = [];
        $additionalFilters = $this->get('events')->filter(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_AdditionalFilter',
            $additionalFilters
        );
        foreach ($additionalFilters as $filter) {
            $builder->andWhere($filter);
        }

        /**
         * Sales calculations
         *
         * Remark ABS(price): since POS returns and Mobile returns have different price orientations (Mobile returns
         * refer to a regular order with positive price, POS returns refer a new return-order with a negative price), we
         * use the absolute value of the order price and choose the orientation by -/+ changeAmount.
         *
         * Remark "-1 * changeAmount": since we consider sales (or returns) the changeAmount for a sale would
         * be negative. Invert the amount to get a positive quantity for sales and purchase values.
         * (SalesAmount = (-1) * stockChange)
         *
         * Throw event to let other plugins manipulate sold quantity selection.
         */
        $selectSalesQuantity = '-1 * PickwareStock.changeAmount';
        $selectSalesQuantity = $this->get('events')->filter(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_SalesQuantitySelection',
            $selectSalesQuantity
        );

        $builder->addSelect('ROUND(SUM(
            ABS(orderDetail.price)
                * (' . $selectSalesQuantity . ')
                * (1 + CASE WHEN (shopOrder.net = 1)
                    THEN CASE WHEN (shopOrder.taxFree = 1)
                        THEN 0
                        ELSE orderDetail.tax_rate / 100
                        END
                    ELSE 0
                    END
                )
                / shopOrder.currencyFactor
           ), 2) AS sales_value');
        $builder->addSelect('SUM(' . $selectSalesQuantity . ') AS sales_quantity');

        $selectSalesValueNet = 'SUM(CASE WHEN orderDetail.id <> 0
            THEN
                ABS(orderDetail.price)
                * (' . $selectSalesQuantity . ')
                / (1 + CASE WHEN (shopOrder.net = 1)
                    THEN 0
                    ELSE (orderDetail.tax_rate / 100)
                    END
                )
                / shopOrder.currencyFactor
            ELSE 0
            END)';
        $builder->addSelect('ROUND(' . $selectSalesValueNet . ', 2) AS sales_value_net');

        /**
         * Purchase calculations
         *
         * Remark TAX RATE: we have no easy way to determine at which tax rate the article was actually
         * purchased. So we use orderDetail->Article->Tax->taxRate (not orderDetails taxRate directly) instead.
         * If the article (and therefore the Tax relation) was deleted, PickwareStocks should've been deleted as well
         * so no fallback value is needed.
         *
         * Remark "-1 * changeAmount": see sales quantity above
         *
         * Throw event to bypass purchase price selection and calculation completely.
         */
        if ($purchasePriceMode === 'net') {
            $selectPurchaseValueGross = 'PickwareStock.purchasePrice * (1 + articleTax.tax / 100)';
            $selectPurchaseValueNet = 'PickwareStock.purchasePrice';
        } else {
            $selectPurchaseValueGross = 'PickwareStock.purchasePrice';
            $selectPurchaseValueNet = 'PickwareStock.purchasePrice / (1 + articleTax.tax / 100)';
        }
        $selectPurchaseQuantity = '-1 * PickwareStock.changeAmount';

        $selectPurchaseValueGross = '(' . $selectPurchaseQuantity . ') * (' . $selectPurchaseValueGross . ')';
        $selectPurchaseValueGross = $this->get('events')->filter(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_PurchaseValueGrossSelection',
            $selectPurchaseValueGross
        );
        $selectPurchaseValueGross = 'SUM(' . $selectPurchaseValueGross . ')';
        $builder->addSelect('ROUND(' . $selectPurchaseValueGross . ', 2) AS purchase_value');

        $selectPurchaseValueNet = '(' . $selectPurchaseQuantity . ') * (' . $selectPurchaseValueNet . ')';
        $selectPurchaseValueNet = $this->get('events')->filter(
            'Shopware_Plugins_ViisonPickwareERP_AnalyticsGrossProfit_PurchaseValueNetSelection',
            $selectPurchaseValueNet
        );
        $selectPurchaseValueNet = 'SUM(' . $selectPurchaseValueNet . ')';
        $builder->addSelect('ROUND(' . $selectPurchaseValueNet . ', 2) AS purchase_value_net');

        /**
         * Profit calculations
         */
        // Gross profit = sales_net - purchases_net
        $selectGrossProfit = '(' . $selectSalesValueNet . ' - ' . $selectPurchaseValueNet . ')';
        $builder->addSelect('ROUND(' . $selectGrossProfit . ', 2) AS gross_profit');

        // Gross profit percentage = gross_profit / sales_value_net
        $builder->addSelect(
            'ROUND(CASE WHEN ' . $selectSalesValueNet . ' <> 0
                THEN (' . $selectGrossProfit . ' / ' . $selectSalesValueNet . ')
                ELSE 0
            END, 4) AS gross_profit_percentage'
        );

        /**
         * Remove "empty" article lines that have no actual sales values.
         * E.g. if you sell 1 article and it is returned, all sales/purchase/profit values are 0.
         */
        $builder->andHaving('(sales_value_net <> 0 OR purchase_value <> 0)');

        return $builder;
    }

    /**
     * Returns the modified (grouped and limited) querybuilder for per-article calculations.
     *
     * @param Doctrine\ORM\QueryBuilder $articleBuilder
     * @return Doctrine\ORM\QueryBuilder
     */
    private function getPerArticleQueryBuilder($articleBuilder)
    {
        $articleBuilder->addGroupBy('orderDetail.articleordernumber');
        foreach ($this->getSorting() as $oneOrderBy) {
            $articleBuilder->addOrderBy($oneOrderBy['property'], $oneOrderBy['direction']);
        }
        if ($this->getStart()) {
            $articleBuilder->setFirstResult($this->getStart());
        }
        if ($this->getLimit()) {
            $articleBuilder->setMaxResults($this->getLimit());
        }

        return $articleBuilder;
    }

    /**
     * Returns "limit" parameter from the request.
     * Use default null if no param was provided
     *
     * @return int
     */
    private function getStart()
    {
        return $this->Request()->getParam('start');
    }

    /**
     * Returns "limit" parameter from the request.
     * Set limit default to null to ensure CSV export contains all entries if no limit was given.
     *
     * @return null|int
     */
    private function getLimit()
    {
        return $this->Request()->getParam('limit');
    }

    /**
     * Returns "fromDate" parameter from the request.
     * Create from-DateTime with time 00:00:00, or leave empty if no param was given.
     *
     * @return DateTime|null
     */
    private function getFromDate()
    {
        $fromDateParam = $this->Request()->getParam('fromDate');

        return (empty($fromDateParam) ? null : new \DateTime($fromDateParam));
    }

    /**
     * Returns "toDate" parameter from the request.
     * Create to-DateTime with time 23:59:59, or leave empty if no param was given.
     *
     * @return DateTime|null
     */
    private function getToDate()
    {
        $toDate = $this->Request()->getParam('toDate', null);
        if ($toDate) {
            $toDate = new \DateTime($toDate);
            $toDate = $toDate->add(new DateInterval('P1D'));
            $toDate = $toDate->sub(new DateInterval('PT1S'));
        }

        return $toDate;
    }

    /**
     * Returns "selectedShops" parameter from the request as an array of IDs or null if no param was given.
     *
     * @return array|null
     */
    private function getSelectedShopIds()
    {
        $selectedShopsParam = trim($this->Request()->getParam('selectedShops', ''));

        return ($selectedShopsParam !== '') ? explode(',', $selectedShopsParam) : null;
    }

    /**
     * Returns "selectedShops" parameter from the request as an array of groupkeys or null if no param was given.
     *
     * @return array|null
     */
    private function getSelectedCustomerGroupIds()
    {
        $selectedCustomerGroupParam = trim($this->Request()->getParam('selectedCustomerGroups', ''));

        return ($selectedCustomerGroupParam !== '') ? explode(',', $selectedCustomerGroupParam) : null;
    }

    /**
     * Returns "sort" parameter from the request or default order if no param was given.
     *
     * @return array|mixed
     */
    private function getSorting()
    {
        $orderBy = $this->Request()->getParam('sort');
        // Set default orderBy if necessary
        if (!$orderBy) {
            $orderBy = [
                [
                    'property' => 'gross_profit_percentage',
                    'direction' => 'DESC',
                ],
                [
                    'property' => 'gross_profit',
                    'direction' => 'DESC',
                ],
            ];
        }

        return $orderBy;
    }
}
