<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\DBAL\Query\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\Plugins\ViisonCommon\Classes\Controllers\Backend\AbstractAnalyticsController;

class Shopware_Controllers_Backend_ViisonPickwareERPAnalyticsReturns extends AbstractAnalyticsController
{
    /**
     * Main action to get return report.
     */
    public function getReturnReportAction()
    {
        // Get query builder for the selected page of the store (limited result)
        $pageResultQueryBuilder = $this->getPaginatedQueryBuilder();
        $pageResults = $pageResultQueryBuilder->execute()->fetchAll(PDO::FETCH_ASSOC);
        $totalNumberOfRows = $this->get('db')->fetchOne('SELECT FOUND_ROWS()');

        // Fix ratio bug (see function doc below)
        foreach ($pageResults as &$pageResult) {
            $pageResult = $this->postProcessReportItem($pageResult);
        }
        unset($pageResult);

        // Send results
        $this->send(
            $pageResults,
            $totalNumberOfRows
        );

        // Get query builder for full data to calculate summary (unlimited result)
        $summaryQueryBuilder = $this->getReturnQueryBuilder();
        $summaryReturns = $summaryQueryBuilder->execute()->fetchAll(PDO::FETCH_ASSOC);
        $summary = $this->getSummaryFromReturns($summaryReturns);
        $this->View()->assign([
            'summary' => $this->postProcessReportItem($summary),
        ]);
    }

    /**
     * The shop filter is not working for written off returns (see shop filter clause in getReturnQueryBuilder)
     *
     * To work around the false results when the filter is active, we NULL the ratios to display them as "-" in the
     * Analytics view. A column tooltip will provide this information for the backend user.
     *
     * @param array $result
     * @return array
     */
    private function postProcessReportItem(array $result)
    {
        if ($this->getSelectedShopIds()) {
            $result['writtenOffRatio'] = null;
            $result['stockedRatio'] = null;
        }

        return $result;
    }

    /**
     * Fetches the basic returnQueryBuilder and adds sorting and limits for the paginated store rows.
     *
     * @return QueryBuilder
     */
    private function getPaginatedQueryBuilder()
    {
        $builder = $this->getReturnQueryBuilder();

        // Add sorting
        foreach ($this->getSorting() as $sorting) {
            $builder->addOrderBy($sorting['property'], $sorting['direction']);
        }

        // Add limits
        if ($this->getStart()) {
            $builder->setFirstResult($this->getStart());
        }
        if ($this->getLimit()) {
            $builder->setMaxResults($this->getLimit());
        }

        return $builder;
    }

    /**
     * Creates a temporary table with shipped and returned quantities,
     * builds a querybuilder on this table and returns it.
     *
     * @return QueryBuilder
     */
    private function getReturnQueryBuilder()
    {
        /**
         * Since we need to calculate and select ratios for sorting,
         * we need to create a temporary table and build our querybuilder on that table.
         *
         * Apply some filters on this table, since we start the querybuilder on this table we do not have to apply
         * them again afterwards.
         */
        $selectShipped = '
            -1 * SUM(
                CASE WHEN stockLedgerEntry.type = :saleType
                    THEN stockLedgerEntry.changeAmount
                    ELSE 0
                END
            )
        ';
        $selectReturned = '
            SUM(
                CASE WHEN stockLedgerEntry.type = :returnType
                    THEN stockLedgerEntry.changeAmount
                    ELSE 0
                END
            )
        ';
        $selectWrittenOff = '
            -1 * SUM(
                CASE WHEN stockLedgerEntry.type = :writeOffType
                    THEN stockLedgerEntry.changeAmount
                    ELSE 0
                END
            )
        ';

        $query = '
            CREATE TEMPORARY TABLE IF NOT EXISTS pickware_erp_analytics_returns_temp (
                articleDetailId INT,
                shipped INT,
                returned INT,
                stocked INT,
                writtenOff INT,
                PRIMARY KEY (articleDetailId)
            )
            SELECT
                articleDetail.id AS articleDetailId,
                ' . $selectShipped . ' AS shipped,
                ' . $selectReturned . ' AS returned,
                ' . $selectReturned . ' - ' . $selectWrittenOff . ' AS stocked,
                ' . $selectWrittenOff . ' AS writtenOff
            FROM pickware_erp_stock_ledger_entries stockLedgerEntry
            LEFT JOIN s_order_details orderDetail
                ON orderDetail.id = stockLedgerEntry.orderDetailId
            LEFT JOIN s_order shopOrder
                ON shopOrder.id = orderDetail.orderId
            LEFT JOIN s_articles_details articleDetail
                ON articleDetail.id = stockLedgerEntry.articleDetailId
            WHERE (stockLedgerEntry.type IN (:saleType, :returnType, :writeOffType))';

        // Set filter
        $sqlParameters = [
            'saleType' => StockLedgerEntry::TYPE_SALE,
            'returnType' => StockLedgerEntry::TYPE_RETURN,
            'writeOffType' => StockLedgerEntry::TYPE_WRITE_OFF,
        ];
        if ($this->getFromDate()) {
            $query .= ' AND stockLedgerEntry.created >= :fromDate';
            $sqlParameters['fromDate'] = $this->getFromDate()->format('Y-m-d H:i:s');
        }
        if ($this->getToDate()) {
            $query .= ' AND stockLedgerEntry.created <= :toDate';
            $sqlParameters['toDate'] = $this->getToDate()->format('Y-m-d H:i:s');
        }
        /**
         * // TODO: investigate this comment
         * Remark shop filter on written off-returns:
         * Since written off-returns are managed by a regular "manual" entry, it is not linked to an OrderDetail or
         * Order. To successfully filter written off-returns we'd need to join the corresponding "return" stock entry
         * via the transactionID string. These joins are expensive and take up too much time.
         * That's why we tolerate the non-working shop filter for written off-returns for now until its handling
         * was improved. See: https://github.com/VIISON/ShopwarePickwareERP/issues/595
         */
        if ($this->getSelectedShopIds()) {
            $query .= ' AND shopOrder.subshopId IN (' . implode(', ', $this->getSelectedShopIds()) . ')';
        }

        // Add group-by and execute SQL to create temporary table
        $query .= ' GROUP BY articleDetail.id';
        $this->get('db')->query($query, $sqlParameters);

        // Build querybuilder on temporary table
        /** @var QueryBuilder $builder */
        $builder = $this->get('dbal_connection')->createQueryBuilder();
        $builder
            ->select(
                'SQL_CALC_FOUND_ROWS Article.id as articleId',
                'Article.name as articleName',
                'ArticleDetail.id as articleDetailId',
                'ArticleDetail.ordernumber as articleNumber',
                'AnalyticsReturns.shipped',
                'AnalyticsReturns.returned',
                'AnalyticsReturns.stocked',
                'AnalyticsReturns.writtenOff',
                'ROUND(AnalyticsReturns.writtenOff / AnalyticsReturns.returned , 4) as writtenOffRatio',
                'ROUND(AnalyticsReturns.stocked / AnalyticsReturns.returned, 4) as stockedRatio',
                'ROUND(AnalyticsReturns.returned / AnalyticsReturns.shipped, 4) as returnedRatio'
            )
            ->from('pickware_erp_analytics_returns_temp', 'AnalyticsReturns')
            ->leftJoin('AnalyticsReturns', 's_articles_details', 'ArticleDetail', 'AnalyticsReturns.articleDetailId = ArticleDetail.id')
            ->leftJoin('ArticleDetail', 's_articles', 'Article', 'ArticleDetail.articleId = Article.id');

        return $builder;
    }

    /**
     * Calculates and returns a summary row for given returns.
     *
     * @param array $returns
     * @return array
     */
    private function getSummaryFromReturns($returns)
    {
        $shippedSum = array_sum(array_column($returns, 'shipped'));
        $returnedSum = array_sum(array_column($returns, 'returned'));

        return [
            'shipped' => $shippedSum,
            'returned' => $returnedSum,
            'stockedRatio' => ($returnedSum === 0) ? 0 : round(array_sum(array_column($returns, 'stocked')) / $returnedSum, 4),
            'writtenOffRatio' => ($returnedSum === 0) ? 0 : round(array_sum(array_column($returns, 'writtenOff')) / $returnedSum, 4),
            'returnedRatio' => ($shippedSum === 0) ? 0 : round($returnedSum / $shippedSum, 4),
        ];
    }

    /**
     * Returns "sort" parameter from the request or default order if no param was given.
     *
     * @return array
     */
    protected function getDefaultSorting()
    {
        $orderBy = $this->Request()->getParam('sort');
        if (!$orderBy) {
            $orderBy = [
                [
                    'property' => 'shipped',
                    'direction' => 'DESC',
                ],
            ];
        }

        return $orderBy;
    }
}
