<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class Shopware_Controllers_Backend_ViisonPickwareERPAnalyticsSalesRatio extends Shopware_Controllers_Backend_Analytics
{

    private function getDefaultSalesRatioSortParam()
    {
        return [
            [
                'property' => 'sales_ratio',
                'direction' => 'DESC',
            ],
        ];
    }

    private function createOrderBy($sortParam, array $sortableProperties, Enlight_Components_Db_Adapter_Pdo_Mysql $db)
    {
        if (empty($sortParam)) {
            $sortParam = $this->getDefaultSalesRatioSortParam();
        }

        if (empty($sortParam)) { // default may be empty, too.
            return '';
        }

        if (!is_array($sortParam)) {
            throw new \Exception('sort parameter must be an array.');
        }

        return 'ORDER BY '
            . implode(
                ', ',
                array_map(
                    function ($sortCriterion) use ($sortableProperties, $db) {
                        if (!is_array($sortCriterion)) {
                            throw new \Exception('Each sort criterion in the sort parameter must be an array.');
                        }
                        if (!isset($sortCriterion['property'])) {
                            throw new \Exception('Each sort criterion must have a \'property\' key.');
                        }
                        if (!isset($sortCriterion['direction'])) {
                            throw new \Exception('Each sort criterion must have a \'direction\' key.');
                        }
                        if (!in_array($sortCriterion['property'], $sortableProperties, true)) {
                            throw new \Exception('Sortable properties are restricted to '
                                . implode(', ', $sortableProperties));
                        }
                        if ($sortCriterion['direction'] !== 'ASC' && $sortCriterion['direction'] !== 'DESC') {
                            throw new \Exception('sort direction must be either ASC or DESC.');
                        }

                        return $db->quoteIdentifier($sortCriterion['property']) . ' ' . $sortCriterion['direction'];
                    },
                    $sortParam
                )
            );
    }

    public function getSalesRatioAction()
    {
        $start = $this->Request()->getParam('start', null);
        $limit = $this->Request()->getParam('limit', null);

        if ($start !== null && preg_match('/^\\d+$/', $start) !== 1) {
            throw new \Exception('start parameter must be an integer.');
        }
        if ($limit !== null && preg_match('/^\\d+$/', $limit) !== 1) {
            throw new \Exception('limit parameter must be an integer.');
        }

        $fromDateParam = $this->Request()->getParam('fromDate', 0);
        $toDateParam = $this->Request()->getParam('toDate', 0);

        /* No special time zone handling. Assumes the parameter's time zone is server time zone (because the parameter
         * does not contain any for requests generated by the Shopware analytics UI). */
        $fromDate = (empty($fromDateParam) ? null : new \DateTime($fromDateParam));

        if (empty($toDateParam)) {
            $toDate = null;
        } else {
            $toDate = new \DateTime($toDateParam);
            // next day == end of day
            $toDate = $toDate->add(new DateInterval('P1D'));
            $toDate = $toDate->sub(new DateInterval('PT1S'));
        }

        $sortParam = $this->Request()->getParam('sort', null);
        if ($sortParam === null) {
            $sortParam = $this->getDefaultSalesRatioSortParam();
        }

        if (!is_array($sortParam) || !is_array($sortParam[0]) || !isset($sortParam[0]['property']) || !isset($sortParam[0]['direction'])) {
            throw new \Exception('sort must be an array with a \'property\' and a \'direction\' key.');
        }

        $sortableProperties = [
            'sales',
            'name',
            'sales_ratio',
            'stocked',
        ];

        // Also validates $sortParam, or throws an exception:
        $orderBy = $this->createOrderBy($sortParam, $sortableProperties, Shopware()->Db());

        $sortsBySalesRatio = false;
        foreach ($sortParam as $sortCriterion) {
            if ($sortCriterion['property'] === 'sales_ratio') {
                $sortsBySalesRatio = true;
            }
        }

        $selectedShops = trim($this->Request()->getParam('selectedShops'));
        $languageIds = explode(',', $selectedShops);
        // Validate
        if (!empty($selectedShops)) {
            foreach ($languageIds as $selectedShopId) {
                if (!is_numeric($selectedShopId)) {
                    throw new \Exception('Non-numeric shop identifiers are not allowed in selectedShops.');
                }
            }
        }
        $filterLanguageIds = !empty($selectedShops) && !empty($languageIds);
        if ($filterLanguageIds) {
            $languageIdsWhereClause = 's_order.language IN (' . implode(',', $languageIds) . ')';
        }

        $selectedCustomerGroups = trim($this->Request()->getParam('selectedCustomerGroups'));
        $customerGroupIds = explode(',', $selectedCustomerGroups);
        // Validate
        if (!empty($selectedCustomerGroups)) {
            foreach ($customerGroupIds as $customerGroupId) {
                if (!is_numeric($customerGroupId)) {
                    throw new \Exception('Non-numeric customer group identifiers are not allowed in selectedCustomerGroups.');
                }
            }
        }
        $filterCustomerGroups = !empty($selectedCustomerGroups) && !empty($customerGroupIds);
        if ($filterCustomerGroups) {
            /* Translate customer group IDs to customer group descriptions, as that is the way they are saved in
             * s_user.customergroup. */
            $customerGroupIdsSelector = 'IN (' . implode(',', $customerGroupIds) . ')';
            $customerGroupsGroupKeyRows = Shopware()->Db()->fetchAll(
                'SELECT
                    groupkey
                FROM
                    s_core_customergroups
                WHERE
                    id ' . $customerGroupIdsSelector
            );
            $customerGroupsGroupKeys = array_map(
                function ($customerGroupRow) {
                    return Shopware()->Db()->quote($customerGroupRow['groupkey']);
                },
                $customerGroupsGroupKeyRows
            );
            $customerGroupsGroupKeysSelector = 'IN (' . implode(',', $customerGroupsGroupKeys) . ')';
        }

        // TODO: other filters?
        // TODO: Support categories filter --> see ArticleAnalytics
        // TODO: Support suppliers filter
        // TODO: Handle discounts
        // TODO: Handle returns

        $sql = '
            SELECT
                ' . (!$sortsBySalesRatio ? 'SQL_CALC_FOUND_ROWS' : '') . '

                s_articles.name AS name,
                s_articles.id as articleId,
                s_articles_details.ordernumber AS article_number,

                `pickware_erp_stock_ledger_entries`.articleDetailId AS articleDetailId,
                SUM(CASE
                    WHEN `pickware_erp_stock_ledger_entries`.type = :saleType
                    THEN -1 * `pickware_erp_stock_ledger_entries`.changeAmount
                    ELSE 0
                    END) AS sales,
                SUM(CASE
                    WHEN `pickware_erp_stock_ledger_entries`.type <> :saleType
                    THEN `pickware_erp_stock_ledger_entries`.changeAmount
                    ELSE 0
                    END) AS stocked

            FROM
                `pickware_erp_stock_ledger_entries`
            INNER JOIN s_articles_details
                ON s_articles_details.id = `pickware_erp_stock_ledger_entries`.articleDetailId
            INNER JOIN s_articles
                ON s_articles.id = s_articles_details.articleID
            LEFT JOIN s_order_details
                ON s_order_details.id = `pickware_erp_stock_ledger_entries`.orderDetailId
            LEFT JOIN s_order
                ON s_order.id = s_order_details.orderID
            '.($filterCustomerGroups ? 'LEFT JOIN s_user ON s_order.userID = s_user.id' : '').'
            WHERE
                1=1
                '.($filterLanguageIds ? 'AND ' . $languageIdsWhereClause : '').'
                '.($filterCustomerGroups ? 'AND s_user.customergroup ' . $customerGroupsGroupKeysSelector : '').'
                '.($fromDate !== null ? 'AND `pickware_erp_stock_ledger_entries`.created > :fromDate' : '').'
                '.($toDate !== null ? 'AND `pickware_erp_stock_ledger_entries`.created < :toDate' : '').'
            GROUP BY
                `pickware_erp_stock_ledger_entries`.articleDetailId
            HAVING
                sales <> 0 OR stocked <> 0
            ' . (!$sortsBySalesRatio ? $orderBy : '');

        if ($sortsBySalesRatio) {
            /* We need to sort by a derived value which we could not compute in the preceding query statement.
             * Hence, we wrap that preceding query as a sub query, compute sales_ratio, and sort by sales_ratio. */
            $sql = 'SELECT
                        SQL_CALC_FOUND_ROWS
                        s.name AS name,
                        s.articleId,
                        s.articleDetailId,
                        s.article_number AS article_number,
                        s.sales AS sales,
                        s.stocked AS stocked,
                        IF(stocked <> 0, (sales / stocked), False) AS sales_ratio
                    FROM (' . $sql . ') s
                    ' . $orderBy;
        }

        if ($start !== null && $limit !== null) {
            $sql .= '
                LIMIT ' . intval($start) . ', ' . intval($limit);
        }

        $sqlParams = [
            'saleType' => StockLedgerEntry::TYPE_SALE,
        ];
        if ($fromDate !== null) {
            $sqlParams['fromDate'] = $fromDate->format('Y-m-d H:i:s');
        }
        if ($toDate !== null) {
            $sqlParams['toDate'] = $toDate->format('Y-m-d H:i:s');
        }

        $result = Shopware()->Db()->fetchAll(
            $sql,
            $sqlParams
        );
        $total = Shopware()->Db()->fetchOne('SELECT FOUND_ROWS() as count');

        // Fix additional names
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($result, 'articleDetailId'));
        foreach ($result as &$row) {
            $additionalText = $additionalTexts[$row['articleDetailId']];
            $row['name'] .= $additionalText ? ', ' . $additionalText : '';
        }

        // If we did not wrap the query (given $sortsBySalesRatio === true), we need to manually calculate the
        // sales_ratio.
        foreach ($result as &$row) {
            if (isset($row['sales_ratio'])) {
                break;
            }
            if ($row['stocked'] != 0) { // No strict comparison. May get a numerical string as result.
                $row['sales_ratio'] = $row['sales'] / $row['stocked'];
            } else {
                $row['sales_ratio'] = false;
            }
        }

        $this->send($result, $total);
    }
}
