<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\DBAL\Connection;
use Shopware\Components\Model\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class Shopware_Controllers_Backend_ViisonPickwareERPAnalyticsStockMovements extends Shopware_Controllers_Backend_Analytics
{
    /**
     * Main action that assigns stock movement report to the view.
     */
    public function getStockMovementsAction()
    {
        $this->send($this->getStockMovementsResult(), $this->getStockMovementsEntriesTotalCount());
    }

    /**
     * Returns the number of stock movements rows that are grouped by article details by using the basic stock movements
     * query builder without limitations.
     *
     * @return int
     */
    private function getStockMovementsEntriesTotalCount()
    {
        $builder = $this->getStockMovementsQueryBuilder()->select('COUNT(DISTINCT articleDetail.id)');

        return $builder->getQuery()->getSingleScalarResult();
    }

    /**
     * Returns the actual stock movements rows by using the basic stock movements with all necessary selects and
     * limitations due to pagination.
     *
     * @return array
     */
    private function getStockMovementsResult()
    {
        $builder = $this->getStockMovementsQueryBuilder();
        $builder
            ->addSelect(
                'articleDetail.id',
                'article.name as articleName',
                'articleDetail.number as articleNumber',
                'SUM(CASE WHEN stockLedgerEntry.changeAmount > 0 THEN stockLedgerEntry.changeAmount ELSE 0 END) AS sumPositiveChanges',
                'SUM(CASE WHEN stockLedgerEntry.changeAmount < 0 THEN stockLedgerEntry.changeAmount ELSE 0 END) AS sumNegativeChanges'
            )
            ->addOrderBy($this->getSorting())
            ->groupBy('articleDetail.id');

        $stockLedgerEntryTypeAliasesByType = [
            StockLedgerEntry::TYPE_PURCHASE => 'sumPurchase',
            StockLedgerEntry::TYPE_SALE => 'sumSale',
            StockLedgerEntry::TYPE_RETURN => 'sumReturn',
            StockLedgerEntry::TYPE_MANUAL => 'sumManual',
            StockLedgerEntry::TYPE_INCOMING => 'sumIncoming',
            StockLedgerEntry::TYPE_OUTGOING => 'sumOutgoing',
            StockLedgerEntry::TYPE_RELOCATION => 'sumRelocation',
            StockLedgerEntry::TYPE_INITIALIZATION => 'sumInitialization',
            StockLedgerEntry::TYPE_STOCKTAKE => 'sumStocktake',
        ];
        foreach ($stockLedgerEntryTypeAliasesByType as $type => $alias) {
            $this->addStockLedgerEntrySumSelectionByType($builder, $type, $alias);
        }

        $start = $this->getStart();
        if ($start) {
            $builder->setFirstResult($start);
        }
        $limit = $this->getLimit();
        if ($limit) {
            $builder->setMaxResults($limit);
        }
        $result = $builder->getQuery()->getArrayResult();

        // Add article variant texts
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($result, 'id'));
        foreach ($result as &$row) {
            $additionalText = $additionalTexts[$row['id']];
            $row['articleName'] .= $additionalText ? ' - ' . $additionalText : '';
        }
        unset($row);

        return $result;
    }

    /**
     * Creates and returns the basic stock movements query builder to fetch all stock movements.
     *
     * @return QueryBuilder
     */
    private function getStockMovementsQueryBuilder()
    {
        /** @var QueryBuilder $builder */
        $builder = $this->get('models')->createQueryBuilder();
        $builder
            ->from(StockLedgerEntry::class, 'stockLedgerEntry')
            ->leftJoin('stockLedgerEntry.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article');

        // Explicitly filter stock entries with changeAmount 0, since we sum up the changeAmounts and not count the
        // number of entries.
        $builder->andWhere('stockLedgerEntry.changeAmount != 0');

        $fromDate = $this->getFromDate();
        if ($fromDate) {
            $builder
                ->andWhere('stockLedgerEntry.created >= (:fromDate)')
                ->setParameter('fromDate', $fromDate->format('Y-m-d H:i:s'));
        }

        $toDate = $this->getToDate();
        if ($toDate) {
            $builder
                ->andWhere('stockLedgerEntry.created <= (:toDate)')
                ->setParameter('toDate', $toDate->format('Y-m-d H:i:s'));
        }

        $selectedWarehouses = $this->getSelectedWarehouses();
        if ($selectedWarehouses) {
            $builder
                ->andWhere('stockLedgerEntry.warehouseId IN (:selectedWarehouseIds)')
                ->setParameter('selectedWarehouseIds', $selectedWarehouses, Connection::PARAM_INT_ARRAY);
        }

        return $builder;
    }

    /**
     * Adds a SQL selection to sum up all stock ledger entry change amounts for a given type to the given query builder.
     * (E.g. "sum all sales as sumSales")
     *
     * @param QueryBuilder $builder
     * @param string $type
     * @param string $alias
     */
    private function addStockLedgerEntrySumSelectionByType($builder, $type, $alias)
    {
        // To compare the type (string) we need to use and set a parameter instead of hard-coding the type.
        $builder
            ->addSelect(
                vsprintf(
                    'SUM(CASE WHEN (stockLedgerEntry.type = :%s) THEN stockLedgerEntry.changeAmount ELSE 0 END) as %s',
                    [
                        $type,
                        $alias,
                    ]
                )
            )
            ->setParameter(':' . $type, $type);
    }

    /**
     * Returns "start" parameter from the request.
     *
     * Use default null if no param was provided
     *
     * @return null|int
     */
    private function getStart()
    {
        return $this->Request()->getParam('start');
    }

    /**
     * Returns "limit" parameter from the request.
     *
     * Set limit default to null to ensure CSV export contains all entries if no limit was given.
     *
     * @return null|int
     */
    private function getLimit()
    {
        return $this->Request()->getParam('limit');
    }

    /**
     * Returns "fromDate" parameter from the request.
     *
     * Create from-DateTime with time 00:00:00, or leave empty if no param was given.
     *
     * @return DateTime|null
     */
    private function getFromDate()
    {
        $fromDateParam = $this->Request()->getParam('fromDate');
        if ($fromDateParam) {
            $toDate = new DateTime($fromDateParam);
            $toDate->setTime(0, 0);

            return $toDate;
        }

        return null;
    }

    /**
     * Returns "toDate" parameter from the request.
     *
     * Create to-DateTime with time 23:59:59, or leave empty if no param was given.
     *
     * @return DateTime|null
     */
    private function getToDate()
    {
        $toDateParam = $this->Request()->getParam('toDate');
        if ($toDateParam) {
            $toDate = new DateTime($toDateParam);
            $toDate->setTime(23, 59, 59);

            return $toDate;
        }

        return null;
    }

    /**
     * Returns "selectedWarehouses" parameter from the request as an array of IDs or null if no param was given.
     *
     * @return array|null
     */
    private function getSelectedWarehouses()
    {
        $selectedWarehousesParam = trim($this->Request()->getParam('selectedWarehouses', ''));

        return ($selectedWarehousesParam !== '') ? explode(',', $selectedWarehousesParam) : null;
    }

    /**
     * Returns "sort" parameter from the request or default order if no param was given.
     *
     * @return array
     */
    private function getSorting()
    {
        $orderBy = $this->Request()->getParam('sort');

        // Set default orderBy if none was provided.
        if (!$orderBy) {
            $orderBy = [
                [
                    'property' => 'articleNumber',
                    'direction' => 'DESC',
                ],
            ];
        }

        return $orderBy;
    }
}
