<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

class Shopware_Controllers_Backend_ViisonPickwareERPAnalyticsStockTurnoverRate extends Shopware_Controllers_Backend_Analytics
{
    /**
     * Default sort order
     *
     * @var array
     */
    protected static $defaultSortOrder = [
        'property' => 'stockTurnoverRate',
        'direction' => 'DESC',
    ];

    /**
     * This report calculates the "stock turnover rate" (Lagerumschlags-
     * häufigkeit) and the "avarage turnover period" (Durchschnittliche
     * Lagerdauer) for all "inventory tracked" articles. The "stock turnover
     * rate" for a given period is generally defined as:
     *
     *                                number of sold articles
     *      stock turnover rate = ---------------------------------
     *                                   average inventory
     *
     * with
     *                                N inventory snapshots
     *        average inventory = ---------------------------------
     *                                         N
     *
     * The number N of inventory snapshots is arbitrary. A simple
     * implementation would take 2 snapshots: the beginning inventory and the
     * ending inventiory of the article with respect to the given period.
     *
     * To keep the complexity of the sql query manageble we use the avarage
     * inventory over all stock changes made within the given period.
     */
    public function getStockTurnoverRateAction()
    {
        $builder = $this->getQuery();
        $data = $builder->execute()->fetchAll(\PDO::FETCH_ASSOC);
        $total = (int) $builder->getConnection()->fetchColumn('SELECT FOUND_ROWS()');

        $this->send(
            $this->postprocessData($data),
            $total
        );
    }

    /**
     * Build the report's query and returns the query builder.
     *
     * @return \Doctrine\DBAL\Query\QueryBuilder
     */
    protected function getQuery()
    {
        $builder = Shopware()->Container()->get('dbal_connection')->createQueryBuilder();

        $soldQuantityStmt = $this->getSoldQuantitySubQueryStmt();

        $fromDate = $this->getFromDate();
        $toDate = $this->getToDate();
        $diffenceInDays = $fromDate->diff($toDate)->format('%a');

        $stockDateCondition = '';
        if ($this->hasFixedPeriod()) {
            $stockDateCondition = ' AND stockLedgerEntry.created >= :fromDate AND stockLedgerEntry.created < :toDate';
            $builder
                ->setParameter(':fromDate', $fromDate->format('Y-m-d H:i:s'))
                ->setParameter(':toDate', $toDate->format('Y-m-d H:i:s'));
        }

        $builder
            ->select([
                'SQL_CALC_FOUND_ROWS articleDetail.id AS articleDetailId',
                'article.id as articleId',
                'articleDetail.ordernumber AS articleOrderNumber',
                'article.name AS articleName',
                $soldQuantityStmt . ' AS soldQuantity',
                'IFNULL(AVG(stockLedgerEntry.newStock), 0.0) AS averageStock',
                'IFNULL(' . $soldQuantityStmt . ' / AVG(stockLedgerEntry.newStock), 0.0) AS stockTurnoverRate',
                'IFNULL(' . $diffenceInDays . ' / (' . $soldQuantityStmt . ' / AVG(stockLedgerEntry.newStock)), 0.0) as averageStoragePeriod',
            ])
            ->from('s_articles_details', 'articleDetail')
            ->leftJoin('articleDetail', 's_articles', 'article', 'article.id = articleDetail.articleID')
            ->leftJoin('articleDetail', 's_articles_attributes', 'articleAttributes', 'articleAttributes.articledetailsID = articleDetail.id')
            ->leftJoin('articleDetail', 'pickware_erp_stock_ledger_entries', 'stockLedgerEntry', 'stockLedgerEntry.articleDetailId = articleDetail.id AND stockLedgerEntry.changeAmount != 0' . $stockDateCondition)
            ->where('articleAttributes.pickware_stock_management_disabled != 1')
            ->addGroupBy('articleDetail.ordernumber');

        $sortOrder = $this->getSortOrder();
        $builder->addOrderBy($sortOrder['property'], $sortOrder['direction']);

        if ($this->getPageOffset()) {
            $builder->setFirstResult($this->getPageOffset());
        }

        if ($this->getPageSize()) {
            $builder->setMaxResults($this->getPageSize());
        }

        return $builder;
    }

    /**
     * Returns the correlated sub query to retrieve the number of article sales.
     *
     * @return string
     */
    protected function getSoldQuantitySubQueryStmt()
    {
        $builder = Shopware()->Container()->get('dbal_connection')->createQueryBuilder();
        $builder
            ->select([
                'IFNULL(SUM(orderDetail.quantity), 0)',
            ])
            ->from('s_order_details', 'orderDetail')
            ->leftJoin('orderDetail', 's_order', 'baseOrder', 'baseOrder.id = orderDetail.orderID')
            ->where('baseOrder.status NOT IN (-1, 4)')
            ->andWhere('orderDetail.quantity > 0')
            ->andWhere('orderDetail.articleordernumber = articleDetail.ordernumber');

        // Add date constraints if necessary
        if ($this->hasFixedPeriod()) {
            $builder
                ->andWhere($builder->expr()->gte('baseOrder.ordertime', ':fromDate'))
                ->andWhere($builder->expr()->lt('baseOrder.ordertime', ':toDate'));
        }

        return '(' . $builder->getSQL() . ')';
    }

    /**
     * Adds the appropriate variant extensions (additional text) to all
     * article names contained in the query result.
     *
     * @param  array  $items [description]
     * @return [type]        [description]
     */
    protected function postprocessData(array $items)
    {
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($items, 'articleDetailId'));
        foreach ($items as &$item) {
            // Fix additional names
            $additionalText = $additionalTexts[$item['articleDetailId']];
            $item['articleName'] = $item['articleName'] . ($additionalText ? ' ' . $additionalText : '');

            // Remove properties, which have only be requested in order
            // to determine the articles additional text
            unset($item['articleId']);
            unset($item['articleDetailId']);
        }

        return $items;
    }

    /**
     * Returns the report's offset used for result paging.
     *
     * @return integer
     */
    protected function getPageOffset()
    {
        $start = $this->Request()->getParam('start', null);

        if ($start !== null && preg_match('/^\\d+$/', $start) !== 1) {
            throw new \Exception('start parameter must be an integer.');
        }

        return $start;
    }

    /**
     * Returns the report's max. number of results used for result paging.
     *
     * @return integer
     */
    protected function getPageSize()
    {
        $limit = $this->Request()->getParam('limit', null);

        if ($limit !== null && preg_match('/^\\d+$/', $limit) !== 1) {
            throw new \Exception('limit parameter must be an integer.');
        }

        return $limit;
    }

    /**
     * Returns true if a fixed reporting period is set.
     *
     * @return boolean
     */
    protected function hasFixedPeriod()
    {
        return $this->getFromDate() !== null && $this->getToDate() !== null;
    }

    /**
     * Returns the report's from-date.
     *
     * @return \DateTime|null
     */
    protected function getFromDate()
    {
        $fromDateParam = $this->Request()->getParam('fromDate', null);

        // No special time zone handling. Assumes the parameter's time zone
        // is server time zone (because the parameter does not contain any
        // for requests generated by the Shopware analytics UI).
        return $fromDateParam ? new \DateTime($fromDateParam) : null;
    }

    /**
     * Returns the report's to-date.
     *
     * @return \DateTime|null
     */
    protected function getToDate()
    {
        $toDateParam = $this->Request()->getParam('toDate', null);

        if (!$toDateParam) {
            return null;
        }

        $toDate = new \DateTime($toDateParam);

        // We want to include the whole last day, no matter which time. For
        // this reason we use the '<' together with midnight of the following
        // day as the boundary
        return $toDate->add(new DateInterval('P1D'));
    }

    /**
     * Returns the report's sort order. If no order is given, the default
     * order is returned.
     *
     * @return array
     * @throws \Exception if the request's sort param is not valid
     */
    protected function getSortOrder()
    {
        $sortParam = $this->Request()->getParam('sort', null);
        if (!$sortParam) {
            // Use default sort order as fallback
            return self::$defaultSortOrder;
        }

        // Validate sort param
        if (!is_array($sortParam) ||
            !is_array($sortParam[0]) ||
            !isset($sortParam[0]['property']) ||
            !isset($sortParam[0]['direction'])
        ) {
            throw new \Exception(
                'sort must be an array with a \'property\' and ' .
                'a \'direction\' key.'
            );
        }

        return  $sortParam[0];
    }
}
