<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\RestApi\RestApiRequest;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

/**
 * Backend controller for the api-log module.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPApiLog extends ViisonCommonBaseController
{
    /**
     * @return array
     */
    public function getViewParams()
    {
        return [
            'apiLogSize' => $this->get('pickware.erp.rest_api_request_logger_service')->getLogSize(),
        ];
    }

    /**
     * List logs for the backend
     */
    public function getLogsAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $sort = $this->Request()->getParam('sort', []);

        // Fix sort field names
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], 'log.') !== 0) {
                $sortField['property'] = 'log.' . $sortField['property'];
            }
        }
        unset($sortField);

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        $builder = $entityManager->createQueryBuilder();
        $builder->select(
            'log'
        )->from(RestApiRequest::class, 'log')
            ->addOrderBy($sort)
            ->setFirstResult($start)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query');
        if (!empty($searchQuery)) {
            $builder->andWhere($builder->expr()->orX(
                'log.date LIKE :searchQuery',
                'log.method LIKE :searchQuery',
                'log.url LIKE :searchQuery',
                'log.requestHeaders LIKE :searchQuery',
                'log.requestData LIKE :searchQuery',
                'log.responseCode LIKE :searchQuery',
                'log.responseHeaders LIKE :searchQuery',
                'log.responseData LIKE :searchQuery',
                'log.user LIKE :searchQuery',
                'log.ipAddress LIKE :searchQuery'
            ))->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $total = $paginator->count();
        $data = $paginator->getIterator()->getArrayCopy();

        foreach ($data as &$row) {
            $row['curlCommand'] = $this->createCurlCommand($row);
            // Clean URL (must be done after creating the cURL command)
            $row['url'] = $this->getDisplayUrlFragment($row['url']);

            // 'Prettify' the JSON-ecoded fields
            if ($row['requestHeaders'] && $row['requestData'] && $this->isContentTypeJson($row['requestHeaders'])) {
                $row['requestData'] = json_encode(json_decode($row['requestData']), JSON_PRETTY_PRINT);
            }
            if ($row['responseHeaders'] && $row['responseData'] && $this->isContentTypeJson($row['responseHeaders'])) {
                $responseData = json_decode($row['responseData']);
                // Keep the original response data if json_decode failed
                if ($responseData !== null) {
                    $row['responseData'] = json_encode($responseData, JSON_PRETTY_PRINT);
                }
            }
        }
        unset($row);

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $total,
        ]);
    }

    /**
     * Truncates the API log using the api request logger service and reponds
     * the new size of the log table.
     */
    public function truncateLogAction()
    {
        $apiRequestLogger = $this->get('pickware.erp.rest_api_request_logger_service');
        $apiRequestLogger->truncateLog();

        $this->View()->assign([
            'success' => true,
            'logSize' => $apiRequestLogger->getLogSize(),
        ]);
    }

    /**
     * Convenience function to check the raw header string for its content-type
     *
     * @param string $headersString
     * @return boolean
     */
    private function isContentTypeJson($headersString)
    {
        $headers = explode("\n", $headersString);
        foreach ($headers as $header) {
            if (mb_stripos($header, 'content-type: application/json') === 0) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param string $url A fully-qualified URL
     * @return string The given URL, with everything before /api removed and url-decoded for better readability
     */
    private function getDisplayUrlFragment($url)
    {
        // We used to save the url-decoded URL, but this caused the generated cURL requests to break in some cases. Because
        // of this, we now save the real, still-urlencoded URL. For better readability in the API log, decode it now.
        // Note that decoding URLs of old entries twice (once before saving, and here again) won't really hurt
        // readability.
        $decodedUrl = urldecode($url);
        // Split the URL and remove everything before the API resource
        $urlComponents = parse_url($decodedUrl);
        $pathComponents = explode('/', $urlComponents['path']);
        $resource = array_slice($pathComponents, (array_search('api', $pathComponents) + 1));

        // Re-assemble the URL
        $urlDisplayFragment = '/' . implode('/', $resource);
        if (!empty($urlComponents['query'])) {
            $urlDisplayFragment .= '?' . $urlComponents['query'];
        }

        return $urlDisplayFragment;
    }

    /**
     * Creates and returns a new cURL bash command based on the given log data.
     *
     * @param array $logData
     * @return string
     */
    private function createCurlCommand($logData)
    {
        // Devicelogs can contain requestData which can exceed the maximum of `escapeshellarg`.
        // If this occurs we dont want to generate a curl request.
        // Since `escapeshellarg` can only handle 2621440bytes we agreed to set a maximum of 200000 characters.
        if (mb_strlen($logData['requestData']) > 2000000) {
            return '';
        }

        $curl = 'curl -i ' . escapeshellarg($logData['url']);
        // Auth
        /** @var User $user */
        $user = $this->get('models')->find(User::class, $this->get('auth')->getIdentity()->id);
        if ($user && !empty($user->getApiKey())) {
            // Default to DIGEST auth and appending the api key, use BASIC auth and no password for login route.
            // This is done because we do not know the pickware pin, only its hash. Curl will ask the user for the
            // pin in such a case.
            $isLogin = mb_substr($logData['url'], -1 * mb_strlen('/pickware/login')) === '/pickware/login';
            $authMethod = $isLogin ? 'BASIC' : 'DIGEST';
            $curl .= ' --' . $authMethod . ' -u ' . $user->getUsername() . (($isLogin) ? '' : (':' . $user->getApiKey()));
        }
        // Method
        $curl .= ' -X ' . $logData['method'];
        if ($logData['method'] === 'GET') {
            // Allow data to be URL encoded
            $curl .= ' --globoff';
        }
        // Headers
        $headers = explode("\n", $logData['requestHeaders']);
        $headers = static::removeBlacklistedHeaders($headers);
        $isJsonRequest = false;
        foreach ($headers as $header) {
            $parts = explode(': ', $header);
            if (mb_strtolower($parts[0]) === 'content-type') {
                $isJsonRequest = mb_strtolower($parts[1]) === 'application/json';
            } elseif (mb_strtolower($parts[0]) === 'user-agent') {
                // Add postfix to make the request easier to identify in the logs
                $parts[1] .= ' curl';
            }
            $curl .= ' -H ' . escapeshellarg(implode(': ', $parts));
        }
        // Request data
        if (!empty($logData['requestData'])) {
            $curl .= (($isJsonRequest) ? ' --data-binary ' : ' --data ') . escapeshellarg($logData['requestData']);
        }

        return $curl;
    }

    /**
     * Returs an array that contains only those entries of the given $headers, which are not
     * blacklisted.
     *
     * @param array $headers
     * @return array
     */
    private static function removeBlacklistedHeaders($headers)
    {
        $headerBlacklist = [
            'authorization', // Hidden when logged
            'php-auth-user', // Hidden when logged
            'php-auth-pw', // Hidden when logged
            'accept-encoding', // important: humans can't read compressed text (curl doesn't decompress the output)
            'content-length', // important: contains wrong value, let curl calculate the content length
            'cookie', // cookies aren't required
            'connection', // 'connection: close' required for non-continuous connection; curl should do this on its own
            'x-forwarded-proto', // http proxy info (won't fail but shouldn't be required)
            'x-forwarded-for', // http proxy info (won't fail but shouldn't be required)
            'x-forwarded-host', // http proxy info (won't fail but shouldn't be required)
            'x-forwarded-port', // http proxy info (won't fail but shouldn't be required)
            'x-real-ip', // http proxy info (won't fail but shouldn't be required)
            'forwarded', // http proxy info (won't fail but shouldn't be required)
            'via', // http proxy info (won't fail but shouldn't be required)
            'surrogate-capability', // http proxy info (won't fail but shouldn't be required)
            // see: https://support.cloudflare.com/hc/en-us/articles/200170986-How-does-Cloudflare-handle-HTTP-Request-headers
            'cf-ipcountry', // cloudflare proxy info, shouldn't be included
            'cf-connecting-ip', // cloudflare proxy info, shouldn't be included
            'cf-ray', // cloudflare proxy info, shouldn't be included
            'cf-visitor', // cloudflare proxy info, shouldn't be included
            'true-client-ip', // cloudflare proxy info, shouldn't be included
        ];

        return array_filter($headers, function ($header) use ($headerBlacklist) {
            $parts = explode(': ', $header);

            return !in_array(mb_strtolower($parts[0]), $headerBlacklist);
        });
    }
}
