<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

class Shopware_Controllers_Backend_ViisonPickwareERPArticleBinLocationList extends ViisonCommonBaseController
{
    /**
     * Responds a filtered and sorted list of bin location article detail mapping data incl. bin location and warehouse
     * data.
     */
    public function getBinLocationMappingsAction()
    {
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder
            ->select(
                'binLocationMapping',
                'articleDetail',
                'binLocation',
                'warehouse',
                'articleDetailConfigurations',
                'CASE WHEN binLocation.id = warehouse.nullBinLocationId THEN 1 ELSE 0 END AS HIDDEN isNullBinLocation'
            )
            ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->join('binLocationMapping.articleDetail', 'articleDetail')
            ->join('binLocationMapping.binLocation', 'binLocation')
            ->join('binLocation.warehouse', 'warehouse')
            ->join(
                'warehouse.articleDetailConfigurations',
                'articleDetailConfigurations',
                'WITH',
                'articleDetail.id = articleDetailConfigurations.articleDetailId AND warehouse.id = articleDetailConfigurations.warehouseId'
            )
            ->addFilter($filter)
            ->addOrderBy($sort);

        $mappings = $builder->getQuery()->getArrayResult();
        foreach ($mappings as &$mapping) {
            // Warehouse <-> stocks mapping is one to one in this case, express it in the returned object
            $mapping['binLocation']['warehouse']['articleDetailConfigurations'] = $mapping['binLocation']['warehouse']['articleDetailConfigurations'][0];
        }

        $this->View()->assign([
            'success' => true,
            'data' => $mappings,
            'total' => count($mappings),
        ]);
    }

    /**
     * Updates one or more existing bin location article detail mappings using the POSTed data. Currently only the
     * 'defaultMapping' flag can be updated, because all other fields must only be updated by the stock manager. Hence
     * after updating the 'defaultMapping' flag, the stock manager is used to truncate any empty mappings, that are not
     * the default mapping (anymore).
     */
    public function updateBinLocationMappingsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!is_array($data)) {
            $this->View()->success = false;

            return;
        }

        // Update only the 'defaultMapping' field of all given bin location article details
        foreach ($data as $binLocationArticleDetailData) {
            if (!isset($binLocationArticleDetailData['defaultMapping'])) {
                return;
            }

            // Try to find mapping
            $binLocationArticleDetail = $this->get('models')->find(
                ArticleDetailBinLocationMapping::class,
                $binLocationArticleDetailData['id']
            );
            if (!$binLocationArticleDetail) {
                continue;
            }

            // Select/deselect the bin location mapping as default, if necessary
            $defaultMappingFlag = $binLocationArticleDetailData['defaultMapping'];
            if ($defaultMappingFlag !== $binLocationArticleDetail->isDefaultMapping()) {
                $articleDetail = $binLocationArticleDetail->getArticleDetail();
                $warehouse = $binLocationArticleDetail->getBinLocation()->getWarehouse();
                $newDefaultBinLocation = ($defaultMappingFlag) ? $binLocationArticleDetail->getBinLocation() : null;
                $this->get('pickware.erp.stock_ledger_service')->selectDefaultBinLocation(
                    $articleDetail,
                    $warehouse,
                    $newDefaultBinLocation
                );
            }
        }

        $this->View()->success = true;
    }
}
