<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\CSRFWhitelistAware;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

class Shopware_Controllers_Backend_ViisonPickwareERPArticleOrders extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    /**
     * Disables the renderer and output buffering for all 'exportArticleOrdersList' requests
     * to be able to display CSVs as response.
     */
    public function init()
    {
        parent::init();
        if (in_array($this->Request()->getActionName(), ['exportArticleOrdersList'])) {
            Shopware()->Plugins()->Controller()->ViewRenderer()->setNoRender();
            $this->Front()->setParam('disableOutputBuffering', true);
        }
    }

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'exportArticleOrdersList'
        ];
    }

    /**
     * Skips the pre dispatch for all 'exportArticleOrdersList' requests to be able to display
     * CSVs as response.
     */
    public function preDispatch()
    {
        if (!in_array($this->Request()->getActionName(), ['exportArticleOrdersList'])) {
            parent::preDispatch();
        }
    }

    /**
     * Responds a list of filtered, paginated article order records as well as the
     * total result count.
     */
    public function getArticleOrdersListAction()
    {
        // Get query parameters
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $filter = $this->Request()->getParam('filter', []);
        $sort = $this->Request()->getParam('sort', []);
        if (empty($filter)) {
            $this->View()->success = false;

            return;
        }

        // Create the queries and fetch the results
        $total = $this->getArticleOrdersTotalCount($filter);
        $data = $this->getArticleOrdersData($filter, $sort, $start, $limit);

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => intval($total),
        ]);
    }

    /**
     * Exports a list of article orders matching the given filter to a CSV file, which
     * is then responded.
     */
    public function exportArticleOrdersListAction()
    {
        // Get query parameters
        $filter = $this->Request()->getParam('filter', []);
        if (empty($filter)) {
            $this->View()->success = false;

            return;
        }

        // Create the query and fetch the result
        $data = $this->getArticleOrdersData($filter);

        $csvRows = [];
        // Create the localised CSV header
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_article_orders/main');
        $csvRows[] = [
            $namespace->get('model/article_order/order_number'),
            $namespace->get('model/article_order/date'),
            $namespace->get('model/article_order/order_status'),
            $namespace->get('model/article_order/payment_status'),
            $namespace->get('model/article_order/article_number'),
            $namespace->get('model/article_order/article_name'),
            $namespace->get('model/article_order/quantity'),
            $namespace->get('model/article_order/price'),
            $namespace->get('model/article_order/dispatch_method'),
            $namespace->get('model/article_order/payment_method'),
            $namespace->get('model/article_order/customer_number'),
            $namespace->get('model/article_order/billing_company'),
            $namespace->get('model/article_order/billing_department'),
            $namespace->get('model/article_order/billing_salutation'),
            $namespace->get('model/article_order/billing_first_name'),
            $namespace->get('model/article_order/billing_last_name'),
            $namespace->get('model/article_order/billing_street'),
            $namespace->get('model/article_order/billing_zip_code'),
            $namespace->get('model/article_order/billing_city'),
            $namespace->get('model/article_order/billing_country'),
            $namespace->get('model/article_order/shipping_company'),
            $namespace->get('model/article_order/shipping_department'),
            $namespace->get('model/article_order/shipping_salutation'),
            $namespace->get('model/article_order/shipping_first_name'),
            $namespace->get('model/article_order/shipping_last_name'),
            $namespace->get('model/article_order/shipping_street'),
            $namespace->get('model/article_order/shipping_zip_code'),
            $namespace->get('model/article_order/shipping_city'),
            $namespace->get('model/article_order/shipping_country'),
        ];

        // Create the data rows
        foreach ($data as $row) {
            $csvRows[] = [
                $row['orderNumber'],
                $row['date']->format('d.m.Y H:i:s'),
                $row['orderStatus'],
                $row['paymentStatus'],
                $row['articleNumber'],
                $row['articleName'],
                $row['quantity'],
                $row['price'],
                $row['dispatchMethod'],
                $row['paymentMethod'],
                $row['customerNumber'],
                $row['billingCompany'],
                $row['billingDepartment'],
                $namespace->get('salutation/' . $row['billingSalutation']),
                $row['billingFirstName'],
                $row['billingLastName'],
                $row['billingStreet'],
                $row['billingZipCode'],
                $row['billingCity'],
                $row['billingCountry'],
                $row['shippingCompany'],
                $row['shippingDepartment'],
                $namespace->get('salutation/' . $row['shippingSalutation']),
                $row['shippingFirstName'],
                $row['shippingLastName'],
                $row['shippingStreet'],
                $row['shippingZipCode'],
                $row['shippingCity'],
                $row['shippingCountry'],
            ];
        }

        // Send CSV as response
        $filename = 'article_orders.csv';
        ViisonCommonUtil::respondWithCSV($this->Response(), $csvRows, $filename);
    }

    /**
     * Returns the article order rows that match the given filters as well as pagination
     * start and limit.
     *
     * @param array $filter
     * @param array $sort
     * @param int $start
     * @param int $limit
     * @return array
     */
    private function getArticleOrdersData(array $filter, array $sort = [], $start = 0, $limit = 0)
    {
        // Create the base query builder
        $builder = $this->getArticleOrdersBaseQueryBuilder($filter);

        // Join additional tables
        $builder->leftJoin('order_.orderStatus', 'orderStatus_')
                ->leftJoin('order_.paymentStatus', '_paymentStatus')
                ->leftJoin('order_.dispatch', 'dispatch')
                ->leftJoin('order_.payment', 'payment')
                ->leftJoin('order_.customer', 'customer')
                ->leftJoin('order_.billing', 'billing')
                ->leftJoin('order_.shipping', 'shipping')
                ->leftJoin('billing.country', 'bCountry')
                ->leftJoin('shipping.country', 'sCountry');

        // Add field selection to query builder
        $builder->select(
            'order_.id as orderId',
            'order_.number as orderNumber',
            'order_.orderTime as date',
            'orderStatus_.name as orderStatus',
            '_paymentStatus.name as paymentStatus',
            'orderDetail.articleId as articleId',
            'orderDetail.articleNumber as articleNumber',
            'orderDetail.articleName as articleName',
            'orderDetail.quantity as quantity',
            'orderDetail.price as price',
            'dispatch.name as dispatchMethod',
            'payment.description as paymentMethod',
            'customer.id as userId',
            'customer.number as customerNumber',
            'billing.company as billingCompany',
            'billing.department as billingDepartment',
            'billing.salutation as billingSalutation',
            'billing.firstName as billingFirstName',
            'billing.lastName as billingLastName',
            'billing.street as billingStreet',
            'billing.zipCode as billingZipCode',
            'billing.city as billingCity',
            'bCountry.name as billingCountry',
            'shipping.company as shippingCompany',
            'shipping.department as shippingDepartment',
            'shipping.salutation as shippingSalutation',
            'shipping.firstName as shippingFirstName',
            'shipping.lastName as shippingLastName',
            'shipping.street as shippingStreet',
            'shipping.zipCode as shippingZipCode',
            'shipping.city as shippingCity',
            'sCountry.name as shippingCountry'
        );

        // Apply sorting if necessary
        foreach ($sort as $sortData) {
            if (isset($sortData['property'])
                && isset($sortData['direction'])
            ) {
                $builder->orderBy(
                    $sortData['property'],
                    $sortData['direction']
                );
            }
        }

        // Add pagination
        $builder->setFirstResult($start);
        if ($limit > 0) {
            $builder->setMaxResults($limit);
        }

        return $builder->getQuery()->getArrayResult();
    }

    /**
     * Returns the total count of the given article orders base query builder.
     *
     * @param array $filter
     * @return int
     */
    private function getArticleOrdersTotalCount(array $filter)
    {
        // Select only the count of the order details
        $builder = $this->getArticleOrdersBaseQueryBuilder($filter);
        $builder->select('COUNT(orderDetail)');

        return $builder->getQuery()->getSingleScalarResult();
    }

    /**
     * Creates the base query builder, which fetches the order details and respective orders
     * and applies some general filters as well as the filters contained in the given $filter
     * array.
     *
     * @param array $filter
     * @return \Doctrine\ORM\QueryBuilder
     */
    private function getArticleOrdersBaseQueryBuilder(array $filter)
    {
        return $this->get('models')->createQueryBuilder()
            ->from(OrderDetail::class, 'orderDetail')
            ->leftJoin('orderDetail.order', 'order_')
            // Only select 'standard' articles, no vouchers (mode 2) etc.
            ->andWhere('orderDetail.mode = 0')
            // Only select orders that were not cancelled (status -1)
            ->andWhere('order_.status != -1')
            ->addFilter($filter)
            ->addOrderBy('order_.orderTime', 'DESC');
    }
}
