<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;

class Shopware_Controllers_Backend_ViisonPickwareERPArticleStock extends ViisonCommonBaseController
{
    /**
     * Responds a list of sorted, filtered and paginated stock records as well as the
     * total result count.
     */
    public function getStockListAction()
    {
        // Get query parameters
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        foreach ($sort as $element) {
            if ($element['property'] == 'created') {
                // Add an extra sort element to also sort by ID in the same direction as the created timestamp
                // to account for stock entries created in the same second
                $sort[] = [
                    'property' => 'stockLedgerEntry.id',
                    'direction' => ($element['direction']) ?: 'ASC',
                ];
                break;
            }
        }

        // Create queries and fetch results
        $repository = $this->get('models')->getRepository(StockLedgerEntry::class);
        $data = $repository->getAllStocksQuery($start, $limit, $sort, $filter)->getArrayResult();
        $total = $repository->getAllStocksTotalCountQuery($filter)->getSingleScalarResult();

        /** @var StockLedgerService $stockLedgerService */
        $stockLedgerService = $this->get('pickware.erp.stock_ledger_service');
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($data, 'articleDetailId'));
        foreach ($data as &$row) {
            // Add additional text
            $additionalText = $additionalTexts[$row['articleDetailId']];
            $row['articleName'] .= $additionalText ? ' - ' . $additionalText : '';

            $row['changedBinLocations'] = $stockLedgerService->calculateChangedBinLocations($row['id']);
        }

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => intval($total),
        ]);
    }

    /**
     * Updates the purchase price of the stock entry with the POSTed ID and, if the 'updateLotAssignedEntries'
     * is set to 'true', the purchase price of all its lot assigned entries, whose purchase price hasn't changed.
     */
    public function updateStockAction()
    {
        // Try to find the stock entry
        $stockLedgerEntryId = $this->Request()->getParam('id', 0);
        $stockEntry = $this->get('models')->find(StockLedgerEntry::class, $stockLedgerEntryId);
        if (!$stockEntry) {
            $this->View()->assign([
                'success' => false,
                'message' => sprintf('Stock entry with given ID "%d" not found', $stockLedgerEntryId),
            ]);

            return;
        }

        // Update its purchase price
        $oldPurchasePrice = $stockEntry->getPurchasePrice();
        $newPurchasePrice = $this->Request()->getParam('purchasePrice');
        $stockEntry->setPurchasePrice($newPurchasePrice);

        // Update its comment
        $oldComment = $stockEntry->getComment();
        $commentInfix = ($oldComment) ? ' / ' : '';
        $date = new \DateTime();
        $commentSuffix = 'EK wurde am ' . $date->format('d.m.Y') . ' von ' . $this->get('auth')->getIdentity()->name . ' manuell angepasst';
        $stockEntry->setComment($oldComment . $commentInfix . $commentSuffix);

        // Check whether the stock entry's lot assigned entries should be updated too
        if ($this->Request()->getParam('updateLotAssignedEntries') === 'true') {
            foreach ($stockEntry->getLotAssignedEntries() as $lotAssignedEntry) {
                if ($lotAssignedEntry->getPurchasePrice() !== $oldPurchasePrice) {
                    continue;
                }

                // Update purchase price and comment
                $lotAssignedEntry->setPurchasePrice($newPurchasePrice);
                $oldComment = $lotAssignedEntry->getComment();
                $commentInfix = ($oldComment) ? ' / ' : '';
                $lotAssignedEntry->setComment($oldComment . $commentInfix . $commentSuffix);
            }
        }

        // Save changes
        $this->get('models')->flush();

        // Create entry in backend log for price change
        $formattedOldPurchasePrice = CurrencyUtil::getFormattedPriceString($oldPurchasePrice);
        $formattedNewPurchasePrice = CurrencyUtil::getFormattedPriceString($stockEntry->getPurchasePrice());
        $text = 'Der Preis des Bestandseintrages vom ' . $stockEntry->getCreated()->format('d.m.Y') . ' für Artikel-Nr. ' . $stockEntry->getArticleDetail()->getNumber() . ' wurde von ' . $formattedOldPurchasePrice . ' auf ' . $formattedNewPurchasePrice . ' geändert.';
        ViisonCommonUtil::createBackendLogEntry('Pickware', $text);

        $this->View()->assign([
            'success' => true,
            'data' => [
                'id' => $stockLedgerEntryId,
                'purchasePrice' => $stockEntry->getPurchasePrice(),
                'comment' => $stockEntry->getComment(),
            ],
        ]);
    }

    /**
     * Sets the availabe stock value (inStock) of a given article detail to
     * a given value.
     */
    public function saveAvailableStockAction()
    {
        $articleDetailId = $this->Request()->getParam('articleDetailId', null);
        if ($articleDetailId === null) {
            throw new \Exception('Parameter \'articleDetailId\' missing.');
        }

        $articleDetail = $this->get('models')->find(ArticleDetail::class, $articleDetailId);
        if ($articleDetail === null) {
            throw new \Exception('Article detail not found.');
        }

        $availableStock = $this->Request()->getParam('availableStock', null);
        if ($availableStock === null) {
            throw new \Exception('Parameter \'availableStock\' missing.');
        }

        // Udpate available stock
        $articleDetail->setInStock($availableStock);
        $this->get('models')->flush($articleDetail);

        $this->View()->assign([
            'success' => true,
        ]);
    }
}
