<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

/**
 * Backend controller used for auto loading the article module extension that adds a new tab
 * for supplier assignment to the article details window.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPArticleSupplierAssignment extends ViisonCommonBaseController
{

    /**
     * 'Transfers' (copies) all supplier assignments of the article detail with the POSTed ID
     * to all other article details belonging to the same (main) article. Furthermore all
     * assignments that previously contained those article details are removed.
     */
    public function transferSupplierArticleDetailsAction()
    {
        // Try to find the article detail
        $articleDetailId = $this->Request()->getParam('articleDetailId', 0);
        $articleDetail = $this->get('models')->find(ArticleDetail::class, $articleDetailId);
        if (!$articleDetail) {
            $this->View()->success = false;

            return;
        }

        // Get all supplier assignments of the article detail
        $supplierArticleDetails = $this->get('models')->getRepository(ArticleDetailSupplierMapping::class)->findBy([
            'articleDetailId' => $articleDetailId,
        ]);

        // Replace the supplier assignments of all other article details
        foreach ($articleDetail->getArticle()->getDetails() as $otherArticleDetail) {
            if ($otherArticleDetail->getId() === $articleDetail->getId()) {
                continue;
            }

            // Remove all old assignments
            $oldSupplierArticleDetails = $this->get('models')->getRepository(ArticleDetailSupplierMapping::class)->findBy([
                'articleDetailId' => $otherArticleDetail->getId(),
            ]);
            foreach ($oldSupplierArticleDetails as $oldSupplierArticleDetail) {
                $this->get('models')->remove($oldSupplierArticleDetail);
            }
        }

        // Save changes (deletions), because Doctrine cannot handle insertions and deletions
        // in the same operation when they contain conflicting constraints
        $this->get('models')->flush();

        // Replace the supplier assignments of all other article details
        foreach ($articleDetail->getArticle()->getDetails() as $otherArticleDetail) {
            if ($otherArticleDetail->getId() === $articleDetail->getId()) {
                continue;
            }

            // Add the assignments
            foreach ($supplierArticleDetails as $supplierArticleDetail) {
                $newSupplierArticleDetail = new ArticleDetailSupplierMapping(
                    $otherArticleDetail,
                    $supplierArticleDetail->getSupplier()
                );
                $newSupplierArticleDetail->setMinimumOrderAmount($supplierArticleDetail->getMinimumOrderAmount());
                $newSupplierArticleDetail->setPackingUnit($supplierArticleDetail->getPackingUnit());
                $newSupplierArticleDetail->setPurchasePrice($supplierArticleDetail->getPurchasePrice());
                $newSupplierArticleDetail->setDefaultSupplier($supplierArticleDetail->isDefaultSupplier());
                $this->get('models')->persist($newSupplierArticleDetail);
            }
        }

        // Save changes (insertions)
        $this->get('models')->flush();

        $this->View()->success = true;
    }
}
