<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrder;

/**
 * Backend controller used for auto loading the article module extension that adds a new tab
 * for listing all supplier orders in the article details window.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPArticleSupplierOrderList extends Shopware_Controllers_Backend_ExtJs
{
    /**
     * Responds a paginated, filtered and sorted list of supplier orders for a specific article detail.
     */
    public function getOrderListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update sort fields
        foreach ($sort as &$sortField) {
            switch ($sortField['property']) {
                case 'created':
                    $sortField['property'] = 'supplierOrder.created';
                    break;
                case 'supplierId':
                    $sortField['property'] = 'supplier.name';
                    break;
                case 'orderNumber':
                    $sortField['property'] = 'supplierOrder.orderNumber';
                    break;
                case 'statusId':
                    $sortField['property'] = 'supplierOrder.statusId';
                    break;
                case 'deliveryDate':
                    $sortField['property'] = 'supplierOrder.deliveryDate';
                    break;
            }
        }

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'supplierOrder',
            'supplier',
            'attachments',
            'items',
            'articleDetail',
            'article',
            'attribute',
            'manufacturer',
            'warehouse'
        )->from(SupplierOrder::class, 'supplierOrder')
            ->leftJoin('supplierOrder.warehouse', 'warehouse')
            ->leftJoin('supplierOrder.supplier', 'supplier')
            ->leftJoin('supplierOrder.attachments', 'attachments')
            ->leftJoin('supplierOrder.items', 'items')
            ->leftJoin('items.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'attribute')
            ->leftJoin('article.supplier', 'manufacturer')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        // Format the results and save the total incoming stock
        $totalIncomingStock = 0;
        if (count($result) > 0 && count($result[0]['items']) > 0) {
            $totalIncomingStock = $result[0]['items'][0]['articleDetail']['attribute']['pickwareIncomingStock'];
        }

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
            'totalIncomingStock' => $totalIncomingStock,
        ]);
    }
}
