<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\CSRFWhitelistAware;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BarcodeLabelPreset;
use Shopware\Plugins\ViisonCommon\Classes\TranslationServiceFactory;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\Article\ArticleBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelGridPaperLayout;

class Shopware_Controllers_Backend_ViisonPickwareERPBarcodeLabelPresets extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    /**
     * Event to be fired before the presets are listed so that plugins can modify the data.
     */
    const EVENT_NOTIFY_GET_PRESETS = 'Shopware_Plugins_ViisonPickwareERP_BarcodeLabelConfigurator_GetPresets';

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'renderPreview',
        ];
    }

    /**
     * Returns a list of all barcode label presets for a given type. If no specific type
     * is given all barcode label presets are returned.
     */
    public function getPresetsAction()
    {
        $presetType = $this->Request()->getParam('type', null);

        $barcodeLabelPresetRepository = $this->get('models')->getRepository(BarcodeLabelPreset::class);
        $query = $barcodeLabelPresetRepository->getBarcodeLabelPresetListQuery($presetType);
        $presets = $query->getArrayResult();

        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');

        $lastPrintedPresets = $this->get('pickware.erp.plugin_config_service')->getLastPrintedPresetIds();
        $translations = $this->getPresetNameTranslations();
        foreach ($presets as &$preset) {
            $paperLayout = BarcodeLabelGridPaperLayout::createLabelGridPaperLayout(
                $preset['paperLayoutIdentifier'],
                $preset['paperWidth'],
                $preset['paperHeight']
            );
            $preset['columnAmount'] = $paperLayout->columnAmount;
            $preset['rowAmount'] = $paperLayout->rowAmount;

            // The "default" preset will be preselected in the label printing dialog
            $preset['default'] = isset($lastPrintedPresets[$preset['type']]) && $lastPrintedPresets[$preset['type']] === $preset['id'];

            $preset['translatedName'] = ($translations[$preset['id']]) ?: $preset['name'];
            $preset['isFunctionalityInstalled'] = $barcodeLabelService->getBarcodeLabelTypeByName($preset['type']) !== null;
        }

        $eventResult = $this->get('events')->notify(self::EVENT_NOTIFY_GET_PRESETS, ['presets' => $presets]);

        $this->View()->assign([
            'success' => true,
            'data' => $eventResult && $eventResult->getReturn() ? $eventResult->getReturn() : $presets,
            'total' => count($presets),
        ]);
    }

    /**
     * Returns a dictionary of all available translations of barcode label
     * preset names.
     *
     * @return array
     */
    private function getPresetNameTranslations()
    {
        $locale = $this->get('auth')->getIdentity()->locale;
        $translationComponent = TranslationServiceFactory::createTranslationService();

        $translationBatch = $translationComponent->readBatchWithFallback(
            $locale->getId(),
            null,
            'viison_pickware_erp_barcode_label_preset_name'
        );

        $translations = [];
        foreach ($translationBatch as $key => $translation) {
            $translations[$key] = $translation['name'];
        }

        return $translations;
    }

    /**
     * Get paper layouts and translate their names.
     *
     * @return array
     */
    private function getTranslatedPaperLayouts()
    {
        $snippetManager = $this->get('snippets');
        $paperLayouts = $this->get('pickware.erp.barcode_label_service')->getPaperLayouts();

        $paperLayoutsArray = [];
        foreach ($paperLayouts as $paperLayout) {
            $paperLayoutData = $paperLayout->toArray();
            $paperLayoutData['description'] = $snippetManager->getNamespace(
                $paperLayout->snippetNamespace
            )->get($paperLayout->snippetName);

            $paperLayoutsArray[] = $paperLayoutData;
        }

        return $paperLayoutsArray;
    }

    /**
     * Returns a list of all available barcode label templates.
     */
    public function getBarcodeLabelTemplatesAction()
    {
        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');

        // Create preview image URLs
        $shopUrl = $this->Request()->getScheme() . '://' . $this->Request()->getHttpHost() . rtrim($this->Request()->getBasePath(), '/') . '/';
        $templates = array_map(
            function ($template) use ($shopUrl) {
                $template = $template->toArray();
                $template['imageFile'] = $shopUrl . $template['imageFile'];
                $template['snippetName'] = 'template/' . $template['name'] . '/name';
                $template['snippetNamespace'] = 'backend/viison_pickware_erp_barcode_label_presets/barcode_label_templates';

                return $template;
            },
            $barcodeLabelService->getLabelTemplates(ArticleBarcodeLabelType::IDENTIFIER)
        );

        $this->View()->assign([
            'success' => true,
            // Convert array of template objects (with nested block objects)
            // to an array of nested arrays
            'data' => array_values($templates),
            'total' => count($templates),
        ]);
    }

    /**
     * Returns a list of all available barcode label fields.
     */
    public function getBarcodeLabelFieldsAction()
    {
        /** @var \Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabelService $labelPrinter */
        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');
        $fields = array_map(function ($field) {
            return $field->toArray();
        }, $barcodeLabelService->getBarcodeLabelTypeByName(ArticleBarcodeLabelType::IDENTIFIER)->getFields());

        $this->View()->assign([
            'success' => true,
            // Convert array of field objects to an array of arrays
            'data' => array_values($fields),
        ]);
    }

    /**
     * Returns a list of all available barcode label paper layouts
     */
    public function getBarcodeLabelPaperLayoutsAction()
    {
        $paperLayouts = $this->getTranslatedPaperLayouts();

        $this->View()->assign([
            'success' => true,
            'data' => $paperLayouts,
        ]);
    }

    /**
     * Renders and returns a PDF preview of a barcode label preset completely specified by
     * request parameters.
     */
    public function renderPreviewAction()
    {
        /** @var \Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabelService $barcodeLabelService */
        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');
        $barcodeTypeIdentifier = $this->Request()->getParam('type');
        $barcodeLabelType = $barcodeLabelService->getBarcodeLabelTypeByName($barcodeTypeIdentifier);
        if (!$barcodeLabelType) {
            throw new \InvalidArgumentException(
                sprintf('Barcode label type with identifier "%s" not found.', $barcodeTypeIdentifier)
            );
        }

        $preset = new BarcodeLabelPreset();
        $preset->setType($barcodeLabelType::IDENTIFIER);
        $barcodeLabelService->updatePresetWithValues($preset, $this->Request()->getParams());

        $previewIdentifier = $this->Request()->getParam('previewIdentifier');
        $itemProvider = $barcodeLabelType->createItemProviderForPreview(['identifier' => $previewIdentifier]);
        $renderedDocument = $barcodeLabelService->renderBarcodeLabels($itemProvider, $preset);
        $renderedDocument->sendPdfAsHttpResponse($this->Response(), 'preview.pdf');
    }

    /**
     * Creates a new or updates a given barcode label preset.
     */
    public function savePresetAction()
    {
        /** @var \Shopware\Components\Model\ModelManager $entityManager */
        $entityManager = $this->get('models');

        $presetId = $this->Request()->getParam('id', null);
        $preset = $entityManager->find(BarcodeLabelPreset::class, $presetId);
        if ($preset === null) {
            $preset = new BarcodeLabelPreset();
        }

        $entityManager->getConnection()->beginTransaction();
        try {
            $this->get('pickware.erp.barcode_label_service')->updatePresetWithValues(
                $preset,
                $this->Request()->getParams()
            );

            // Persists presets and its blocks
            $entityManager->persist($preset);
            $entityManager->flush($preset);

            $entityManager->getConnection()->commit();
        } catch (InvalidArgumentException $exception) {
            $entityManager->getConnection()->rollback();

            $this->View()->assign([
                'success' => false,
                'internal_description' => 'An error occured. The barcode label preset could not be saved.',
            ]);

            return;
        }

        $this->View()->assign([
            'success' => true,
            'id' => $preset->getId(),
        ]);
    }

    /**
     * Deletes a barcode label preset specified by its id.
     */
    public function deletePresetAction()
    {
        /** @var \Shopware\Components\Model\ModelManager $entityManager */
        $entityManager = $this->get('models');

        $presetId = $this->Request()->getParam('id', null);
        $preset = $entityManager->find(BarcodeLabelPreset::class, $presetId);
        if ($preset === null) {
            $this->View()->assign([
                'success' => false,
                'internal_description' => 'Barcode label preset not found.',
            ]);

            return;
        }

        $entityManager->remove($preset);
        $entityManager->flush($preset);

        $this->View()->assign([
            'success' => true,
        ]);
    }

    /**
     * Clones a barcode label presets specified by its id and sets the name of the
     * newly created preset to a given value.
     */
    public function clonePresetAction()
    {
        /** @var \Shopware\Components\Model\ModelManager $entityManager */
        $entityManager = $this->get('models');

        $sourcePresetId = $this->Request()->getParam('id', null);
        $sourcePreset = $entityManager->find(BarcodeLabelPreset::class, $sourcePresetId);
        if ($sourcePreset === null) {
            $this->View()->assign([
                'success' => false,
                'internal_description' => 'Barcode label preset not found.',
            ]);

            return;
        }

        $name = $this->Request()->getParam('name', null);
        if ($name === null) {
            throw new \InvalidArgumentException('Parameter \'name\' missing.');
        }

        // Clone preset and its blocks
        $preset = clone $sourcePreset;
        $preset->setName($name);
        foreach ($sourcePreset->getBlocks() as $sourcePresetBlock) {
            $presetBlock = clone $sourcePresetBlock;
            $presetBlock->setPreset($preset);
            $preset->getBlocks()->add($presetBlock);
        }

        $entityManager->persist($preset);
        $entityManager->flush($preset);

        $this->View()->assign([
            'success' => true,
            'id' => $preset->getId(),
        ]);
    }
}
