<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\EntityNotFoundException;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\CSRFWhitelistAware;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\ArticleBarcodeLabel;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BarcodeLabelPreset;
use Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\BinLocationBarcodeLabel;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Article\Detail;
use Shopware\Models\Customer\Group as CustomerGroup;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\AbstractBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\Article\ArticleBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelFacadeService;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BinLocation\BinLocationBarcodeLabelType;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabel;

class Shopware_Controllers_Backend_ViisonPickwareERPBarcodeLabelPrinting extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    /**
     * {@inheritdoc}
     *
     * Disables the renderer and output buffering for all 'getBarcodeLabelPDF' requests
     * to be able to display PDFs as response.
     */
    public function init()
    {
        parent::init();
        if (in_array($this->Request()->getActionName(), ['renderBarcodeLabelsAction'])) {
            Shopware()->Plugins()->Controller()->ViewRenderer()->setNoRender();
            $this->Front()->setParam('disableOutputBuffering', true);
        }
    }

    /**
     * {@inheritdoc}
     *
     * Overwrite the preDispatch method from Shopware_Controllers_Backend_ExtJs to disable the JSON renderer for our
     * actions that create PDFs.
     */
    public function preDispatch()
    {
        if (!in_array($this->Request()->getActionName(), ['renderBarcodeLabelsAction'])) {
            parent::preDispatch();
        }
    }

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'renderBarcodeLabels',
        ];
    }

    /**
     * @inheritdoc
     */
    public function getExtraSnippetNamespaces()
    {
        return [
            'backend/customer/view/detail',
            'backend/viison_common_grid_label_printing/main',
        ];
    }

    /**
     * Adds a barcode label to the printing queue.
     */
    public function enqueueBarcodeLabelAction()
    {
        /** @var BarcodeLabelFacadeService $barcodeLabelService */
        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');

        $barcodeTypeIdentifier = $this->Request()->getParam('type');
        $barcodeType = $barcodeLabelService->getBarcodeLabelTypeByName($barcodeTypeIdentifier);
        if (!$barcodeType) {
            throw new \InvalidArgumentException(
                sprintf('Barcode label type with identifier "%s" not found.', $barcodeTypeIdentifier)
            );
        }

        $identifierArray = $this->Request()->getParam('identifiers');
        if (!is_array($identifierArray) || count($identifierArray) === 0) {
            throw new \InvalidArgumentException(
                sprintf('No identifier array given for barcode label type %s.', $barcodeTypeIdentifier)
            );
        }

        $quantitiesArray = $this->Request()->getParam('quantities');
        if (is_array($quantitiesArray) && (count($quantitiesArray) !== count($identifierArray))) {
            throw new \InvalidArgumentException(sprintf(
                'Wrong number of arguments given. %d quantities for %d identifiers.',
                count($quantitiesArray),
                count($identifierArray)
            ));
        }

        foreach ($identifierArray as $key => $identifier) {
            $quantity = is_array($quantitiesArray) ? $quantitiesArray[$key] : 1;
            $barcodeType->enqueueForPrinting($identifier, $quantity);
        }

        $this->View()->success = true;
    }

    /**
     * This action adds all article labels for all articles that are matching the product filter.
     * The format used in this parameter is the same as in the Shopware article list.
     *
     * @throws \RuntimeException
     */
    public function addAllFilteredArticleBarcodeLabelsAction()
    {
        $productFilter = $this->Request()->getParam('productFilter');
        $productFilter = json_decode($productFilter, true);
        if ($productFilter === false) {
            throw new \RuntimeException('Could not decode product filter');
        }

        // Get all article details matching the 'productFilter' filter
        $result = $this->get('multi_edit.product')->filter($productFilter, 0, null);

        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');
        $barcodeType = $barcodeLabelService->getBarcodeLabelTypeByName(ArticleBarcodeLabelType::IDENTIFIER);
        foreach ($result['data'] as $article) {
            $barcodeType->enqueueForPrinting($article['Detail_number']);
        }

        // Save the data
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * This action adds barcode labels for all bin locations in the warehouse with the given ID
     * and matching the given, optional search query.
     */
    public function addAllFilteredBinLocationBarcodeLabelsAction()
    {
        $warehouseId = $this->Request()->getParam('warehouseId');
        $searchQuery = $this->Request()->getParam('searchQuery');

        // Find all non-default bin locations matching the warehouse and query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select('binLocation.id')
                ->from(BinLocation::class, 'binLocation')
                ->where('binLocation.code != :defaultLocationCode')
                ->setParameter('defaultLocationCode', BinLocation::NULL_BIN_LOCATION_CODE);
        if ($warehouseId) {
            $builder->andWhere('binLocation.warehouseId = :warehouseId')
                    ->setParameter('warehouseId', $warehouseId);
        }
        if ($searchQuery) {
            $builder->andWhere('binLocation.code LIKE :searchQuery')
                    ->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }
        $binLocationIds = $builder->getQuery()->getArrayResult();

        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');
        $barcodeType = $barcodeLabelService->getBarcodeLabelTypeByName(BinLocationBarcodeLabelType::IDENTIFIER);
        foreach ($binLocationIds as $row) {
            $barcodeType->enqueueForPrinting($row['id']);
        }

        // Save the data
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Updates the article barcode label entry with the given id using the values from the request.
     */
    public function updateArticleBarcodeLabelAction()
    {
        $this->handleUpdateBarcodeLabel(ArticleBarcodeLabel::class);
    }

    /**
     * Updates the bin location barcode label entry with the given id using the values from the request.
     */
    public function updateBinLocationBarcodeLabelAction()
    {
        $this->handleUpdateBarcodeLabel(BinLocationBarcodeLabel::class);
    }

    /**
     * Generic handler for updating barcode labels.
     *
     * @param string $labelClass the fully-qualified type of barcode to update
     */
    private function handleUpdateBarcodeLabel($labelClass)
    {
        $barcodeLabelsData = $this->Request()->getParam('data', []);
        if (count($barcodeLabelsData) !== 1) {
            $this->View()->assign([
                'success' => false,
                'message' => 'Please specify exactly one barcode label.',
            ]);

            return;
        }
        $barcodeLabelData = $barcodeLabelsData[0];

        /** @var BarcodeLabel $barcodeLabel */
        $barcodeLabel = $this->get('models')->find($labelClass, $barcodeLabelData['id']);
        if (!$barcodeLabel) {
            $this->View()->assign([
                'success' => false,
                'message' => sprintf('Barcode label with ID %d does not exist.', $barcodeLabelData['id']),
            ]);

            return;
        }

        // Get the quantity from the request
        $quantity = $barcodeLabelData['quantity'] ? intval($barcodeLabelData['quantity']) : null;
        if ($quantity === null) {
            $this->View()->assign([
                'success' => false,
                'message' => 'Required parameter "quantity" missing.',
            ]);

            return;
        }

        // Save the quantity in the database
        $barcodeLabel->setQuantity($quantity);
        $this->get('models')->flush($barcodeLabel);

        $this->View()->success = true;
    }

    /**
     * Returns a list of sorted, filtered and paginated article barcode labels.
     */
    public function getArticleBarcodeLabelListAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 13);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Fetch the paginated barcode labels
        $query = $this->get('models')->getRepository(ArticleBarcodeLabel::class)->getArticleBarcodeLabelListQuery($start, $limit, $sort, $filter);
        $paginator = new Paginator($query);
        $result = $paginator->getIterator();
        $totalResult = $paginator->count();

        // Format the result data
        $data = [];
        foreach ($result as $label) {
            try {
                $labelData = $this->mapBasicLabelData($label);
                /** @var Detail $articleDetail */
                $articleDetail = $label->getArticleDetail();
                $name = $articleDetail->getArticle()->getName();
                $additionalText = ViisonCommonUtil::getVariantAdditionalText($articleDetail->getArticle()->getId(), $articleDetail->getId(), $articleDetail->getNumber());
                if (!empty($additionalText)) {
                    $name .= ' - ' . $additionalText;
                }
                $labelData['articleName'] = $name;
                $labelData['articleNumber'] = $articleDetail->getNumber();
                $labelData['price'] = $this->barcodeLabelPriceForVariant($articleDetail);
                $data[] = $labelData;
            } catch (EntityNotFoundException $e) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement
                // Ignore labels that no longer have an article
            }
        }

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $totalResult,
        ]);
    }

    /**
     * Returns a list of sorted, filtered and paginated bin location barcode labels.
     */
    public function getBinLocationBarcodeLabelListAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 13);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Fetch the paginated barcode labels
        $query = $this->get('models')->getRepository(BinLocationBarcodeLabel::class)->getBinLocationBarcodeLabelListQuery($start, $limit, $sort, $filter);
        $paginator = new Paginator($query);
        $result = $paginator->getIterator();
        $totalResult = $paginator->count();

        // Format the result data
        $data = [];
        foreach ($result as $label) {
            $labelData = $this->mapBasicLabelData($label);
            /** @var BinLocation $binLocation */
            $binLocation = $label->getBinLocation();
            $labelData['binLocationId'] = $binLocation->getId();
            $labelData['binLocationCode'] = $binLocation->getCode();
            $labelData['warehouseCode'] = $binLocation->getWarehouse()->getCode();
            $labelData['warehouseName'] = $binLocation->getWarehouse()->getName();
            $data[] = $labelData;
        }

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $totalResult,
        ]);
    }

    /**
     * Map a barcode label's common data for display.
     *
     * @param BarcodeLabel $label
     * @return array
     */
    private function mapBasicLabelData($label)
    {
        return [
            'id' => $label->getId(),
            'quantity' => $label->getQuantity(),
            'added' => $label->getAdded(),
        ];
    }

    /**
     * Deletes all barcode labels, which match the given filter, from the database.
     */
    public function deleteArticleBarcodeLabelsAction()
    {
        $this->handleDeleteBarcodeLabel(ArticleBarcodeLabel::class);
    }

    /**
     * Deletes all barcode labels, which match the given filter, from the database.
     */
    public function deleteBinLocationBarcodeLabelsAction()
    {
        $this->handleDeleteBarcodeLabel(BinLocationBarcodeLabel::class);
    }

    /**
     * Actually perform deleteArticleBarcodeLabelsAction / deleteBinLocationBarcodeLabelsAction.
     *
     * @param string $labelClass the fully-qualified entity type to delete
     */
    private function handleDeleteBarcodeLabel($labelClass)
    {
        $data = $this->Request()->getParam('data', []);

        if (!is_array($data)) {
            $this->View()->success = false;

            return;
        }

        $builder = $this->get('models')->createQueryBuilder();
        $builder->delete($labelClass, 'barcodeLabel');

        if (count($data) > 0) {
            $labelIds = array_map(function ($label) {
                return $label['id'];
            }, $data);

            $builder->where('barcodeLabel.id IN (:labelIds)')
                ->setParameter('labelIds', $labelIds, \Doctrine\DBAL\Connection::PARAM_INT_ARRAY);
        }

        $builder->getQuery()->execute();

        $this->View()->success = true;
    }

    /**
     * Renders all barcode labels that are in the printing queue.
     */
    public function renderBarcodeLabelsAction()
    {
        $barcodeLabelService = $this->get('pickware.erp.barcode_label_service');

        $barcodeTypeIdentifier = $this->Request()->getParam('type');
        /** @var AbstractBarcodeLabelType $barcodeType */
        $barcodeType = $barcodeLabelService->getBarcodeLabelTypeByName($barcodeTypeIdentifier);

        if (!$barcodeType) {
            throw new \InvalidArgumentException(
                sprintf('Barcode label type with identifier "%s" not found.', $barcodeTypeIdentifier)
            );
        }

        $filter = $this->Request()->getParam('filter', []);
        $params = json_decode($this->Request()->getParam('params', '[]'), true);
        $sort = $this->Request()->getParam('sort', []);
        $offsetX = intval($this->Request()->getParam('startX', 1)) - 1;
        $offsetY = intval($this->Request()->getParam('startY', 1)) - 1;

        $presetId = intval($this->Request()->getParam('preset', 0));
        /** @var \Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel\Repository $repository */
        $preset = $this->get('models')->getRepository(BarcodeLabelPreset::class)->findOneBy([
            'id' => $presetId,
            'type' => $barcodeTypeIdentifier,
        ]);
        if ($preset === null) {
            throw new \InvalidArgumentException(
                sprintf('Barcode label preset with id %d and type "%s" not found.', $presetId, $barcodeTypeIdentifier)
            );
        }

        $itemProvider = $barcodeType->createItemProvider($params, $filter, $sort);
        $renderedDocument = $barcodeLabelService->renderBarcodeLabels($itemProvider, $preset, $offsetX, $offsetY);
        $this->get('pickware.erp.plugin_config_service')->setLastPrintedPresetId($barcodeTypeIdentifier, $presetId);

        $renderedDocument->sendPdfAsHttpResponse($this->Response(), 'labels.pdf');
    }

    /**
     * Calculates the gross price for the article variant.
     *
     * @param ArticleDetail $variant The article variant whose price shall be calculated.
     * @param int|null $customerGroupId
     * @return float
     */
    private function barcodeLabelPriceForVariant(ArticleDetail $variant, $customerGroupId = null)
    {
        if ($variant->getPrices()->isEmpty()) {
            // Needed for CSV imported articles without a price
            return 0.0;
        }
        $tax = $variant->getArticle()->getTax()->getTax();
        $selectedPrice = $variant->getPrices()->first();

        // Find the given customer group, use default group otherwise
        $customerGroup = null;
        if ($customerGroupId) {
            $customerGroup = $this->get('models')->find(CustomerGroup::class, $customerGroupId);
        }
        if (!$customerGroup) {
            $customerGroup = $this->get('models')->getRepository(Shop::class)->getDefault()->getCustomerGroup();
        }
        foreach ($variant->getPrices() as $price) {
            $priceCustomerGroup = $price->getCustomerGroup();
            if (!$priceCustomerGroup) {
                // May happen if a customer group was deleted
                continue;
            }
            if ($priceCustomerGroup->getKey() === $customerGroup->getKey() && $price->getFrom() === 1) {
                $selectedPrice = $price;
                break;
            }
        }

        return round($selectedPrice->getPrice() / 100 * (100 + $tax), 2);
    }
}
