<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\BinLocationStockChange;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\RelocationStockChangeList;

class Shopware_Controllers_Backend_ViisonPickwareERPBinLocationEditor extends ViisonCommonBaseController
{
    use BackendExceptionHandling;

    /**
     * Logs a new 'incoming' stock entry for the POSTed bin location.
     */
    public function saveIncomingStockAction()
    {
        try {
            $stockChange = intval($this->Request()->getParam('incomingStock', 0));
            if ($stockChange === 0) {
                $this->View()->assign([
                    'success' => false,
                    'message' => 'Parameter "incomingStock" must not be zero.',
                ]);

                return;
            }

            $articleDetail = $this->getArticleDetailParam();
            $warehouse = $this->getWarehouseParam();
            $binLocation = $this->getBinLocationParam('binLocationId', $warehouse);

            // Log an 'incoming' stock entry
            $stockChanges = $this->get('pickware.erp.stock_change_list_factory_service')->createSingleBinLocationStockChangeList(
                $binLocation,
                $stockChange
            );
            $purchasePrice = floatval($this->Request()->getParam('purchasePrice'));
            $comment = $this->Request()->getParam('comment');
            $this->get('pickware.erp.stock_ledger_service')->recordIncomingStock(
                $articleDetail,
                $stockChanges,
                $purchasePrice,
                $comment
            );

            $this->View()->assign([
                'success' => true,
                'data' => $this->getReturnData($articleDetail),
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * Logs a new 'outgoing' stock entry for the POSTed bin location.
     */
    public function saveOutgoingStockAction()
    {
        // Get the required parameters
        try {
            $stockChange = -1 * intval($this->Request()->getParam('outgoingStock', 0));
            if ($stockChange === 0) {
                $this->View()->assign([
                    'success' => false,
                    'message' => 'Parameter "outgoingStock" must not be zero.',
                ]);

                return;
            }
            $articleDetail = $this->getArticleDetailParam();
            $warehouse = $this->getWarehouseParam();
            $binLocation = $this->getBinLocationParam('binLocationId', $warehouse);

            // Log an 'outgoing' stock entry
            $stockChanges = $this->get('pickware.erp.stock_change_list_factory_service')->createSingleBinLocationStockChangeList(
                $binLocation,
                $stockChange
            );
            $comment = $this->Request()->getParam('comment');
            $this->get('pickware.erp.stock_ledger_service')->recordOutgoingStock($articleDetail, $stockChanges, $comment);

            $this->View()->assign([
                'success' => true,
                'data' => $this->getReturnData($articleDetail),
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * Logs a new 'relocation' stock entry for the POSTed bin locations.
     */
    public function saveRelocationAction()
    {
        // Get the required parameters
        try {
            $articleDetail = $this->getArticleDetailParam();
            $warehouse = $this->getWarehouseParam();
            $sourceBinLocation = $this->getBinLocationParam('sourceBinLocationId', $warehouse);
            $destinationBinLocation = $this->getBinLocationParam('destinationBinLocationId');
        } catch (\Exception $e) {
            $this->View()->assign([
                'success' => false,
                'message' => $e->getMessage(),
            ]);

            return;
        }

        // Log an 'incoming' stock entry
        $relocatedStock = intval($this->Request()->getParam('relocatedStock', 0));
        $sourceBinLocationChange = new BinLocationStockChange($sourceBinLocation, (-1 * $relocatedStock));
        $destinationBinLocationChange = new BinLocationStockChange($destinationBinLocation, $relocatedStock);
        $relocationStockChanges = new RelocationStockChangeList(
            [$sourceBinLocationChange],
            $destinationBinLocationChange
        );
        $comment = $this->Request()->getParam('comment');
        $this->get('pickware.erp.stock_ledger_service')->recordRelocatedStock(
            $articleDetail,
            $relocationStockChanges,
            $comment
        );

        $this->View()->assign([
            'success' => true,
            'data' => [
                'data' => $this->getReturnData($articleDetail),
            ],
        ]);
    }

    /**
     * Saves the stock limits (minimum stock and target stock) of a given article detail / warehouse combination.
     */
    public function saveStockLimitsAction()
    {
        $warehouseConfigId = $this->Request()->getParam('warehouseConfigId', null);
        $minimumStock = $this->Request()->getParam('minimumStock', 0);
        $targetStock = $this->Request()->getParam('targetStock', 0);

        if (!$warehouseConfigId) {
            $this->View()->assign([
                'success' => false,
                'message' => 'Parameter \'warehouseConfigId\' missing.',
            ]);

            return;
        }

        /** @var WarehouseArticleDetailConfiguration|null $warehouseConfig */
        $warehouseConfig = $this->get('models')->find(WarehouseArticleDetailConfiguration::class, $warehouseConfigId);
        if (!$warehouseConfig) {
            $this->View()->assign([
                'success' => false,
                'message' => 'Warehouse stock entity missing.',
            ]);

            return;
        }

        $warehouseConfig->setMinimumStock($minimumStock);
        $warehouseConfig->setTargetStock($targetStock);
        $this->get('models')->flush($warehouseConfig);

        $this->View()->assign([
            'success' => true,
            'data' => $this->getReturnData($warehouseConfig->getArticleDetail()),
        ]);
    }

    /**
     * Responds a list of all comments set in the 'stockEntryComments' plugin config field.
     */
    public function getCommentListAction()
    {
        // Parse the stock entry comments from the app config
        $pluginConfig = $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config();
        $comments = $pluginConfig->get('stockEntryComments', '');
        $comments = ViisonCommonUtil::safeExplode(',', $comments);

        // Remove leading and trailing quotation marks and whitespace
        $comments = array_map(
            function ($comment) {
                return [
                    'value' => trim(ViisonCommonUtil::trimN($comment, '"')),
                ];
            },
            $comments
        );

        $this->View()->assign([
            'success' => true,
            'data' => $comments,
            'total' => count($comments),
        ]);
    }

    /**
     * @return ArticleDetail
     * @throws \Exception If the 'articleDetailId' parameter is not set or the respective article detail does not exist.
     */
    protected function getArticleDetailParam()
    {
        $articleDetailId = $this->Request()->getParam('articleDetailId', 0);
        $articleDetail = $this->get('models')->find(ArticleDetail::class, $articleDetailId);
        if (!$articleDetail) {
            throw new \Exception(
                sprintf('Article detail with ID %d does not exist.', $articleDetailId)
            );
        }

        return $articleDetail;
    }

    /**
     * @return Warehouse
     * @throws \Exception If the 'warehouseId' parameter is not set or the respective warehouse does not exist.
     */
    protected function getWarehouseParam()
    {
        $warehouseId = $this->Request()->getParam('warehouseId', 0);
        $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
        if (!$warehouse) {
            throw new \Exception(
                sprintf('Warehouse with ID %d does not exist.', $warehouseId)
            );
        }

        return $warehouse;
    }

    /**
     * @param string $paramName
     * @param Warehouse $warehouse (optional)
     * @return Warehouse
     * @throws \Exception If the bin location ID parameter is not set or the respective bin location does not exist.
     */
    protected function getBinLocationParam($paramName, Warehouse $warehouse = null)
    {
        $binLocationId = $this->Request()->getParam($paramName, 0);
        $filterParams = [
            'id' => $binLocationId,
        ];
        if ($warehouse) {
            $filterParams['warehouse'] = $warehouse;
        }
        $binLocation = $this->get('models')->getRepository(BinLocation::class)->findOneBy($filterParams);
        if (!$binLocation) {
            throw new \Exception(
                sprintf('Bin location with ID %d does not exist.', $binLocationId)
            );
        }

        return $binLocation;
    }

    /**
     * @param ArticleDetail $articleDetail
     * @return array
     */
    protected function getReturnData(ArticleDetail $articleDetail)
    {
        return [
            'inStock' => $articleDetail->getInStock(),
            'stockMin' => $articleDetail->getStockMin(),
            'pickwarePhysicalStockForSale' => $articleDetail->getAttribute()->getPickwarePhysicalStockForSale(),
        ];
    }
}
