<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\CSRFWhitelistAware;
use Shopware\Components\Model\ModelManager;
use Shopware\Components\Model\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentStatus;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\Document\Document as OrderDocument;
use Shopware\Models\Order\Order;
use Shopware\Models\Tax\Tax;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\ParameterHasWrongFormatException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\ParameterMissingException;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\Cancellation\OrderCanceler;
use Shopware\Plugins\ViisonPickwareERP\Components\Document\CancellationInvoiceContent;
use Shopware\Plugins\ViisonPickwareERP\Components\Document\OrderDocumentCreationService;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentProcessor;

class Shopware_Controllers_Backend_ViisonPickwareERPCancellation extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    use BackendExceptionHandling;

    const GROUP_UNSHIPPED = 'UNSHIPPED';
    const GROUP_RETURN_SHIPMENT = 'RETURN_SHIPMENT';
    const GROUP_PSEUDO = 'PSEUDO';

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'showDocumentPreview',
        ];
    }

    public function getCancellationDataAction()
    {
        try {
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');

            $orderId = $this->Request()->getParam('orderId', null);
            ParameterValidator::assertIsInteger($orderId, 'orderId');
            /** @var Order $order */
            $order = $entityManager->find(Order::class, $orderId);
            ParameterValidator::assertEntityFound($order, Order::class, $orderId, 'orderId');

            $returnShipmentId = $this->Request()->getParam('returnShipmentId', null);
            $returnShipment = null;
            if (!empty($returnShipmentId)) {
                ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');
                /** @var ReturnShipment $returnShipment */
                $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentId);
                ParameterValidator::assertEntityFound(
                    $returnShipment,
                    ReturnShipment::class,
                    $returnShipmentId,
                    'returnShipmentId'
                );
            }

            $preSelectAllReturnShipments = $this->Request()->getParam('autoPreselectAllReturnShipments');

            $documentData = [
                'id' => null,
                'createDocument' => true,
                'invoiceNumber' => '',
                'documentDate' => new DateTime(),
                'documentComment' => '',
            ];
            /** @var OrderDocument[] $invoiceDocuments */
            $invoiceDocuments = array_values(
                array_filter(
                    $order->getDocuments()->toArray(),
                    function (OrderDocument $orderDocument) {
                        return $orderDocument->getTypeId() === Document::DOCUMENT_TYPE_ID_INVOICE; // 1 = invoice
                    }
                )
            );
            if (count($invoiceDocuments) > 0) {
                $documentData['invoiceNumber'] = $invoiceDocuments[0]->getDocumentId();
            } else {
                $documentData['createDocument'] = false;
            }
            $itemGroups = $this->getCancelableItemGroups($order);
            if ($preSelectAllReturnShipments) {
                $this->preSelectAllReturmShipments($itemGroups);
            } elseif ($returnShipment !== null) {
                $this->preSelectReturnShipment($itemGroups, $returnShipment);
            }
            $preselectedOrderDetailIds = json_decode($this->Request()->getParam('preselectedOrderDetailIds'));
            if (is_array($preselectedOrderDetailIds)) {
                $this->preSelectOrderDetails($itemGroups, $preselectedOrderDetailIds);
            }

            $data = [
                'orderId' => $order->getId(),
                'orderNumber' => $order->getNumber(),
                'shippingCosts' => -1 * ($order->getNet() ? $order->getInvoiceShippingNet() : $order->getInvoiceShipping()),
                'shippingCostsNet' => $order->getInvoiceShippingNet(),
                'shippingCostsGross' => $order->getInvoiceShipping(),
                'shippingCostsTaxRate' => ViisonCommonUtil::getShippingCostsTaxRateForOrder($order),
                'cancelShippingCosts' => $order->getInvoiceShipping() !== 0.0,
                'documentData' => $documentData,
                'itemGroups' => $itemGroups,
            ];

            $this->View()->assign(
                [
                    'success' => true,
                    'data' => $data,
                ]
            );
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * @param Order $order
     * @return array
     */
    private function getCancelableItemGroups(Order $order)
    {
        /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculator */
        $orderDetailQuantityCalculator = $this->get('pickware.erp.order_detail_quantity_calculator_service');
        $groups = [];

        // First group is for unshipped items
        $groupUnshipped = [
            'type' => self::GROUP_UNSHIPPED,
            'returnShipmentId' => null,
            'returnShipmentNumber' => '',
            'cancelItemGroup' => true,
            'items' => [],
        ];
        /** @var OrderDetail $orderDetail */
        foreach ($order->getDetails() as $orderDetail) {
            $cancellableQuantity = $orderDetailQuantityCalculator->calculateRemainingQuantityToShip($orderDetail);
            if ($cancellableQuantity === 0) {
                continue;
            }
            $taxId = null;
            if ($orderDetail->getTax() && $orderDetail->getTax()->getId() !== 0) {
                $taxId = $orderDetail->getTax()->getId();
            }
            $groupUnshipped['items'][] = [
                'orderDetailId' => $orderDetail->getId(),
                'returnShipmentItemId' => null,
                'articleNumber' => $orderDetail->getArticleNumber(),
                'articleName' => $orderDetail->getArticleName(),
                'quantity' => $cancellableQuantity,
                'cancelQuantity' => 0,
                'refundQuantity' => 0,
                'price' => -1 * $orderDetail->getPrice(),
                'taxId' => $taxId,
                'taxRate' => $orderDetail->getTaxRate(),
            ];
        }
        if (count($groupUnshipped['items']) !== 0) {
            $groups[] = $groupUnshipped;
        }

        // All other groups are for canceling return shipments
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var ReturnShipment[] $returnShipments */
        $returnShipments = $entityManager->getRepository(ReturnShipment::class)->findBy(
            [
                'order' => $order,
            ]
        );
        foreach ($returnShipments as $returnShipment) {
            $groupReturnShipment = [
                'type' => self::GROUP_RETURN_SHIPMENT,
                'returnShipmentId' => $returnShipment->getId(),
                'returnShipmentNumber' => $returnShipment->getNumber(),
                'cancelItemGroup' => false,
                'items' => [],
            ];
            /** @var ReturnShipmentItem $returnShipmentItem */
            foreach ($returnShipment->getItems() as $returnShipmentItem) {
                $orderDetail = $returnShipmentItem->getOrderDetail();
                $cancellableQuantity = $returnShipmentItem->getReturnedQuantity() - $returnShipmentItem->getCancelledQuantity();
                if ($cancellableQuantity === 0) {
                    // Do not add items that cannot be canelled at all.
                    continue;
                }
                $taxId = null;
                if ($orderDetail->getTax() && $orderDetail->getTax()->getId() !== 0) {
                    $taxId = $orderDetail->getTax()->getId();
                }
                $item = [
                    'orderDetailId' => $orderDetail->getId(),
                    'returnShipmentItemId' => $returnShipmentItem->getId(),
                    'articleNumber' => $orderDetail->getArticleNumber(),
                    'articleName' => $orderDetail->getArticleName(),
                    'quantity' => $cancellableQuantity,
                    'cancelQuantity' => $cancellableQuantity,
                    'refundQuantity' => $cancellableQuantity,
                    'price' => -1 * $orderDetail->getPrice(),
                    'taxId' => $taxId,
                    'taxRate' => $orderDetail->getTaxRate(),
                ];
                $groupReturnShipment['items'][] = $item;
            }
            if (count($groupReturnShipment['items']) === 0) {
                // Do not show empty groups
                continue;
            }
            $groups[] = $groupReturnShipment;
        }

        foreach ($groups as $groupKey => &$group) {
            $group['id'] = $groupKey;
        }
        unset($group);

        // Ensure pseudo items group (items who will appear on the invoice only)
        // is displayed at the end of the groups list
        $groupPseudo = [
            'type' => self::GROUP_PSEUDO,
            'returnShipmentId' => null,
            'returnShipmentNumber' => '',
            'cancelItemGroup' => true,
            'items' => [],
        ];
        $idsOfOrderDetailsThatAlreadyExistInOtherGroups = [];
        foreach ($groups as $group) {
            foreach ($group['items'] as $item) {
                if ($item['orderDetailId']) {
                    $idsOfOrderDetailsThatAlreadyExistInOtherGroups[] = $item['orderDetailId'];
                }
            }
        }
        $orderDetailsThatDontExistInOtherGroups = $order->getDetails()->filter(function (OrderDetail $orderDetail) use ($idsOfOrderDetailsThatAlreadyExistInOtherGroups) {
            return !in_array($orderDetail->getId(), $idsOfOrderDetailsThatAlreadyExistInOtherGroups, true);
        });
        /** @var OrderDetail $orderDetail */
        foreach ($orderDetailsThatDontExistInOtherGroups as $orderDetail) {
            $taxId = null;
            if ($orderDetail->getTax() && $orderDetail->getTax()->getId() !== 0) {
                $taxId = $orderDetail->getTax()->getId();
            }
            $groupPseudo['items'][] = [
                'isPseudoItem' => true,
                'orderDetailId' => null,
                'returnShipmentItemId' => null,
                'articleNumber' => $orderDetail->getArticleNumber(),
                'articleName' => $orderDetail->getArticleName(),
                'quantity' => 0,
                'cancelQuantity' => 0,
                'refundQuantity' => 0,
                'price' => -1 * $orderDetail->getPrice(),
                'taxId' => $taxId,
                'taxRate' => $orderDetail->getTaxRate(),
            ];
        }

        $groups[] = $groupPseudo;

        return $groups;
    }

    /**
     * @param array $itemGroups
     * @param ReturnShipment $returnShipment
     */
    private function preSelectReturnShipment(array &$itemGroups, ReturnShipment $returnShipment)
    {
        foreach ($itemGroups as &$itemGroup) {
            if ($itemGroup['type'] === self::GROUP_RETURN_SHIPMENT
                && $itemGroup['returnShipmentId'] === $returnShipment->getId()
            ) {
                $itemGroup['cancelItemGroup'] = true;
            }
        }
    }

    /**
     * @param array $itemGroups
     */
    private function preSelectAllReturmShipments(array &$itemGroups)
    {
        foreach ($itemGroups as &$itemGroup) {
            if ($itemGroup['type'] === self::GROUP_RETURN_SHIPMENT) {
                $itemGroup['cancelItemGroup'] = true;
            }
        }
    }

    /**
     * @param array $itemGroups
     * @param array $preselectedOrderDetailIds
     */
    private function preSelectOrderDetails(array &$itemGroups, array $preselectedOrderDetailIds)
    {
        foreach ($itemGroups as &$itemGroup) {
            if ($itemGroup['type'] === self::GROUP_UNSHIPPED) {
                foreach ($itemGroup['items'] as &$item) {
                    if (in_array($item['orderDetailId'], $preselectedOrderDetailIds)) {
                        $item['cancelQuantity'] = $item['quantity'];
                        $item['refundQuantity'] = $item['quantity'];
                    }
                }
            }
        }
    }

    public function getTaxesAction()
    {
        /** @var QueryBuilder $builder */
        $builder = $this->get('models')->createQueryBuilder();
        $taxes = $builder
            ->select('tax')
            ->from(Tax::class, 'tax')
            ->getQuery()
            ->getArrayResult();

        $this->View()->assign([
            'success' => true,
            'data' => $taxes,
            'total' => count($taxes),
        ]);
    }

    public function showDocumentPreviewAction()
    {
        try {
            $cancellationData = json_decode($this->Request()->getParam('data', null), true);
            $this->validateCancellationData($cancellationData);
            $cancellationInvoiceContent = $this->createCancellationInvoiceContent($cancellationData);
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');
            /** @var OrderDocumentCreationService $orderDocumentCreationService */
            $orderDocumentCreationService = $this->get('pickware.erp.order_document_creation_service');
            $cancellationInvoiceContent->showDocumentPreview($orderDocumentCreationService, $entityManager);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function performCancellationAction()
    {
        try {
            $cancellationData = $this->Request()->getParam('data', null);

            $this->validateCancellationData($cancellationData);
            $this->performCancellation($cancellationData);
            $this->updateReturnShipmentStatusToCancelled($cancellationData);
            $document = $this->createCancellationDocument($cancellationData);
            if ($document) {
                $this->assignDocumentToReturnShipments($document, $cancellationData);
            }

            $this->View()->assign([
                'success' => true, // false,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * @param mixed $cancellationData
     * @throws ParameterHasWrongFormatException
     * @throws CustomValidationException
     * @throws ParameterMissingException
     */
    private function validateCancellationData($cancellationData)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        ParameterValidator::assertIsArray($cancellationData, 'data');
        $treePosition = 'data.';

        ParameterValidator::assertArrayElementIsInteger($cancellationData, 'orderId', $treePosition . 'orderId');
        /** @var Order $order */
        $order = $entityManager->find(Order::class, $cancellationData['orderId']);
        ParameterValidator::assertEntityFound($order, Order::class, $cancellationData['orderId'], $treePosition . 'orderId');

        ParameterValidator::assertArrayElementIsArray($cancellationData, 'documentData', $treePosition . 'documentData');
        $documentData = $cancellationData['documentData'];
        if ($documentData['createDocument']) {
            if (isset($documentData['documentDate'])) {
                try {
                    new \DateTime($documentData['documentDate']);
                } catch (\Exception $e) {
                    throw new ParameterHasWrongFormatException($treePosition . 'documentData.documentDate', 'ISO 8601 formatted date');
                }
            }
            if ($cancellationData['cancelShippingCosts']) {
                ParameterValidator::assertArrayElementIsNumber($cancellationData, 'shippingCostsGross', $treePosition . 'shippingCostsGross');
                ParameterValidator::assertArrayElementIsNumber($cancellationData, 'shippingCostsNet', $treePosition . 'shippingCostsNet');
                ParameterValidator::assertArrayElementIsNumber($cancellationData, 'shippingCostsTaxRate', $treePosition . 'shippingCostsTaxRate');
            }
        }

        ParameterValidator::assertArrayElementIsArray($cancellationData, 'itemGroups', $treePosition . 'itemGroups');
        foreach ($cancellationData['itemGroups'] as $itemGroupKey => $itemGroupData) {
            $treePosition = 'data.itemGroups[' . $itemGroupKey . ']';
            ParameterValidator::assertIsArray($itemGroupData, $treePosition);
            $treePosition .= '.';
            ParameterValidator::assertArrayElementIsNotNull($itemGroupData, 'type', $treePosition . 'type');
            if (!in_array($itemGroupData['type'], [self::GROUP_RETURN_SHIPMENT, self::GROUP_UNSHIPPED, self::GROUP_PSEUDO], true)) {
                throw new ParameterHasWrongFormatException(
                    $treePosition . 'type',
                    sprintf('"%s"|"%s"|"%s"', self::GROUP_RETURN_SHIPMENT, self::GROUP_UNSHIPPED, self::GROUP_PSEUDO)
                );
            }

            if ($itemGroupData['type'] === self::GROUP_RETURN_SHIPMENT) {
                ParameterValidator::assertArrayElementIsNotNull($itemGroupData, 'returnShipmentId', $treePosition . 'returnShipmentId');
                /** @var ReturnShipment $returnShipment */
                $returnShipment = $entityManager->find(ReturnShipment::class, $itemGroupData['returnShipmentId']);
                ParameterValidator::assertEntityFound(
                    $returnShipment,
                    ReturnShipment::class,
                    $itemGroupData['returnShipmentId'],
                    $treePosition . 'returnShipmentId'
                );
            }

            ParameterValidator::assertArrayElementIsArray($itemGroupData, 'items');
            foreach ($itemGroupData['items'] as $itemKey => $itemData) {
                $treePosition = 'data.itemGroups[' . $itemGroupKey . '].items[' . $itemKey . ']';
                ParameterValidator::assertIsArray($itemData, $treePosition);
                $treePosition .= '.';

                if ($cancellationData['createDocument'] && $itemGroupData['type'] === self::GROUP_PSEUDO) {
                    ParameterValidator::assertArrayElementIsInteger($itemData, 'quantity', $treePosition . 'quantity');
                    ParameterValidator::assertArrayElementIsNotNull($itemData, 'articleName', $treePosition . 'articleName');
                    ParameterValidator::assertArrayElementIsNumber($itemData, 'price', $treePosition . 'price');

                    if (isset($itemData['taxId'])) {
                        ParameterValidator::assertArrayElementIsInteger($itemData, 'taxId', $treePosition . 'taxId');
                        /** @var Tax $tax */
                        $tax = $entityManager->find(Tax::class, $itemData['taxId']);
                        if (!isset($itemData['taxRate'])) {
                            ParameterValidator::assertEntityFound($tax, Tax::class, $itemData['taxId'], $treePosition . 'taxId');
                        } elseif (!$tax) {
                            ParameterValidator::assertArrayElementIsNumber($itemData, 'taxRate', $treePosition . 'taxRate');
                        }
                    } elseif (isset($itemData['taxRate'])) {
                        ParameterValidator::assertArrayElementIsNumber($itemData, 'taxRate', $treePosition . 'taxRate');
                    } else {
                        throw new ParameterMissingException($treePosition . 'taxId');
                    }
                }

                if ($itemGroupData['type'] === self::GROUP_RETURN_SHIPMENT || $itemGroupData['type'] === self::GROUP_UNSHIPPED) {
                    ParameterValidator::assertArrayElementIsInteger($itemData, 'cancelQuantity', $treePosition . 'cancelQuantity');
                    if ($cancellationData['createDocument']) {
                        ParameterValidator::assertArrayElementIsInteger($itemData, 'refundQuantity', $treePosition . 'refundQuantity');
                    }
                }

                if ($itemGroupData['type'] === self::GROUP_RETURN_SHIPMENT) {
                    ParameterValidator::assertArrayElementIsInteger($itemData, 'returnShipmentItemId', $treePosition . 'returnShipmentItemId');
                    /** @var ReturnShipmentItem $returnShipmentItem */
                    $returnShipmentItem = $entityManager->find(ReturnShipmentItem::class, $itemData['returnShipmentItemId']);
                    ParameterValidator::assertEntityFound(
                        $returnShipmentItem,
                        ReturnShipmentItem::class,
                        $itemData['returnShipmentItemId'],
                        $treePosition . 'returnShipmentItemId'
                    );
                    if ($returnShipmentItem->getReturnShipment()->getId() !== $returnShipment->getId()) {
                        throw new CustomValidationException(
                            $treePosition . 'returnShipmentItemId',
                            $itemData['returnShipmentItemId'],
                            sprintf(
                                'ReturnShipmentItem (id=%d) does not belong to the given ReturnShipment (id=%d)',
                                $returnShipmentItem->getId(),
                                $returnShipment->getId()
                            )
                        );
                    }
                }
                if ($itemGroupData['type'] === self::GROUP_UNSHIPPED || ($itemGroupData['type'] === self::GROUP_PSEUDO && isset($itemData['orderDetailId']))) {
                    ParameterValidator::assertArrayElementIsInteger($itemData, 'orderDetailId', $treePosition . 'orderDetailId');
                    /** @var OrderDetail $orderDetail */
                    $orderDetail = $entityManager->find(OrderDetail::class, $itemData['orderDetailId']);
                    ParameterValidator::assertEntityFound(
                        $orderDetail,
                        OrderDetail::class,
                        $itemData['orderDetailId'],
                        $treePosition . 'orderDetailId'
                    );
                    if ($orderDetail->getOrder()->getId() !== $order->getId()) {
                        throw new CustomValidationException(
                            $treePosition . 'orderDetailId',
                            $itemData['orderDetailId'],
                            sprintf(
                                'OrderDetail (id=%d) does not belong to the given Order (id=%d)',
                                $orderDetail->getId(),
                                $order->getId()
                            )
                        );
                    }
                }
            }
        }
    }

    /**
     * @param array $cancellationData
     */
    private function performCancellation(array $cancellationData)
    {
        /** @var OrderCanceler $orderCancelerService */
        $orderCancelerService = $this->get('pickware.erp.order_canceler_service');
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        foreach ($cancellationData['itemGroups'] as $itemGroupData) {
            if (!$itemGroupData['cancelItemGroup']) {
                continue;
            }
            foreach ($itemGroupData['items'] as $itemData) {
                if ($itemGroupData['type'] === self::GROUP_RETURN_SHIPMENT) {
                    /** @var ReturnShipmentItem $returnShipmentItem */
                    $returnShipmentItem = $entityManager->find(ReturnShipmentItem::class, $itemData['returnShipmentItemId']);
                    $orderCancelerService->cancelReturnedQuantityOfReturnShipmentItem($returnShipmentItem, $itemData['cancelQuantity']);
                } elseif ($itemGroupData['type'] === self::GROUP_UNSHIPPED) {
                    /** @var OrderDetail $orderDetail */
                    $orderDetail = $entityManager->find(OrderDetail::class, $itemData['orderDetailId']);
                    $orderCancelerService->cancelRemainingQuantityToShipOfOrderDetail($orderDetail, $itemData['cancelQuantity']);
                }
            }
        }

        if ($cancellationData['cancelShippingCosts']) {
            /** @var Order $order */
            $order = $entityManager->find(Order::class, $cancellationData['orderId']);
            $orderCancelerService->cancelShippingCostsOfOrder($order);
        }
    }

    /**
     * @param array $cancellationData
     */
    private function updateReturnShipmentStatusToCancelled(array $cancellationData)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $entitiesToFlush = [];
        foreach ($cancellationData['itemGroups'] as $itemGroupData) {
            if (!$itemGroupData['cancelItemGroup']) {
                continue;
            }
            if ($itemGroupData['type'] !== self::GROUP_RETURN_SHIPMENT) {
                continue;
            }
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $entityManager->find(ReturnShipment::class, $itemGroupData['returnShipmentId']);
            /** @var ReturnShipmentStatus $statusCancelled */
            $statusCancelled = $entityManager->find(ReturnShipmentStatus::class, ReturnShipmentStatus::STATUS_CANCELLED_ID);
            $returnShipment->setStatusToAtLeast($statusCancelled);
            $entitiesToFlush[] = $returnShipment;
        }
        $entityManager->flush($entitiesToFlush);

        /** @var Order $order */
        $order = $entityManager->find(Order::class, $cancellationData['orderId']);
        /** @var ReturnShipmentProcessor $returnShipmentProcessor */
        $returnShipmentProcessor = $this->get('pickware.erp.return_shipment_processor_service');
        $returnShipmentProcessor->updateAccumulatedReturnShipmentStatus($order);
    }

    /**
     * @param array $cancellationData
     * @return OrderDocument|null
     */
    private function createCancellationDocument(array $cancellationData)
    {
        $documentData = $cancellationData['documentData'];
        if (!$documentData['createDocument']) {
            return null;
        }

        $cancelationInvoiceContent = $this->createCancellationInvoiceContent($cancellationData);

        if ($cancelationInvoiceContent->isEmpty()) {
            return null;
        }

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var OrderDocumentCreationService $orderDocumentCreationService */
        $orderDocumentCreationService = $this->get('pickware.erp.order_document_creation_service');

        return $cancelationInvoiceContent->createDocument($orderDocumentCreationService, $entityManager);
    }

    /**
     * @param array $cancellationData
     * @return CancellationInvoiceContent
     */
    private function createCancellationInvoiceContent(array $cancellationData)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        /** @var Order $order */
        $order = $entityManager->find(Order::class, $cancellationData['orderId']);

        $documentData = $cancellationData['documentData'];
        $cancelationInvoiceContent = new CancellationInvoiceContent($order, $documentData['invoiceNumber']);
        $cancelationInvoiceContent->setComment($documentData['documentComment']);
        if (isset($documentData['documentDate'])) {
            $cancelationInvoiceContent->setDate(new \DateTime($documentData['documentDate']));
        }
        if ($cancellationData['cancelShippingCosts']) {
            $cancelationInvoiceContent->setShippingCosts(
                $cancellationData['shippingCostsGross'],
                $cancellationData['shippingCostsNet'],
                $cancellationData['shippingCostsTaxRate']
            );
        }

        foreach ($cancellationData['itemGroups'] as $itemGroupData) {
            if (!$itemGroupData['cancelItemGroup']) {
                continue;
            }
            foreach ($itemGroupData['items'] as $itemData) {
                if ($itemGroupData['type'] === self::GROUP_RETURN_SHIPMENT) {
                    if ($itemData['refundQuantity'] === 0) {
                        continue;
                    }
                    /** @var ReturnShipmentItem $returnShipmentItem */
                    $returnShipmentItem = $entityManager->find(ReturnShipmentItem::class, $itemData['returnShipmentItemId']);
                    $cancelationInvoiceContent->addPositionForOrderDetail(
                        $returnShipmentItem->getOrderDetail(),
                        $itemData['refundQuantity']
                    );
                } elseif ($itemGroupData['type'] === self::GROUP_UNSHIPPED) {
                    if ($itemData['refundQuantity'] === 0) {
                        continue;
                    }
                    /** @var OrderDetail $orderDetail */
                    $orderDetail = $entityManager->find(OrderDetail::class, $itemData['orderDetailId']);
                    $cancelationInvoiceContent->addPositionForOrderDetail($orderDetail, $itemData['refundQuantity']);
                } else {
                    if ($itemData['quantity'] === 0) {
                        continue;
                    }
                    if ($itemData['orderDetailId']) {
                        /** @var OrderDetail $orderDetail */
                        $orderDetail = $entityManager->find(OrderDetail::class, $itemData['orderDetailId']);
                        $cancelationInvoiceContent->addPositionForOrderDetail($orderDetail, $itemData['refundQuantity']);
                    } else {
                        $tax = null;
                        if (isset($itemData['taxId'])) {
                            /** @var Tax $tax */
                            $tax = $entityManager->find(Tax::class, $itemData['taxId']);
                        }
                        if (!$tax) {
                            $tax = $itemData['taxRate'];
                        }
                        $cancelationInvoiceContent->addPositionWithoutOrderDetail(
                            $itemData['articleName'],
                            -1 * $itemData['price'],
                            $itemData['quantity'],
                            $tax,
                            $itemData['articleNumber'] ?: ''
                        );
                    }
                }
            }
        }

        return $cancelationInvoiceContent;
    }

    /**
     * @param OrderDocument $document
     * @param array $cancellationData
     */
    private function assignDocumentToReturnShipments(OrderDocument $document, array $cancellationData)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $entitiesToFlush = [];
        foreach ($cancellationData['itemGroups'] as $itemGroupData) {
            if ($itemGroupData['type'] !== self::GROUP_RETURN_SHIPMENT || !$itemGroupData['cancelItemGroup']) {
                continue;
            }
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $entityManager->find(ReturnShipment::class, $itemGroupData['returnShipmentId']);
            $returnShipment->setDocument($document);
            $entitiesToFlush[] = $returnShipment;
        }
        $entityManager->flush($entitiesToFlush);
    }
}
