<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

class Shopware_Controllers_Backend_ViisonPickwareERPCustomerArticles extends ViisonCommonBaseController
{
    /**
     * Responds a list of filtered, sorted and paginated customer article records as well as the
     * total result count.
     */
    public function getArticleListAction()
    {
        // Get query parameters
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);
        $searchQuery = $this->Request()->getParam('query');
        if (empty($filter)) {
            $this->View()->success = false;

            return;
        }

        // Create the queries and fetch the results
        $total = $this->getCustomerArticlesTotalCount($filter, $searchQuery);
        $data = $this->getCustomerArticlesData($filter, $sort, $start, $limit, $searchQuery);

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $total,
        ]);
    }

    /**
     * Returns the customer article rows that match the given filters as well as pagination
     * start and limit by applying the given sort oder.
     *
     * @param array $filter
     * @param array $sort
     * @param int $start
     * @param int $limit
     * @param string|null $searchQuery (optional)
     * @return array
     */
    private function getCustomerArticlesData(array $filter, array $sort, $start, $limit, $searchQuery = null)
    {
        // Create the base query builder
        $builder = $this->getCustomerArticlesBaseQueryBuilder($filter, $searchQuery);

        // Join additional tables
        $builder->leftJoin('orderDetail.attribute', 'orderDetailAttribute');

        // Add field selection to query builder
        $builder->select(
            'order_.id AS orderId',
            'order_.number AS orderNumber',
            'order_.orderTime AS orderTime',
            'orderDetail.articleId AS articleId',
            'orderDetail.articleNumber AS articleNumber',
            'orderDetail.articleName AS articleName',
            'orderDetail.quantity AS quantity',
            'orderDetail.price AS price',
            'orderDetail.taxRate AS taxRate',
            'orderDetailAttribute.pickwareCanceledQuantity AS canceledQuantity'
        );

        // Add sort order and pagination
        $builder->addOrderBy($sort)
                ->setFirstResult($start)
                ->setMaxResults($limit);

        return $builder->getQuery()->getArrayResult();
    }

    /**
     * Returns the total count of the given article orders base query builder.
     *
     * @param array $filter
     * @param string|null $searchQuery (optional)
     * @return int
     */
    private function getCustomerArticlesTotalCount(array $filter, $searchQuery = null)
    {
        // Select only the count of the order details
        $builder = $this->getCustomerArticlesBaseQueryBuilder($filter, $searchQuery);
        $builder->select('COUNT(orderDetail)');

        return $builder->getQuery()->getSingleScalarResult();
    }

    /**
     * Creates the base query builder, which fetches the order details and respective orders
     * and applies the given $filter array and optional search query.
     *
     * @param array $filter
     * @param string|null $searchQuery (optional)
     * @return \Doctrine\ORM\QueryBuilder
     */
    private function getCustomerArticlesBaseQueryBuilder(array $filter, $searchQuery = null)
    {
        // Create base builder
        $builder = $this->get('models')->createQueryBuilder()
            ->from(OrderDetail::class, 'orderDetail')
            ->leftJoin('orderDetail.order', 'order_')
            ->addFilter($filter);

        // Check for a search query
        if ($searchQuery) {
            $builder->andWhere($builder->expr()->orX(
                'order_.number LIKE :searchQuery',
                'orderDetail.articleNumber LIKE :searchQuery',
                'orderDetail.articleName LIKE :searchQuery'
            ));
            $builder->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        return $builder;
    }
}
