<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Models\Order\Document\Document;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Components\FileStorage\FileStorage;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

class Shopware_Controllers_Backend_ViisonPickwareERPOrderDocumentDeletion extends ViisonCommonBaseController
{
    use BackendExceptionHandling;

    /**
     * Fires the event 'ViisonPickwareERP_after_document_deletion' and, if non of the
     * subscribers returns 'false', removes the document with the POSTed ID both from the
     * database and the file system.
     */
    public function deleteDocumentAction()
    {
        try {
            // Try to find document
            $documentId = $this->Request()->getParam('documentId', 0);
            $document = $this->get('models')->find(Document::class, $documentId);
            if (!$documentId) {
                $this->View()->assign(
                    [
                        'success' => false,
                        'message' => sprintf('Document with ID "%d" not found.', $documentId),
                    ]
                );

                return;
            }

            // Fire event to allow other plugins to prevent the document deletion
            $notifyUntilResult = $this->get('events')->notifyUntil(
                'ViisonPickwareERP_before_document_deletion',
                [
                    'document' => $document,
                ]
            );
            if ($notifyUntilResult && $notifyUntilResult->getReturn() === false) {
                $this->View()->success = false;

                return;
            }

            $this->get('models')->remove($document);
            $this->get('models')->flush();

            /** @var FileStorage $documentFileStorage */
            $documentFileStorage = $this->get('viison_common.document_file_storage_service');
            $documentFileStorage->deleteFile(DocumentUtil::getDocumentFileName($document));

            $this->View()->success = true;
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }
}
