<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Models\Order\Document\Document;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;

/**
 * Backend controller for the document mailer.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPOrderDocumentMailer extends ViisonCommonBaseController
{
    use BackendExceptionHandling;

    /**
     * @inheritdoc
     */
    public function getViewParams()
    {
        // With the release of SW 5.3 Shopware introduced its own send mail button for documents. Since the new
        // Shopware functionality lacks some features of our implementation (document type specific templates) while
        // at the same time adds some new features (multiple documents as attachement) we decided to keep both buttons
        // until Shopware has integrated the missing features into their own implementation (PR is pending).
        // Nevertheless in SW < 5.3 our button must still be available as usual. Hence we need this flag to alter the
        // buttons in the JS code depending on the Shopware version.
        return [
            'useAlternativeMailButton' => ViisonCommonUtil::assertMinimumShopwareVersion('5.3'),
        ];
    }

    /**
     * Return the default content of the mail to be displayed in the mail window.
     */
    public function getMailAction()
    {
        try {
            $documentId = $this->Request()->getParam('documentId');
            $mail = $this->createMail($documentId);
            $mailPartFileNames = array_map(function ($mailPart) {
                return $mailPart->filename;
            }, $mail->getParts());
            $result = [
                'success' => true,
                'mail' => [
                    'toAddress' => implode(', ', $mail->getTo()),
                    'subject' => $mail->getPlainSubject(),
                    'content' => $mail->getPlainBodyText(), // returns plain text
                    'contentHtml' => $mail->getPlainBody(), // returns html text
                    'isHtml' => !empty($mail->getPlainBody()), // usage as in Shopware_Controllers_Backend_Order::getMailForOrder
                    'attachment' => implode(', ', $mailPartFileNames),
                    'fromName' => $mail->getFromName(),
                    'fromMail' => $mail->getFrom(),
                ],
            ];
            $this->View()->assign($result);
        } catch (\Exception $exception) {
            $this->handleException($exception);
        }
    }

    /**
     * Action to send the invoice PDF for a given order via email to the customer.
     */
    public function sendAction()
    {
        try {
            $documentId = $this->Request()->getParam('documentId', null);
            $mailData = $this->Request()->getParam('mail', []);
            $toAddress = (!empty($mailData['toAddress'])) ? $mailData['toAddress'] : null;
            $mail = $this->createMail($documentId, $toAddress);
            if ($mailData['subject']) {
                $mail->clearSubject();
                $mail->setSubject($mailData['subject']);
            }
            if ($mailData['isHtml']) {
                $mail->setBodyHtml($mailData['contentHtml']);
            } else {
                $mail->setBodyText($mailData['content']);
            }

            $mail->send();

            $this->View()->success = true;
        } catch (\Exception $exception) {
            $this->handleException($exception);
        }
    }

    /**
     * Create the specific mail for the given document
     *
     * @param int $documentId
     * @param string $toAddress An optional receiver address.
     * @return \Enlight_Components_Mail
     */
    private function createMail($documentId, $toAddress = null)
    {
        /** @var Document $document */
        $document = $this->get('models')->find(Document::class, $documentId);
        ParameterValidator::assertEntityFound($document, Document::class, $documentId, 'documentId');

        // Prepare mail context
        $order = $document->getOrder();
        $toAddress = ($toAddress) ?: $order->getCustomer()->getEmail();
        $context = [
            'orderNumber' => $order->getNumber(),
            'userSalutation' => $order->getCustomer()->getSalutation(),
            'userFirstName' => $order->getCustomer()->getFirstname(),
            'userLastName' => $order->getCustomer()->getLastname(),
        ];

        // Retrieve Shopware's default mail context by triggering the creation of an arbitrary order mail
        // in order to create the default context as an internal state of the template mail component, and
        // afterwards retrieve this internal state
        Shopware()->Modules()->Order()->createStatusMail($order->getId(), 1, 'sORDERSTATEMAIL1');
        $shopwareDefaultContext = Shopware()->TemplateMail()->getStringCompiler()->getContext();

        // Create mail from template
        $mail = Shopware()->TemplateMail()->createMail(
            Shopware_Plugins_Core_ViisonPickwareERP_Bootstrap::DOCUMENT_MAILER_MAIL_TEMPLATE_NAME . '_DocType' . $document->getTypeId(),
            array_merge($shopwareDefaultContext, $context),
            $order->getLanguageSubShop()
        );
        $mail->addTo($toAddress);

        DocumentUtil::attachDocumentToMail($document, $mail);

        return $mail;
    }
}
