<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\Model\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentProcessor;

class Shopware_Controllers_Backend_ViisonPickwareERPOrderReturnShipment extends ViisonCommonBaseController
{
    use BackendExceptionHandling;

    public function getReturnShipmentsAction()
    {
        try {
            $sort = $this->Request()->getParam('sort', []);
            $filter = $this->Request()->getParam('filter', []);

            foreach ($sort as &$sortElement) {
                if ($sortElement['property'] === 'userName') {
                    $sortElement['property'] = 'user.name';
                } elseif (mb_strpos($sortElement['property'], 'returnShipment.') !== 0) {
                    $sortElement['property'] = 'returnShipment.' . $sortElement['property'];
                }
            }
            unset($sortElement);

            /** @var QueryBuilder $queryBuilder */
            $queryBuilder = $this->get('models')->createQueryBuilder();
            $queryBuilder
                ->setAlias('returnShipment')
                ->select(
                    'returnShipment',
                    'item',
                    'orderDetail',
                    'user'
                )
                ->from(ReturnShipment::class, 'returnShipment')
                ->leftJoin('returnShipment.items', 'item')
                ->leftJoin('item.orderDetail', 'orderDetail')
                ->leftJoin('returnShipment.user', 'user')
                ->addFilter($filter)
                ->addOrderBy($sort);

            $query = $queryBuilder->getQuery();
            $query->setHydrationMode(Query::HYDRATE_ARRAY);
            $paginator = new Paginator($query);
            $total = $paginator->count();
            $returnShipments = $paginator->getIterator()->getArrayCopy();

            $data = array_map(function ($returnShipment) {
                $numberOfReturnedItems = array_sum(array_column($returnShipment['items'], 'returnedQuantity'));

                $returnShipment['numberOfReturnedItems'] = $numberOfReturnedItems;
                $returnShipment['userName'] = $returnShipment['user']['username'];
                unset($returnShipment['user']);

                $returnShipment['items'] = array_map(function ($returnShipmentItem) {
                    $returnShipmentItem['articleName'] = $returnShipmentItem['orderDetail']['articleName'];
                    $returnShipmentItem['articleNumber'] = $returnShipmentItem['orderDetail']['articleNumber'];
                    $returnShipmentItem['articleId'] = $returnShipmentItem['orderDetail']['articleId'];
                    unset($returnShipmentItem['orderDetail']);

                    return $returnShipmentItem;
                }, $returnShipment['items']);

                return $returnShipment;
            }, $returnShipments);

            $this->View()->assign([
                'success' => true,
                'data' => $data,
                'total' => $total,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function deleteReturnShipmentAction()
    {
        try {
            $returnShipmentId = $this->Request()->getParam('returnShipmentId');
            ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $this->get('models')->find(ReturnShipment::class, $returnShipmentId);
            ParameterValidator::assertEntityFound($returnShipment, ReturnShipment::class, $returnShipmentId, 'returnShipmentId');

            /** @var ReturnShipmentProcessor $returnShipmentProcessor */
            $returnShipmentProcessor = $this->get('pickware.erp.return_shipment_processor_service');
            $order = $returnShipment->getOrder();
            $returnShipmentProcessor->safelyDeleteReturnShipment($returnShipment);
            $returnShipmentProcessor->updateAccumulatedReturnShipmentStatus($order);

            $this->View()->assign([
                'success' => true,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }
}
