<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Models\Config\Element;
use Shopware\Models\Config\Form;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;

class Shopware_Controllers_Backend_ViisonPickwareERPPurchasePriceConverter extends ViisonCommonBaseController
{
    /**
     * @inheritdoc
     */
    public function getViewParams()
    {
        return [
            'ViisonPickwareERPPurchasePriceConverter' => [
                'purchasePriceModes' => $this->getPurchasePriceModes(),
            ],
        ];
    }

    /**
     * Responds both the currently active and configured purchase price modes.
     */
    public function getPurchasePriceModesAction()
    {
        $this->View()->assign([
            'success' => true,
            'purchasePriceModes' => $this->getPurchasePriceModes(),
        ]);
    }

    /**
     * Hides all future change notifications until the mode changes again.
     */
    public function hideChangeNotificationAction()
    {
        $this->hideFutureChangeNotifications();
        $this->View()->assign([
            'success' => true,
            'purchasePriceModes' => $this->getPurchasePriceModes(),
        ]);
    }

    /**
     * Compares the POSTed 'oldMode' and 'newMode' with the active and configured purchase price modes
     * respectively and, if the values match and the mode has changed, converts all purchase prices
     * depending on the currently active mode. That is, if the active mode is 'net', all purchase prices
     * are converted from net to gross prices, and if the active mode is 'gross', all purchase prices are
     * converted from gross to net prices. Finally, future change notifications are hidden until the mode
     * changes again.
     */
    public function convertPurchasePricesAction()
    {
        // Get the configured and active purchase price modes
        $configuredMode = $this->getConfiguredPurchasePriceMode();
        $activeMode = $this->getActivePurchasePriceModeConfigElement()->getValue();

        // Make sure the mode hasn't changed again since the notification was displayed
        if ($activeMode != $this->Request()->getParam('oldMode') || $configuredMode != $this->Request()->getParam('newMode') || $configuredMode == $activeMode) {
            $this->View()->assign([
                'success' => false,
                'message' => $this->get('snippets')->getNamespace('backend/viison_pickware_erp_purchase_price_converter/main')->get('convert_prices/error/mode_changed'),
                'purchasePriceModes' => $this->getPurchasePriceModes(),
            ]);

            return;
        }

        // Convert all purchase prices
        $operator = ($activeMode == 'net') ? '*' : '/';
        // Article details
        $this->get('db')->query(
            'UPDATE s_articles_details ad
            JOIN s_articles a
                ON a.id = ad.articleID
            JOIN s_core_tax t
                ON t.id = a.taxID
            SET ad.purchaseprice = ROUND(ad.purchaseprice ' . $operator . ' (1 + t.tax / 100), 2)'
        );
        // Stock entries
        $this->get('db')->query(
            'UPDATE `pickware_erp_stock_ledger_entries` AS `stockLedgerEntries`
            JOIN s_articles_details ad
                ON ad.id = `stockLedgerEntries`.articleDetailId
            JOIN s_articles a
                ON a.id = ad.articleID
            JOIN s_core_tax t
                ON t.id = a.taxID
            SET `stockLedgerEntries`.purchasePrice = ROUND(`stockLedgerEntries`.purchasePrice ' . $operator . ' (1 + t.tax / 100), 2)'
        );
        // Supplier article assignments
        $this->get('db')->query(
            'UPDATE `pickware_erp_article_detail_supplier_mappings` AS `supplierMappings`
            JOIN s_articles_details ad
                ON ad.id = `supplierMappings`.articleDetailId
            JOIN s_articles a
                ON a.id = ad.articleID
            JOIN s_core_tax t
                ON t.id = a.taxID
            SET `supplierMappings`.purchasePrice = ROUND(`supplierMappings`.purchasePrice ' . $operator . ' (1 + t.tax / 100), 2)'
        );
        // Supplier order articles
        $this->get('db')->query(
            'UPDATE `pickware_erp_supplier_order_items` AS `supplierOrderItems`
            JOIN s_articles_details ad
                ON ad.id = `supplierOrderItems`.articleDetailId
            JOIN s_articles a
                ON a.id = ad.articleID
            JOIN s_core_tax t
                ON t.id = a.taxID
            SET `supplierOrderItems`.price = ROUND(`supplierOrderItems`.price ' . $operator . ' (1 + t.tax / 100), 2)'
        );

        $this->hideFutureChangeNotifications();

        // Clear the template cache, to make sure the correct snippets are shown after reloading the backend
        $this->get('shopware.cache_manager')->clearTemplateCache();

        $this->View()->assign([
            'success' => true,
            'purchasePriceModes' => $this->getPurchasePriceModes(),
        ]);
    }

    /**
     * Saves the currently configured purchase price mode as the active purchase price mode
     * to prevent showing the change notification until the mode changes again.
     */
    private function hideFutureChangeNotifications()
    {
        // Get the configured purchase price mode and save it as the active mode
        $element = $this->getActivePurchasePriceModeConfigElement();
        $purchasePriceMode = $this->getConfiguredPurchasePriceMode();
        $element->setValue($purchasePriceMode);
        $this->get('models')->flush($element);
    }

    /**
     * @return array
     */
    private function getPurchasePriceModes()
    {
        return [
            'active' => $this->getActivePurchasePriceModeConfigElement()->getValue(),
            'configured' => $this->getConfiguredPurchasePriceMode(),
        ];
    }

    /**
     * @return string
     */
    private function getConfiguredPurchasePriceMode()
    {
        // Use 'gross' as default value, because it might be the case, that this method is called
        // before the config is added to the database (e.g. when downloading a new version of the
        // plugin, but only running the update after reloading the backend at least once). It is
        // save to use 'gross', because the described scenario can only happen when updating the
        // plugin, which will add the config element with value 'gross'.
        return $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->get('purchasePriceMode', 'gross');
    }

    /**
     * @return \Shopware\Models\Config\Element
     */
    private function getActivePurchasePriceModeConfigElement()
    {
        $elementName = 'viisonPickwareERPActivePurchasePriceMode';
        $element = $this->get('models')->getRepository(Element::class)->findOneBy([
            'name' => $elementName,
        ]);
        if (!$element) {
            // Create new config element using a fake form, since all elements must belong to a form
            $element = new Element('text', $elementName, []);
            $fakeForm = $this->get('models')->getPartialReference(Form::class, 0);
            $element->setForm($fakeForm);
            $purchasePriceMode = $this->getConfiguredPurchasePriceMode();
            $element->setValue($purchasePriceMode);
            $this->get('models')->persist($element);
            $this->get('models')->flush($element);
        }

        return $element;
    }
}
