<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Bundle\MediaBundle\MediaService;
use Shopware\Components\CSRFWhitelistAware;
use Shopware\Components\Model\ModelManager;
use Shopware\Components\Model\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentAttachment;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentInternalComment;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentStatus;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Media\Album;
use Shopware\Models\Media\Media;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\Document\Document as OrderDocument;
use Shopware\Models\Order\Order;
use Shopware\Models\Order\Status;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\CustomValidationException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\EntityNotFoundException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\ParameterHasWrongFormatException;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\ValidationExceptions\ParameterMissingException;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\OrderDetailUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentMailingException;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentMailingService;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentProcessor;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentProcessorService;
use Shopware\Plugins\ViisonPickwareERP\Components\ReturnShipment\ReturnShipmentException;

class Shopware_Controllers_Backend_ViisonPickwareERPReturnShipment extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    use BackendExceptionHandling;

    const MAIL_TYPE_RETURN_SHIPMENT_RECEIVED = 'MAIL_TYPE_RETURN_SHIPMENT_RECEIVED';
    const MAIL_TYPE_RETURN_SHIPMENT_FINALIZED = 'MAIL_TYPE_RETURN_SHIPMENT_FINALIZED';

    /**
     * @inheritdoc
     */
    public function getViewParams()
    {
        $album = $this->get('pickware.erp.plugin_config_service')->getReturnShipmentMediaAlbum();
        if ($album) {
            return [
                'viisonPickwareERPReturnShipmentAlbumId' => $album->getId(),
            ];
        }

        return [];
    }

    /**
     * Disables the renderer and output buffering for all 'downloadOrderPDF' requests
     * to be able to display PDFs as response.
     */
    public function init()
    {
        parent::init();
        if ($this->Request()->getActionName() === 'openCorrectionOfInvoice') {
            Shopware()->Plugins()->Controller()->ViewRenderer()->setNoRender();
            $this->Front()->setParam('disableOutputBuffering', true);
        }
    }

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'openCorrectionOfInvoice',
        ];
    }

    public function getReturnShipmentsAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);
        $query = $this->Request()->getParam('query', '');

        $sortPropertyMappings = [
            'customerEmail' => 'customer.email',
            'customerGroupKey' => 'customer.groupKey',
            'customerName' => 'customer.lastname',
            'customerNumber' => 'customer.number',
            'orderNumber' => '_order.number',
            'orderShopName' => 'shop.name',
            'orderStatusName' => 'orderStatus.name',
            'paymentMethodName' => 'payment.description',
            'paymentStatusName' => 'paymentStatus.name',
            'shippingMethodName' => 'dispatch.name',
            'userName' => 'user.name',
        ];
        foreach ($sort as &$sortElement) {
            if (isset($sortPropertyMappings[$sortElement['property']])) {
                $sortElement['property'] = $sortPropertyMappings[$sortElement['property']];
            } elseif (mb_strpos($sortElement['property'], 'returnShipment.') !== 0) {
                $sortElement['property'] = 'returnShipment.' . $sortElement['property'];
            }
        }
        unset($sortElement);

        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->get('models')->createQueryBuilder();
        $queryBuilder
            ->setAlias('returnShipment')
            ->select(
                'returnShipment',
                'items',
                'user',
                '_order',
                'paymentStatus',
                'orderStatus',
                'customer',
                'dispatch',
                'internalComment',
                'document',
                'shop',
                'payment',
                'orderDetail',
                'billing'
            )
            ->from(ReturnShipment::class, 'returnShipment')
            ->leftJoin('returnShipment.items', 'items')
            ->leftJoin('items.orderDetail', 'orderDetail')
            ->leftJoin('returnShipment.user', 'user')
            ->leftJoin('returnShipment.order', '_order')
            ->leftJoin('_order.paymentStatus', 'paymentStatus')
            ->leftJoin('_order.orderStatus', 'orderStatus')
            ->leftJoin('_order.customer', 'customer')
            ->leftJoin('_order.dispatch', 'dispatch')
            ->leftJoin('_order.shop', 'shop')
            ->leftJoin('_order.payment', 'payment')
            ->leftJoin('_order.billing', 'billing')
            ->leftJoin('returnShipment.internalComments', 'internalComment')
            ->leftJoin('returnShipment.document', 'document')
            ->addFilter($filter)
            ->addOrderBy($sort);

        if ($offset !== null) {
            $queryBuilder->setFirstResult($offset);
        }
        if ($limit !== null) {
            $queryBuilder->setMaxResults($limit);
        }

        if (!empty($query)) {
            $queryBuilder->andWhere(
                $queryBuilder->expr()->orX(
                    'internalComment.comment LIKE :searchQuery',
                    'returnShipment.number LIKE :searchQuery',
                    '_order.number LIKE :searchQuery',
                    'customer.firstname LIKE :searchQuery',
                    'customer.lastname LIKE :searchQuery',
                    'customer.number LIKE :searchQuery',
                    'user.email LIKE :searchQuery',
                    'user.name LIKE :searchQuery',
                    'billing.company LIKE :searchQuery'
                )
            )->setParameter('searchQuery', '%' . $query . '%');
        }

        $query = $queryBuilder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $total = $paginator->count();
        $returnShipments = $paginator->getIterator()->getArrayCopy();

        $data = array_map(function ($returnShipment) {
            $numberOfReturnedItems = array_sum(array_column($returnShipment['items'], 'returnedQuantity'));

            return [
                'id' => $returnShipment['id'],
                'statusId' => $returnShipment['statusId'],
                'created' => $returnShipment['created'],
                'number' => $returnShipment['number'],
                'orderId' => $returnShipment['order']['id'],
                'orderNumber' => $returnShipment['order']['number'],
                'orderStatusName' => $returnShipment['order']['orderStatus']['name'],
                'orderShopName' => $returnShipment['order']['shop']['name'],
                'paymentStatusName' => $returnShipment['order']['paymentStatus']['name'],
                'paymentMethodName' => $returnShipment['order']['payment']['description'],
                'numberOfReturnedItems' => $numberOfReturnedItems,
                'userId' => $returnShipment['user']['id'],
                'userName' => $returnShipment['user']['username'],
                'customerId' => $returnShipment['order']['customer']['id'],
                'customerName' => $returnShipment['order']['customer']['firstname'] . ' ' . $returnShipment['order']['customer']['lastname'],
                'customerNumber' => $returnShipment['order']['customer']['number'],
                'customerEmail' => $returnShipment['order']['customer']['email'],
                'customerCompanyName' => $returnShipment['order']['billing']['company'],
                'customerGroupKey' => $returnShipment['order']['customer']['groupKey'],
                'shippingMethodName' => $returnShipment['order']['dispatch']['name'],
                'documentId' => $returnShipment['document'] ? $returnShipment['document']['id'] : null,
                'amount' => array_sum(array_map(function ($returnShipmentItem) {
                    return $returnShipmentItem['returnedQuantity'] * $returnShipmentItem['orderDetail']['price'];
                }, $returnShipment['items'])),
            ];
        }, $returnShipments);

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $total,
        ]);
    }

    public function getReturnShipmentStatesAction()
    {
        try {
            /** @var QueryBuilder $queryBuilder */
            $queryBuilder = $this->get('models')->createQueryBuilder();
            $queryBuilder
                ->select('status')
                ->from(ReturnShipmentStatus::class, 'status')
                ->orderBy('status.processStep', 'ASC');

            $returnShipmentStatuses = $queryBuilder->getQuery()->getResult(Query::HYDRATE_ARRAY);

            $this->View()->assign([
                'success' => true,
                'data' => $returnShipmentStatuses,
                'total' => count($returnShipmentStatuses),
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function openCorrectionOfInvoiceAction()
    {
        try {
            $documentId = $this->Request()->getParam('documentId', null);
            ParameterValidator::assertIsInteger($documentId, 'documentId');
            /** @var OrderDocument $document */
            $document = $this->get('models')->find(OrderDocument::class, $documentId);
            ParameterValidator::assertEntityFound($document, OrderDocument::class, $documentId, 'documentId');
            DocumentUtil::respondWithDocumentPdf($this->Response(), $document);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function getReturnableOrdersAction()
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);
        $query = $this->Request()->getParam('query', '');

        $sortPropertyMappings = [
            'customerGroupKey' => 'customer.groupKey',
            'customerName' => 'customer.lastname',
            'customerNumber' => 'customer.number',
            'orderNumber' => '_order.number',
            'orderShopName' => 'shop.name',
            'orderStatusName' => 'orderStatus.name',
            'paymentMethodName' => 'payment.description',
            'paymentStatusName' => 'paymentStatus.name',
            'shippingMethodName' => 'dispatch.name',
        ];
        foreach ($sort as &$sortElement) {
            if (isset($sortPropertyMappings[$sortElement['property']])) {
                $sortElement['property'] = $sortPropertyMappings[$sortElement['property']];
            } elseif (mb_strpos($sortElement['property'], '_order.') !== 0) {
                $sortElement['property'] = '_order.' . $sortElement['property'];
            }
        }
        unset($sortElement);

        // Create a sub-query to filter only orders where a return shipment is possible.
        // This is when shippedQuantity > returnedQuantity
        // Using COALESCE as replacement for IFNULL as DQL does not know IFNULL
        $subQuery = sprintf(
            'SELECT __order.id
            FROM %s AS __order
            LEFT JOIN %s AS orderDetail
                WITH orderDetail.order = __order
            LEFT JOIN %s AS returnShipmentItem
                WITH returnShipmentItem.orderDetailId = orderDetail.id
            GROUP BY __order.id
            HAVING
                COALESCE(SUM(orderDetail.shipped), 0) > COALESCE(SUM(returnShipmentItem.returnedQuantity), 0)',
            Order::class,
            OrderDetail::class,
            ReturnShipmentItem::class
        );

        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $entityManager->createQueryBuilder();
        $queryBuilder
            ->setAlias('_order')
            ->select(
                '_order',
                'customer',
                'dispatch',
                'orderStatus',
                'paymentStatus',
                'shop',
                'payment',
                'billing'
            )
            ->from(Order::class, '_order')
            ->leftJoin('_order.customer', 'customer')
            ->leftJoin('_order.dispatch', 'dispatch')
            ->leftJoin('_order.orderStatus', 'orderStatus')
            ->leftJoin('_order.paymentStatus', 'paymentStatus')
            ->leftJoin('_order.shop', 'shop')
            ->leftJoin('_order.payment', 'payment')
            ->leftJoin('_order.billing', 'billing')
            ->addFilter($filter)
            ->andWhere($queryBuilder->expr()->in('_order.id', $subQuery))
            ->addOrderBy($sort);

        if ($offset !== null) {
            $queryBuilder->setFirstResult($offset);
        }
        if ($limit !== null) {
            $queryBuilder->setMaxResults($limit);
        }

        if (!empty($query)) {
            $queryBuilder->andWhere(
                $queryBuilder->expr()->orX(
                    '_order.number LIKE :searchQuery',
                    'customer.firstname LIKE :searchQuery',
                    'customer.lastname LIKE :searchQuery',
                    'customer.number LIKE :searchQuery',
                    'billing.company LIKE :searchQuery'
                )
            )->setParameter('searchQuery', '%' . $query . '%');
        }

        $query = $queryBuilder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $total = $paginator->count();
        $orders = $paginator->getIterator()->getArrayCopy();

        $data = array_map(function ($order) {
            return [
                'id' => $order['id'],
                'orderTime' => $order['orderTime'],
                'number' => $order['number'],
                'orderShopName' => $order['shop']['name'],
                'customerId' => $order['customer']['id'],
                'customerName' => $order['customer']['lastname'] . ' ' . $order['customer']['firstname'],
                'customerNumber' => $order['customer']['number'],
                'customerCompanyName' => $order['billing']['company'],
                'customerGroupKey' => $order['customer']['groupKey'],
                'shippingMethodName' => $order['dispatch']['name'],
                'orderStatusName' => $order['orderStatus']['name'],
                'paymentMethodName' => $order['payment']['description'],
                'paymentStatusName' => $order['paymentStatus']['name'],
            ];
        }, $orders);

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $total,
        ]);
    }

    public function deleteReturnShipmentAction()
    {
        try {
            $returnShipmentId = $this->Request()->getParam('returnShipmentId');
            ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $this->get('models')->find(ReturnShipment::class, $returnShipmentId);
            ParameterValidator::assertEntityFound($returnShipment, ReturnShipment::class, $returnShipmentId, 'returnShipmentId');

            /** @var ReturnShipmentProcessor $returnShipmentProcessor */
            $returnShipmentProcessor = $this->get('pickware.erp.return_shipment_processor_service');
            $order = $returnShipment->getOrder();
            $returnShipmentProcessor->safelyDeleteReturnShipment($returnShipment);
            $returnShipmentProcessor->updateAccumulatedReturnShipmentStatus($order);

            $this->View()->assign([
                'success' => true,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function getReturnShipmentAction()
    {
        try {
            $returnShipmentId = $this->Request()->getParam('returnShipmentId', null);
            $orderId = $this->Request()->getParam('orderId', null);

            if (!empty($returnShipmentId)) {
                $data = $this->getExistingReturnShipment($returnShipmentId);
            } elseif (!empty($orderId)) {
                $data = $this->getReturnShipmentTemplate($orderId);
            } else {
                throw new ParameterMissingException('returnShipmentId');
            }

            $this->View()->assign([
                'success' => true,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * @param mixed $orderId
     * @return array
     * @throws EntityNotFoundException
     */
    private function getReturnShipmentTemplate($orderId)
    {
        ParameterValidator::assertIsInteger($orderId, 'orderId');

        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->get('models')->createQueryBuilder();
        $queryBuilder
            ->select(
                '_order',
                'payment',
                'customer',
                'paymentStatus',
                'customerGroup',
                'customerDefaultBillingAddress'
            )
            ->from(Order::class, '_order')
            ->leftJoin('_order.customer', 'customer')
            ->leftJoin('_order.payment', 'payment')
            ->leftJoin('_order.paymentStatus', 'paymentStatus')
            ->leftJoin('customer.group', 'customerGroup')
            ->leftJoin('customer.defaultBillingAddress', 'customerDefaultBillingAddress')
            ->andWhere('_order.id = :orderId')
            ->setParameter('orderId', $orderId);

        $order = $queryBuilder->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);
        if (!$order) {
            throw new EntityNotFoundException('orderId', Order::class, $orderId);
        }

        /** @var Warehouse $defaultWarehouse */
        $defaultWarehouse = $this->get('models')->getRepository(Warehouse::class)->getDefaultReturnShipmentWarehouse();
        /** @var ReturnShipmentStatus $initialStatus */
        $initialStatus = $this->get('models')->getRepository(ReturnShipmentStatus::class)->getInitialReturnShipmentStatus();

        $returnShipment = [
            'id' => null,
            'number' => '',
            'statusId' => $initialStatus->getId(),
            'targetWarehouseId' => $defaultWarehouse->getId(),
            'documentId' => null,
            'items' => [],
            'attachments' => [],
            'internalComments' => [],
        ];

        $order['paymentMethodName'] = $order['payment']['description'];
        unset($order['payment']);
        $returnShipment['customer'] = [
            'id' => $order['customer']['id'],
            'number' => $order['customer']['number'],
            'name' => $order['customer']['firstname'] . ' ' . $order['customer']['lastname'],
            'email' => $order['customer']['email'],
            'groupName' => $order['customer']['group']['name'],
            'phone' => $order['customer']['defaultBillingAddress']['phone'],
            'company' => $order['customer']['defaultBillingAddress']['company'],
        ];
        unset($order['customer']);
        $returnShipment['order'] = $order;

        return $returnShipment;
    }

    /**
     * @param mixed $returnShipmentId
     * @return array
     * @throws EntityNotFoundException
     */
    private function getExistingReturnShipment($returnShipmentId)
    {
        ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');

        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->get('models')->createQueryBuilder();
        $queryBuilder
            ->select(
                'returnShipment',
                'item',
                '_order',
                'payment',
                'customer',
                'paymentStatus',
                'internalComment',
                'customerGroup',
                'customerDefaultBillingAddress',
                'orderDetail',
                'attachment',
                'media',
                'commentUser',
                'document'
            )
            ->from(ReturnShipment::class, 'returnShipment')
            ->leftJoin('returnShipment.items', 'item')
            ->leftJoin('returnShipment.order', '_order')
            ->leftJoin('_order.customer', 'customer')
            ->leftJoin('_order.payment', 'payment')
            ->leftJoin('_order.paymentStatus', 'paymentStatus')
            ->leftJoin('returnShipment.internalComments', 'internalComment')
            ->leftJoin('customer.group', 'customerGroup')
            ->leftJoin('customer.defaultBillingAddress', 'customerDefaultBillingAddress')
            ->leftJoin('item.orderDetail', 'orderDetail')
            ->leftJoin('returnShipment.attachments', 'attachment')
            ->leftJoin('attachment.media', 'media')
            ->leftJoin('internalComment.user', 'commentUser')
            ->leftJoin('returnShipment.document', 'document')
            ->andWhere('returnShipment.id = :returnShipmentId')
            ->setParameter('returnShipmentId', $returnShipmentId);

        $returnShipment = $queryBuilder->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);
        if (!$returnShipment) {
            throw new EntityNotFoundException('returnShipmentId', ReturnShipment::class, $returnShipmentId);
        }

        $returnShipment['documentId'] = $returnShipment['document'] ? $returnShipment['document']['id'] : null;
        $returnShipment['order']['paymentMethodName'] = $returnShipment['order']['payment']['description'];
        unset($returnShipment['order']['payment']);
        $returnShipment['customer'] = [
            'id' => $returnShipment['order']['customer']['id'],
            'number' => $returnShipment['order']['customer']['number'],
            'name' => $returnShipment['order']['customer']['firstname'] . ' ' . $returnShipment['order']['customer']['lastname'],
            'email' => $returnShipment['order']['customer']['email'],
            'groupName' => $returnShipment['order']['customer']['group']['name'],
            'phone' => $returnShipment['order']['customer']['defaultBillingAddress']['phone'],
            'company' => $returnShipment['order']['customer']['defaultBillingAddress']['company'],
        ];
        unset($returnShipment['order']['customer']);
        $returnShipment['items'] = array_map(function ($returnShipmentItem) {
            $returnShipmentItem['articleName'] = $returnShipmentItem['orderDetail']['articleName'];
            $returnShipmentItem['articleNumber'] = $returnShipmentItem['orderDetail']['articleNumber'];
            $returnShipmentItem['articleId'] = $returnShipmentItem['orderDetail']['articleId'];
            $returnShipmentItem['orderDetailId'] = $returnShipmentItem['orderDetail']['id'];
            $returnShipmentItem['shippedQuantity'] = $this->calculateShippedAndNotReturnedQuantity($returnShipmentItem['orderDetail']['id']) + $returnShipmentItem['returnedQuantity']; // Todo: rename shipped quantity, as this is totally not the shipped quantity but the maximum possible value for returnedQuantity
            unset($returnShipmentItem['orderDetail']);

            return $returnShipmentItem;
        }, $returnShipment['items']);
        $returnShipment['internalComments'] = array_map(function ($internalComment) {
            $internalComment['userName'] = $internalComment['user']['name'];
            unset($internalComment['user']);

            return $internalComment;
        }, $returnShipment['internalComments']);

        usort($returnShipment['internalComments'], function ($commentLh, $commentRh) {
            return $commentLh['created']->getTimestamp() - $commentRh['created']->getTimestamp();
        });

        $this->addThumbnailUrlsToReturnShipment($returnShipment);

        return $returnShipment;
    }

    /**
     * @param int $orderDetailId
     * @return int
     */
    private function calculateShippedAndNotReturnedQuantity($orderDetailId)
    {
        /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculatorService */
        $orderDetailQuantityCalculatorService = $this->get('pickware.erp.order_detail_quantity_calculator_service');
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var OrderDetail $orderDetail */
        $orderDetail = $entityManager->find(OrderDetail::class, $orderDetailId);

        return $orderDetailQuantityCalculatorService->calculateShippedAndNotReturnedQuantity($orderDetail);
    }

    /**
     * Adds the thumbnail URL to every return attachment with a media that has a thumbnail.
     *
     * @param array $returnShipment
     */
    private function addThumbnailUrlsToReturnShipment(array &$returnShipment)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        if (!is_array($returnShipment['attachments'])) {
            return;
        }
        // Add a thumbnail path to the returnShipment item if possible
        foreach ($returnShipment['attachments'] as &$attachment) {
            if (!is_array($attachment['media']) || $attachment['media']['type'] !== Media::TYPE_IMAGE) {
                continue;
            }
            /** @var Media $media */
            $media = $entityManager->find(Media::class, $attachment['media']['id']);
            if ($media) {
                $attachment['media']['thumbnail'] = $this->getThumbnailUrlForMedia($media);
            }
        }
    }

    /**
     * @param Media $media
     * @return string|null
     */
    private function getThumbnailUrlForMedia(Media $media)
    {
        /** @var MediaService $mediaService */
        $mediaService = $this->get('shopware_media.media_service');
        $thumbnails = $media->getThumbnails();
        if (isset($thumbnails['140x140']) && $mediaService->has($thumbnails['140x140'])) {
            return $mediaService->getUrl($thumbnails['140x140']);
        }

        return null;
    }

    public function getReturnShipmentItemCandidatesAction()
    {
        try {
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');
            /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculatorService */
            $orderDetailQuantityCalculatorService = $this->get('pickware.erp.order_detail_quantity_calculator_service');

            $orderId = $this->Request()->getParam('orderId', null);
            ParameterValidator::assertIsInteger($orderId, 'orderId');
            /** @var Order $order */
            $order = $entityManager->find(Order::class, $orderId);
            $orderDetails = $order->getDetails();

            $data = [];
            /** @var OrderDetail $orderDetail */
            foreach ($orderDetails as $orderDetail) {
                $shippedAndNotReturnedQuantity = $orderDetailQuantityCalculatorService->calculateShippedAndNotReturnedQuantity($orderDetail);
                if ($shippedAndNotReturnedQuantity === 0) {
                    continue;
                }
                $data[] = [
                    'articleId' => $orderDetail->getArticleId(),
                    'orderDetailId' => $orderDetail->getId(),
                    'articleNumber' => $orderDetail->getArticleNumber(),
                    'articleName' => $orderDetail->getArticleName(),
                    'shippedQuantity' => $shippedAndNotReturnedQuantity,
                    'returnedQuantity' => 0,
                    'writtenOffQuantity' => 0,
                ];
            }

            $this->View()->assign([
                'success' => true,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function updateReturnShipmentAction()
    {
        try {
            $returnShipmentData = $this->Request()->getParam('data', null);

            $this->validateReturnShipmentData($returnShipmentData);
            $returnShipment = $this->createOrUpdateReturnShipment($returnShipmentData);

            $this->Request()->setParam('returnShipmentId', $returnShipment->getId());
            $this->getReturnShipmentAction();
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * @param mixed $returnShipmentData
     * @throws CustomValidationException
     */
    private function validateReturnShipmentData($returnShipmentData)
    {
        ParameterValidator::assertIsArray($returnShipmentData, 'data');
        $treePosition = 'data.';

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $order = null;
        if (isset($returnShipmentData['id'])) {
            ParameterValidator::assertIsInteger($returnShipmentData['id'], $treePosition . 'id');
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentData['id']);
            ParameterValidator::assertEntityFound($returnShipment, ReturnShipment::class, $returnShipmentData['id'], $treePosition . 'id');
            $order = $returnShipment->getOrder();
        } else {
            ParameterValidator::assertArrayElementIsArray($returnShipmentData, 'order', $treePosition . 'order');
            ParameterValidator::assertArrayElementIsInteger($returnShipmentData['order'], 'id', $treePosition . 'order.id');
            /** @var Order $order */
            $order = $entityManager->find(Order::class, $returnShipmentData['order']['id']);
            ParameterValidator::assertEntityFound($order, Order::class, $returnShipmentData['order']['id'], $treePosition . 'order.id');
        }

        if (isset($returnShipmentData['targetWarehouseId'])) {
            ParameterValidator::assertIsInteger($returnShipmentData['targetWarehouseId'], $treePosition . 'targetWarehouseId');
            /** @var Warehouse $targetWarehouse */
            $targetWarehouse = $entityManager->find(Warehouse::class, $returnShipmentData['targetWarehouseId']);
            ParameterValidator::assertEntityFound(
                $targetWarehouse,
                Warehouse::class,
                $returnShipmentData['targetWarehouseId'],
                $treePosition . 'targetWarehouseId'
            );
        }

        if (isset($returnShipmentData['statusId'])) {
            ParameterValidator::assertIsInteger($returnShipmentData['statusId'], $treePosition . 'statusId');
            /** @var ReturnShipmentStatus $status */
            $status = $entityManager->find(ReturnShipmentStatus::class, $returnShipmentData['statusId']);
            ParameterValidator::assertEntityFound(
                $status,
                ReturnShipmentStatus::class,
                $returnShipmentData['statusId'],
                $treePosition . 'statusId'
            );
        }

        if (isset($returnShipmentData['items'])) {
            ParameterValidator::assertIsArray($returnShipmentData['items'], $treePosition . 'items');

            foreach ($returnShipmentData['items'] as $returnShipmentItemKey => $returnShipmentItemData) {
                $treePosition = 'data.items[' . $returnShipmentItemKey . ']';
                ParameterValidator::assertIsArray($returnShipmentItemData, $treePosition);
                $treePosition .= '.';

                if (isset($returnShipmentItemData['id'])) {
                    ParameterValidator::assertIsInteger($returnShipmentItemData['id'], $treePosition . 'id');
                    /** @var ReturnShipmentItem $item */
                    $item = $entityManager->find(ReturnShipmentItem::class, $returnShipmentItemData['id']);
                    ParameterValidator::assertEntityFound($item, ReturnShipmentItem::class, $returnShipmentItemData['id'], $treePosition . 'id');

                    if (!$returnShipmentData['id'] || $item->getReturnShipment()->getId() !== intval($returnShipmentData['id'])) {
                        throw new CustomValidationException($treePosition . 'id', $returnShipmentItemData['id'], 'The ReturnShipmentItem (given by named id) does not belong to the given ReturnShipment');
                    }
                } else {
                    ParameterValidator::assertIsInteger($returnShipmentItemData['orderDetailId'], $treePosition . 'orderDetailId');
                    /** @var OrderDetail $orderDetail */
                    $orderDetail = $entityManager->find(OrderDetail::class, $returnShipmentItemData['orderDetailId']);
                    ParameterValidator::assertEntityFound(
                        $orderDetail,
                        OrderDetail::class,
                        $returnShipmentItemData['orderDetailId'],
                        $treePosition . 'orderDetailId'
                    );

                    if ($orderDetail->getOrder()->getId() !== $order->getId()) {
                        throw new CustomValidationException($treePosition . 'orderDetailId', $returnShipmentItemData['orderDetailId'], 'The OrderDetail (given by named id) does not belong to Order the given ReturnShipment');
                    }
                }

                if (isset($returnShipmentItemData['returnedQuantity'])) {
                    ParameterValidator::assertIsInteger($returnShipmentItemData['returnedQuantity'], $treePosition . 'returnedQuantity');
                }

                if (isset($returnShipmentItemData['writtenOffQuantity'])) {
                    ParameterValidator::assertIsInteger($returnShipmentItemData['writtenOffQuantity'], $treePosition . 'writtenOffQuantity');
                }
            }
        }

        if (isset($returnShipmentData['attachments'])) {
            ParameterValidator::assertIsArray($returnShipmentData['attachments'], 'data.attachments');

            foreach ($returnShipmentData['attachments'] as $attachmentKey => $attachmentData) {
                $treePosition = 'data.attachments[' . $attachmentKey . ']';
                ParameterValidator::assertIsArray($attachmentData, $treePosition);
                $treePosition .= '.';

                if (isset($attachmentData['id'])) {
                    ParameterValidator::assertIsInteger($attachmentData['id'], $treePosition . 'id');
                    /** @var ReturnShipmentItem $item */
                    $attachment = $entityManager->find(ReturnShipmentAttachment::class, $attachmentData['id']);
                    ParameterValidator::assertEntityFound($attachment, ReturnShipmentAttachment::class, $attachmentData['id'], $treePosition . 'id');
                }

                ParameterValidator::assertIsInteger($attachmentData['mediaId'], $treePosition . 'mediaId');
                /** @var Media $media */
                $media = $entityManager->find(Media::class, $attachmentData['mediaId']);
                ParameterValidator::assertEntityFound($media, Order::class, $attachmentData['mediaId'], $treePosition . 'mediaId');
            }
        }

        if (isset($returnShipmentData['internalComments'])) {
            ParameterValidator::assertIsArray($returnShipmentData['attachments'], 'data.internalComments');

            foreach ($returnShipmentData['internalComments'] as $internalCommentKey => $internalCommentData) {
                $treePosition = 'data.internalComments[' . $internalCommentKey . ']';
                ParameterValidator::assertIsArray($internalCommentData, $treePosition);
                $treePosition .= '.';

                ParameterValidator::assertArrayElementIsNotNull($internalCommentData, 'comment', $treePosition . 'comment');
            }
        }
    }

    /**
     * @param array $returnShipmentData
     * @return ReturnShipment
     * @throws ReturnShipmentException
     */
    private function createOrUpdateReturnShipment(array $returnShipmentData)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var ReturnShipmentProcessor $returnShipmentProcessor */
        $returnShipmentProcessor = $this->get('pickware.erp.return_shipment_processor_service');

        // Update the ReturnShipment itself
        if (isset($returnShipmentData['id'])) {
            $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentData['id']);
        } else {
            /** @var Order $order */
            $order = $entityManager->find(Order::class, $returnShipmentData['order']['id']);
            $returnShipment = $returnShipmentProcessor->createReturnShipment($order);
            $returnShipment->setUser(ViisonCommonUtil::getCurrentUser());
            $entityManager->persist($returnShipment);
        }
        if (isset($returnShipmentData['targetWarehouseId'])
            && ($returnShipment->getTargetWarehouse() === null || $returnShipment->getTargetWarehouse()->getId() !== $returnShipmentData['targetWarehouseId'])
        ) {
            $totalReturned = array_reduce($returnShipment->getItems()->toArray(), function ($totalReturned, ReturnShipmentItem $item) {
                return $totalReturned + $item->getReturnedQuantity();
            }, 0);
            if ($totalReturned > 0) {
                throw ReturnShipmentException::targetWarehouseImmutable($returnShipment);
            }
            /** @var Warehouse $targetWarehouse */
            $targetWarehouse = $entityManager->find(Warehouse::class, $returnShipmentData['targetWarehouseId']);
            $returnShipment->setTargetWarehouse($targetWarehouse);
        }
        if (isset($returnShipmentData['statusId'])) {
            /** @var ReturnShipmentStatus $status */
            $status = $entityManager->find(ReturnShipmentStatus::class, $returnShipmentData['statusId']);
            $returnShipment->setStatus($status);
        }
        $entityManager->flush($returnShipment);

        // Update its items
        $itemIdsToKeep = [];
        foreach ($returnShipmentData['items'] as $returnShipmentItemData) {
            if ($returnShipmentItemData['id']) {
                /** @var ReturnShipmentItem $item */
                $item = $entityManager->find(ReturnShipmentItem::class, $returnShipmentItemData['id']);
            } else {
                /** @var OrderDetail $orderDetail */
                $orderDetail = $entityManager->find(OrderDetail::class, $returnShipmentItemData['orderDetailId']);
                $item = new ReturnShipmentItem($returnShipment, $orderDetail);
                $entityManager->persist($item);
            }

            $newReturned = $item->getReturnedQuantity();
            if (isset($returnShipmentItemData['returnedQuantity'])) {
                $newReturned = $returnShipmentItemData['returnedQuantity'];
            }
            $newWrittenOff = $item->getWrittenOffQuantity();
            if (isset($returnShipmentItemData['writtenOffQuantity'])) {
                $newWrittenOff = $returnShipmentItemData['writtenOffQuantity'];
            }

            $validValues = $returnShipmentProcessor->isReturnedAndWrittenOffQuantityAllowed($item, $newReturned, $newWrittenOff);
            if (!$validValues) {
                throw ReturnShipmentException::returnedQuantityOrWrittenOffQuantityNotAllowed($newReturned, $newWrittenOff, $item);
            }

            $item->setReturnedQuantity($newReturned);
            $item->setWrittenOffQuantity($newWrittenOff);
            $entityManager->flush($item);

            $returnShipmentProcessor->writeStockEntriesForItemChangesWithAutomaticBinLocationSelection($item);

            $itemIdsToKeep[] = $item->getId();
        }
        foreach ($returnShipment->getItems() as $item) {
            if (!in_array($item->getId(), $itemIdsToKeep, true)) {
                $returnShipmentProcessor->safelyDeleteReturnShipmentItem($item);
            }
        }

        // Update its attachments
        $affectedAttachments = [];
        $idsOfAttachmentsToKeep = [];
        foreach ($returnShipmentData['attachments'] as $attachmentData) {
            if ($attachmentData['id']) {
                $idsOfAttachmentsToKeep[] = intval($attachmentData['id']);
            } else {
                /** @var Media $media */
                $media = $entityManager->find(Media::class, $attachmentData['mediaId']);

                $attachment = new ReturnShipmentAttachment($returnShipment, $media);
                $entityManager->persist($attachment);

                $idsOfAttachmentsToKeep[] = $attachment->getId();
                $affectedAttachments[] = $attachment;
            }
        }
        /** @var ReturnShipmentAttachment $attachment */
        foreach ($returnShipment->getAttachments() as $attachment) {
            if (!in_array($attachment->getId(), $idsOfAttachmentsToKeep, true)) {
                $returnShipment->removeAttachment($attachment);
                $entityManager->remove($attachment);
                $affectedAttachments[] = $attachment;
            }
        }
        $entityManager->flush($affectedAttachments);

        // Update its comments
        $affectedInternalComments = [];
        $idsOfInternalCommentsToKeep = [];
        foreach ($returnShipmentData['internalComments'] as $internalCommentData) {
            if ($internalCommentData['id']) {
                $idsOfInternalCommentsToKeep[] = intval($internalCommentData['id']);
            } else {
                $internalComment = new ReturnShipmentInternalComment($returnShipment);
                $internalComment->setComment($internalCommentData['comment']);
                $internalComment->setUser(ViisonCommonUtil::getCurrentUser());
                $entityManager->persist($internalComment);

                $idsOfInternalCommentsToKeep[] = $internalComment->getId();
                $affectedInternalComments[] = $internalComment;
            }
        }
        /** @var ReturnShipmentInternalComment $internalComment */
        foreach ($returnShipment->getInternalComments() as $internalComment) {
            if (!in_array($internalComment->getId(), $idsOfInternalCommentsToKeep, true)) {
                $returnShipment->removeInternalComment($internalComment);
                $entityManager->remove($internalComment);
                $affectedInternalComments[] = $internalComment;
            }
        }
        $entityManager->flush($affectedInternalComments);

        // Automatically move status from NEW to RECEIVED if at least one item has been returned
        $totalReturned = array_reduce($returnShipment->getItems()->toArray(), function ($totalReturned, ReturnShipmentItem $item) {
            return $totalReturned + $item->getReturnedQuantity();
        }, 0);
        if ($totalReturned > 0) {
            /** @var ReturnShipmentStatus $statusReceived */
            $statusReceived = $entityManager->find(ReturnShipmentStatus::class, ReturnShipmentStatus::STATUS_RECEIVED_ID);
            $oldStatus = $returnShipment->getStatus();
            $returnShipment->setStatusToAtLeast($statusReceived);
            $newStatus = $returnShipment->getStatus();
            if ($oldStatus->getId() !== $newStatus->getId()) {
                try {
                    /** @var ReturnShipmentMailingService $returnShipmentMailingService */
                    $returnShipmentMailingService = $this->get('pickware.erp.return_shipment_mailing_service');

                    $mail = $returnShipmentMailingService->createReturnReceivedMail($returnShipment);
                    $mailData = self::createArrayFromMail($mail);
                    $mailData['mailType'] = self::MAIL_TYPE_RETURN_SHIPMENT_RECEIVED;

                    $this->View()->assign('mail', $mailData);
                } catch (ReturnShipmentMailingException $e) {
                    $this->View()->assign('mail', null);
                }
            }
        } else {
            /** @var ReturnShipmentStatus $statusNew */
            $statusNew = $entityManager->find(ReturnShipmentStatus::class, ReturnShipmentStatus::STATUS_NEW_ID);
            $returnShipment->setStatusToAtMost($statusNew);
        }
        $entityManager->flush($returnShipment);
        $returnShipmentProcessor->updateAccumulatedReturnShipmentStatus($returnShipment->getOrder());

        return $returnShipment;
    }

    public function finalizeReturnShipmentAction()
    {
        try {
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');

            $returnShipmentId = $this->Request()->getParam('returnShipmentId', null);
            ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentId);
            ParameterValidator::assertEntityFound(
                $returnShipment,
                ReturnShipment::class,
                $returnShipmentId,
                'returnShipmentId'
            );

            /** @var ReturnShipmentStatus $statusCompleted */
            $statusCompleted = $entityManager->getRepository(ReturnShipmentStatus::class)->findOneBy([
                'name' => ReturnShipmentStatus::STATUS_COMPLETED_NAME,
            ]);
            $returnShipment->setStatusToAtLeast($statusCompleted);
            $entityManager->flush($returnShipment);

            /** @var ReturnShipmentProcessor $returnShipmentProcessor */
            $returnShipmentProcessor = $this->get('pickware.erp.return_shipment_processor_service');
            $returnShipmentProcessor->updateAccumulatedReturnShipmentStatus($returnShipment->getOrder());

            $this->getReturnShipmentAction();

            try {
                /** @var ReturnShipmentMailingService $returnShipmentMailingService */
                $returnShipmentMailingService = $this->get('pickware.erp.return_shipment_mailing_service');

                $mail = $returnShipmentMailingService->createReturnCompletedMail($returnShipment);
                $mailData = self::createArrayFromMail($mail);
                $mailData['mailType'] = self::MAIL_TYPE_RETURN_SHIPMENT_FINALIZED;

                $this->View()->assign('mail', $mailData);
            } catch (ReturnShipmentMailingException $e) {
                $this->View()->assign('mail', null);
            }
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function addAttachmentAction()
    {
        try {
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');

            $returnShipmentId = $this->Request()->getParam('returnShipmentId', null);
            ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentId);
            ParameterValidator::assertEntityFound(
                $returnShipment,
                ReturnShipment::class,
                $returnShipmentId,
                'returnShipmentId'
            );

            $mediaId = $this->Request()->getParam('mediaId', null);
            ParameterValidator::assertIsInteger($mediaId, 'mediaId');
            /** @var Media $media */
            $media = $entityManager->find(Media::class, $mediaId);
            ParameterValidator::assertEntityFound($media, Media::class, $mediaId, 'mediaId');

            $attachment = new ReturnShipmentAttachment($returnShipment, $media);
            $entityManager->persist($attachment);
            $entityManager->flush($attachment);

            $data = [
                'id' => $attachment->getId(),
                'mediaId' => $media->getId(),
                'fileName' => $media->getFileName(),
                'filePath' => $media->getPath(),
                'type' => $media->getType(),
                'extension' => $media->getExtension(),
                'thumbnail' => $this->getThumbnailUrlForMedia($media),
            ];

            $this->View()->assign([
                'success' => true,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function batchFinalizeReturnShipmentsAction()
    {
        try {
            $returnShipmentIds = json_decode($this->Request()->getParam('returnShipmentIds', '[]'));
            $shouldSendStatusMails = $this->Request()->getParam('shouldSendStatusMails', 'true') == 'true';
            ParameterValidator::assertIsArray($returnShipmentIds, 'returnShipmentIds');

            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');

            /** @var ReturnShipment[] $returnShipments */
            $returnShipments = [];
            $orders = [];
            foreach ($returnShipmentIds as $returnShipmentKey => $returnShipmentId) {
                /** @var ReturnShipment $returnShipment */
                $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentId);
                ParameterValidator::assertEntityFound(
                    $returnShipment,
                    ReturnShipment::class,
                    $returnShipmentId,
                    'returnShipmentIds[' . $returnShipmentKey . ']'
                );
                $returnShipments[] = $returnShipment;
                $order = $returnShipment->getOrder();
                $orders[$order->getId()] = $order;
            }

            /** @var ReturnShipmentMailingService $returnShipmentMailingService */
            $returnShipmentMailingService = $this->get('pickware.erp.return_shipment_mailing_service');
            /** @var ReturnShipmentStatus $statusCompleted */
            $statusCompleted = $entityManager->find(ReturnShipmentStatus::class, ReturnShipmentStatus::STATUS_COMPLETED_ID);
            foreach ($returnShipments as $returnShipment) {
                $oldStatus = $returnShipment->getStatus();
                $returnShipment->setStatusToAtLeast($statusCompleted);
                $newStatus = $returnShipment->getStatus();
                if ($oldStatus->getId() !== $newStatus->getId() && $shouldSendStatusMails) {
                    try {
                        $returnShipmentMailingService->sendReturnCompletedEmail($returnShipment);
                    } catch (ReturnShipmentMailingException $e) {
                        $this->get('viison_common.logger')->error(
                            'Failed sending a mail during batch processing (completion) of return shipments',
                            [
                                'returnShipment' => [
                                    'id' => $returnShipment->getId(),
                                    'number' => $returnShipment->getNumber(),
                                ],
                                'exception' => ViisonCommonUtil::exceptionToArray($e),
                            ]
                        );
                    }
                }
            }
            $entityManager->flush($returnShipments);

            /** @var ReturnShipmentProcessor $returnShipmentProcessor */
            $returnShipmentProcessor = $this->get('pickware.erp.return_shipment_processor_service');
            foreach ($orders as $order) {
                $returnShipmentProcessor->updateAccumulatedReturnShipmentStatus($order);
            }

            $this->View()->assign([
                'success' => true,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function sendMailAction()
    {
        try {
            $returnShipmentId = (int) $this->Request()->getParam('returnShipmentId', null);
            ParameterValidator::assertIsInteger($returnShipmentId, 'returnShipmentId');
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');
            /** @var ReturnShipment $returnShipment */
            $returnShipment = $entityManager->find(ReturnShipment::class, $returnShipmentId);
            ParameterValidator::assertEntityFound(
                $returnShipment,
                ReturnShipment::class,
                $returnShipmentId,
                'returnShipmentId'
            );

            /** @var ReturnShipmentMailingService $returnShipmentMailingService */
            $returnShipmentMailingService = $this->get('pickware.erp.return_shipment_mailing_service');

            $mailData = json_decode($this->Request()->getParam('mailData', '{}'), true);
            ParameterValidator::assertIsArray($mailData, 'mailData');
            $mailType = $this->Request()->getParam('mailType', null);
            switch ($mailType) {
                case self::MAIL_TYPE_RETURN_SHIPMENT_FINALIZED:
                    $mail = $returnShipmentMailingService->createReturnCompletedMail($returnShipment);
                    break;
                case self::MAIL_TYPE_RETURN_SHIPMENT_RECEIVED:
                    $mail = $returnShipmentMailingService->createReturnReceivedMail($returnShipment);
                    break;
                default:
                    throw new ParameterHasWrongFormatException('mailType', sprintf(
                        '"%s"|"%s"',
                        self::MAIL_TYPE_RETURN_SHIPMENT_RECEIVED,
                        self::MAIL_TYPE_RETURN_SHIPMENT_FINALIZED
                    ));
            }

            if (isset($mailData['toAddress'])) {
                $mail->clearHeader('To');
                $mail->addTo($mailData['toAddress']);
            }
            if (isset($mailData['subject'])) {
                $mail->clearSubject();
                $mail->setSubject($mailData['subject']);
            }
            if (isset($mailData['content']) && !$mailData['isHtml']) {
                $mail->setBodyText($mailData['content']);
            }
            if (isset($mailData['contentHtml']) && $mailData['isHtml']) {
                $mail->setBodyHtml($mailData['contentHtml']);
            }

            $mail->send();

            $this->View()->assign([
                'success' => true,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * @param \Enlight_Components_Mail $mail
     * @return array
     */
    private static function createArrayFromMail(\Enlight_Components_Mail $mail)
    {
        $mailData = [
            'toAddress' => $mail->getTo()[0],
            'content' => $mail->getPlainBodyText(),
            'contentHtml' => $mail->getPlainBody(),
            'subject' => $mail->getPlainSubject(),
            'isHtml' => $mail->getPlainBody() !== null,
            'fromMail' => $mail->getFrom(),
            'fromName' => $mail->getFromName(),
        ];

        if ($mail->hasAttachments) {
            $mailData['attachment'] = implode(', ', array_map(function (Zend_Mime_Part $attachment) {
                return $attachment->filename;
            }, $mail->getParts()));
        }

        return $mailData;
    }
}
