<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\Components\CSRFWhitelistAware;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Util\Localization as LocalizationUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockInitialization;

class Shopware_Controllers_Backend_ViisonPickwareERPStockInitialization extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    use BackendExceptionHandling;
    /**
     * Disables the JSON renderer for getDescriptionHtml requests.
     */
    public function preDispatch()
    {
        if ($this->Request()->getActionName() !== 'getDescriptionHtml') {
            parent::preDispatch();
        }
    }

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'getDescriptionHtml'
        ];
    }

    /**
     * Returns the (translated) description of the stock initializing process (html document).
     *
     * @throws Exception
     */
    public function getDescriptionHtmlAction()
    {
        $backendLocaleString = $this->get('auth')->getIdentity()->locale->getLocale();
        // Shorten the locale to use more general language code (e.g. 'en_GB' -> 'en')
        $backendLocaleString = LocalizationUtil::getLanguageCode($backendLocaleString);

        $templateFile = sprintf(
            'backend/viison_pickware_erp_stock_initialization/iframe/description.%s.tpl',
            mb_strtolower($backendLocaleString)
        );

        if (file_exists(__DIR__ . '/../../Views/' . $templateFile)) {
            $this->View()->loadTemplate($templateFile);
        } else {
            // Use english template as fallback
            $this->View()->loadTemplate('backend/viison_pickware_erp_stock_initialization/iframe/description.en.tpl');
        }

        $this->View()->assign(
            'purchasePriceMode',
            $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->get('purchasePriceMode')
        );
    }

    /**
     * Creates a stocktake for the requested article
     */
    public function createStockEntryAction()
    {
        $articleDetailIds = $this->Request()->getParam('detailIds', []);
        $articleDetails = $this->get('models')->getRepository(ArticleDetail::class)->findBy([
            'id' => $articleDetailIds,
        ]);
        $this->get('pickware.erp.stock_initialization_service')->initializeStocksOfArticleDetails($articleDetails);
    }

    /**
     * Fetches and responds all articles, which don't have a stock entry yet.
     */
    public function getArticlesAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);

        $filterStr = $this->Request()->getParam('filterStr', null);
        $sort = $this->Request()->getParam('sort', []);

        if (isset($sort[0])) {
            $orderColumn = $sort[0]['property'];
            $orderDirection = $sort[0]['direction'];
        } else {
            $orderColumn = '';
            $orderDirection = '';
        }

        $inventory = $this->get('pickware.erp.stock_initialization_service')->getPaginatedUninitializedArticleDetails(
            $filterStr,
            $orderColumn,
            $orderDirection,
            $start,
            $limit
        );

        // Add the additional text
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($inventory['data'], 'detailsId'));
        foreach ($inventory['data'] as &$article) {
            $article['additionalText'] = $additionalTexts[$article['detailsId']];
        }

        $this->View()->success = true;
        $this->View()->assign($inventory);
    }

    /**
     * Fetches and initializes the first batch of articles, which don't have a stock entry yet.
     */
    public function initializeStocksForABatchOfUninitializedArticlesAction()
    {
        try {
            /** @var StockInitialization $stockInitialization */
            $stockInitialization = $this->get('pickware.erp.stock_initialization_service');
            $pendingUninitializedStocks = $stockInitialization->initializeStocksForABatchOfUninitializedArticleDetails();
            $this->View()->assign([
                'success' => true,
                'pending' => $pendingUninitializedStocks,
            ]);
        } catch (Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * Updates instock and price, is used by the row editor
     */
    public function updateArticleAction()
    {
        // Try to find article article and price
        $detailsId = $this->Request()->getParam('detailsId');
        $article = Shopware()->Models()->getRepository(ArticleDetail::class)->findOneById($detailsId);
        if ($article === null) {
            $this->View()->assign([
                'success' => false,
            ]);

            return;
        }

        // Update instock and price
        $instock = $this->Request()->getParam('instock');
        $article->setInStock($instock);

        $purchasePrice = $this->Request()->getParam('purchasePrice');
        $article->setPurchasePrice($purchasePrice);
        Shopware()->Models()->persist($article);
        Shopware()->Models()->flush();

        $this->View()->assign([
            'success' => true,
        ]);
    }
}
