<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Status as OrderStatus;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockInitializationService;

class Shopware_Controllers_Backend_ViisonPickwareERPStockOverview extends ViisonCommonBaseController
{
    /**
     * @return array
     */
    public function getViewParams()
    {
        return [
            'isViisonPickwareMobileInstalled' => ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile'),
        ];
    }

    /**
     * Returns a selection query for the amount of ordered but not sent articles using GREATEST function, so we never
     * get negative values for quantity-shipped, this could occur when an order position gets canceled but the
     * merchandise won't be added to the stock again (broken article). We also deduct any shipped quantity if
     * ViisonPickwareMobile is installed.
     *
     * @return string
     */
    public static function getSelectOpenSumQuery()
    {
        if (ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile')) {
            $pickedQuantitySelection = '- s_order_details_attributes.pickware_picked_quantity';
        }
        $selectOpenSumQuery = sprintf(
            'SUM(CASE
                # Handles open orders and their details created after initialization
                WHEN s_order.ordertime >= s_articles_attributes.pickware_stock_initialization_time
                    AND s_order.status != :stateAborted
                    THEN
                        GREATEST(0, s_order_details.quantity + IFNULL(derivedQuantities.cancelledReturned, 0) - s_order_details.shipped %1$s)

                # Handles open orders and their details created prior initialization
                WHEN s_order.ordertime < s_articles_attributes.pickware_stock_initialization_time
                    AND %2$s
                    THEN
                        GREATEST(0, s_order_details.quantity + IFNULL(derivedQuantities.cancelledReturned, 0) - s_order_details.shipped %1$s)
                ELSE
                    0
            END)',
            isset($pickedQuantitySelection) ? $pickedQuantitySelection : '',
            StockInitializationService::OPEN_ORDERS_STATUS_CONDITION
        );

        return $selectOpenSumQuery;
    }

    /**
     * Fetches and responds all articles, which don't have a stock entry yet.
     */
    public function getArticlesAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $filterStr = $this->Request()->getParam('filterStr', null);
        $filterStockProblems = $this->Request()->getParam('filterStockProblems');
        $planningFrom = $this->Request()->getParam('planningFrom');
        $planningFrom = !empty($planningFrom) ? new DateTime($planningFrom) : null;

        $planningTo = $this->Request()->getParam('planningTo');
        $planningTo = !empty($planningTo) ? new DateTime($planningTo) : null;

        // typecast string to bool ('true' | 'false')
        $filterStockProblems = 'true' == $filterStockProblems;

        $sort = $this->Request()->getParam('sort', []);

        if (isset($sort[0])) {
            $orderColumn = $sort[0]['property'];
            $orderDirection = $sort[0]['direction'];
        } else {
            $orderColumn = '';
            $orderDirection = '';
        }

        $articles = self::getArticles($start, $limit, $filterStr, $filterStockProblems, $orderColumn, $orderDirection, $planningFrom, $planningTo);

        $this->View()->assign([
            'success' => true,
            'data' => $articles['result'],
            'total' => $articles['count'],
        ]);
    }

    public function updateArticleAction()
    {
        $error = null;
        $detailId = $this->Request()->getParam('id');
        $instock = $this->Request()->getParam('instock', null);

        /**
         * @var Shopware\Models\Article\Detail $articleDetail
         */
        $articleDetail = Shopware()->Models()
            ->getRepository(ArticleDetail::class)
            ->findOneById($detailId);

        if ($instock !== null && is_numeric($instock)) {
            $articleDetail->setInStock($instock);
            Shopware()->Models()->persist($articleDetail);
            Shopware()->Models()->flush();
        }

        $this->View()->success = true;
    }

    /**
     * @throws Zend_Db_Statement_Exception
     */
    public function getOrdersAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);

        $detailId = $this->Request()->getParam('articleDetail', 0);

        $articleDetail = $this->get('models')->find(ArticleDetail::class, $detailId);
        if (!$articleDetail) {
            return;
        }

        $isViisonPickwareMobileInstalled = ViisonCommonUtil::isPluginInstalledAndActive(null, 'ViisonPickwareMobile');
        $orders = Shopware()->Db()->fetchAll(
            'SELECT
                SQL_CALC_FOUND_ROWS s_articles_details.id,
                s_order.ordernumber as orderNumber,
                s_order.id as id,
                s_order.ordertime as orderTime,
                s_core_states.name as status,
                SUM(s_order_details.quantity) as quantity,
                ' . (($isViisonPickwareMobileInstalled) ? 'SUM(s_order_details_attributes.pickware_picked_quantity) as picked,' : '') .
                'SUM(s_order_details.shipped) as shipped,
                '. self::getSelectOpenSumQuery() . ' as open
            FROM s_order
            LEFT JOIN s_order_details
                ON s_order_details.orderID = s_order.id
            LEFT JOIN s_articles_details
                ON s_articles_details.ordernumber = s_order_details.articleordernumber
            LEFT JOIN s_articles_attributes
                ON s_articles_attributes.articledetailsID = s_articles_details.id
            LEFT JOIN s_order_details_attributes
                ON s_order_details_attributes.detailID = s_order_details.id
            LEFT JOIN s_core_states
                ON s_core_states.id = s_order.status
            LEFT JOIN (
                SELECT
                    SUM(items.cancelledQuantity) AS cancelledReturned,
                    SUM(items.returnedQuantity) AS returned,
                    items.orderDetailId AS orderDetailId
                FROM pickware_erp_return_shipment_items as items
                GROUP BY items.orderDetailId
            ) AS derivedQuantities 
                ON derivedQuantities.orderDetailId = s_order_details.id
            WHERE s_articles_details.ordernumber = :orderNumber
            AND s_order.status != :stateAborted
            GROUP BY s_order.id
            HAVING open != 0
            LIMIT '.intval($start). ', '.intval($limit),
            [
                'orderNumber' => $articleDetail->getNumber(),
                'stateAborted' => OrderStatus::ORDER_STATE_CANCELLED,
            ]
        );

        $foundRows = Shopware()->Db()->fetchOne('SELECT FOUND_ROWS()');

        $this->View()->assign([
            'success' => true,
            'data' => $orders,
            'total' => $foundRows,
        ]);
    }

    /**
     * @throws Zend_Db_Statement_Exception
     */
    public function salesByDayAction()
    {
        $detailId = $this->Request()->getParam('articleDetail', null);
        if (!$detailId) {
            return;
        }

        $intervalFrom = new DateTime();
        $intervalFrom->sub(new DateInterval('P2Y'));
        $intervalTo = new DateTime();

        $sales = Shopware()->Db()->fetchAll(
            'SELECT
                DATE(`pickware_erp_stock_ledger_entries`.created) as date,
                ABS(SUM(CASE
                  WHEN `pickware_erp_stock_ledger_entries`.type = :stockSaleType
                    THEN `pickware_erp_stock_ledger_entries`.changeAmount
                    ELSE 0
                  END
                )) as sales,
                GROUP_CONCAT(`pickware_erp_stock_ledger_entries`.`newStock`) as `newStocks`,
                GROUP_CONCAT(`pickware_erp_stock_ledger_entries`.id) as stockIds
             FROM
                `pickware_erp_stock_ledger_entries`
             WHERE
                `pickware_erp_stock_ledger_entries`.articleDetailId = :detailId
                AND `pickware_erp_stock_ledger_entries`.created >= :salesIntervalFrom
                AND `pickware_erp_stock_ledger_entries`.created <= :salesIntervalTo
             # ordering by created so the group concat will be ordered
             GROUP BY
                DATE(`pickware_erp_stock_ledger_entries`.created)
            ',
            [
                'detailId' => $detailId,
                'stockSaleType' => StockLedgerEntry::TYPE_SALE,
                'salesIntervalFrom' => $intervalFrom->format(DateTime::ATOM),
                'salesIntervalTo' => $intervalTo->format(DateTime::ATOM),
            ]
        );

        //filter data last stock value per day
        $sales = array_map(
            function ($sale) {
                $stock = explode(',', $sale['newStocks']);
                $stockIds = explode(',', $sale['stockIds']);
                $idStockArray = array_combine($stockIds, $stock);
                $latestEntryId = max($stockIds);

                unset($sale['newStocks']);
                unset($sale['stockIds']);

                $sale['stock'] = $idStockArray[$latestEntryId]; // The last entry holds the latest stock information

                return $sale;
            },
            $sales
        );

        $this->View()->assign([
            'success' => true,
            'data' => $sales,
            'total' => count($sales),
        ]);
    }

    /**
     * @throws Zend_Db_Statement_Exception
     * @return []
     */
    public static function getArticles($start, $limit, $filterStr = '', $filterStockProblems = false, $orderBy = '', $direction = '', $planningFrom = null, $planningTo = null)
    {
        $sortableColumns = [
            'orderNumber',
            'articleName',
            'articleId',
            'quantity',
            'shipped',
            'stock',
            'shipped',
            'open',
            'not_sent',
            'stockmin',
            'supplier',
            'instock',
            'salesWithinPlanningInterval',
            'minimumStock',
            'targetStock',
        ];

        if ($planningFrom == null || $planningTo == null) {
            $planningFrom = new DateTime();
            $planningTo = new DateTime();
        }

        $whereClause = '';
        // Filter articles by a search string
        if (!empty($filterStr) && is_string($filterStr)) {
            $escapedFilter = Shopware()->Db()->quote('%' . $filterStr . '%');
            $whereClause = ' AND ( s_articles_details.orderNumber LIKE ' . $escapedFilter .' OR
                s_articles_details.additionalText LIKE ' . $escapedFilter .' OR
                s_articles.name LIKE ' . $escapedFilter .')';
        }

        $havingClause = '';
        if ($filterStockProblems == true) {
            $havingClause = 'HAVING
                not_sent != open
                OR shipped < 0
                OR not_sent < 0
                OR stock < 0';
        }

        // order the results
        // $filterStockProblems must not be set when sorting, query comment for reason
        if (!$filterStockProblems && in_array($orderBy, $sortableColumns)) {
            $identifier = Shopware()->Db()->quoteIdentifier($orderBy);
            if ($direction == 'ASC') {
                $sortDirection = 'ASC';
            } else {
                $sortDirection = 'DESC';
            }

            $orderQuery = 'ORDER BY ' . $identifier . ' ' . $sortDirection;
        } else {
            $orderQuery = '';
        }

        /*
         * WARNING: MySql bug - in some cases the ORDER BY statement causes the HAVING statement
         * to not work properly and therefore altering the result set.
         * It seems the MySql optimizer does some weird stuff when following conditions are met:
         *      1. aggregation with GROUP BY and a aggregation function with alias
         *          (like 'SUM(CASE ... THEN ...) as open' or 'SUM(rand() * column) as some_alias' )
         *          in which calculations are done
         *      2. A HAVING statement which filters the result set (filtering with the above alias ?)
         *      3. Some sources state that indices of the target tables also play a role
         * Affected MySql Versions: at least from 5.6.11 - 5.6.26
         *
         * http://dba.stackexchange.com/questions/100987/mysql-order-by-affecting-a-query-with-column-alias-group-by-and-having
         * http://bugs.mysql.com/bug.php?id=69638
         *
         * Solution: To avoid this issue sorting won't be possible when "filterStockProblems" is enabled
         */

        /*
         * This query returns a list of articles which have an initialized stock,
         * it aggregates values from s_order_details (quantity, shipped, and the derived 'open').
         *
         * The aggregation of 'open' (amount of articles which have not been shipped yet)
         * differentiates between orders before and after stock initialization,
         * orders prior initialization might not be handled by pickware (shipped might not be set),
         * therefore we can only look at orders which have the state open or partially shipped.
         *
         * 'not_sent' refers to (physical_stock_for_sale - instock).
         * Tt should equal to the amount of articles which were ordered but have not yet left the stock.
         *
         * If the stock is correct then 'open' should equal 'not_sent'.
         */

        $result = Shopware()->Db()->fetchAll(
            'SELECT
                SQL_CALC_FOUND_ROWS s_articles_details.id,
                s_articles_details.orderNumber AS orderNumber,
                s_articles.name AS articleName,
                s_articles.id AS articleId,
                ('. self::getSelectOpenSumQuery() .') AS open,
                IFNULL(SUM(s_order_details.quantity), 0) as quantity,
                IFNULL(SUM(s_order_details.shipped), 0) as shipped,
                s_articles_attributes.pickware_physical_stock_for_sale as stock,
                warehouse_stock_limits.minimumStock AS minimumStock,
                warehouse_stock_limits.targetStock AS targetStock,
                s_articles_details.instock AS instock,
                s_articles_supplier.name AS supplier,
                s_articles_details.id AS id,
                s_articles_details.purchaseprice AS purchasePrice,
                IFNULL(sales_join_table.salesWithinPlanningInterval, 0) AS salesWithinPlanningInterval,
                # not_sent is derived and should be the exact match of open (amount of unsent articles)
                (s_articles_attributes.pickware_physical_stock_for_sale - s_articles_details.instock) AS not_sent
            FROM s_articles_details
            LEFT JOIN s_order_details
                ON s_order_details.articleordernumber = s_articles_details.ordernumber
            LEFT JOIN s_order_details_attributes
                ON s_order_details_attributes.detailID = s_order_details.id
            LEFT JOIN s_order
                ON s_order.id = s_order_details.orderID
            LEFT JOIN s_articles
                ON s_articles.id = s_articles_details.articleID
            LEFT JOIN s_articles_attributes
                ON s_articles_attributes.articledetailsID = s_articles_details.id
            LEFT JOIN s_articles_esd
                ON s_articles_esd.articledetailsID = s_articles_details.id
            LEFT JOIN s_articles_supplier
                ON s_articles.supplierID = s_articles_supplier.id
            LEFT JOIN (
                SELECT
                    articleDetailId,
                    ABS(SUM(`pickware_erp_stock_ledger_entries`.changeAmount)) as salesWithinPlanningInterval
                FROM `pickware_erp_stock_ledger_entries`
                WHERE `pickware_erp_stock_ledger_entries`.type = :stockSaleType
                AND `pickware_erp_stock_ledger_entries`.created >= :salesIntervalFrom
                AND `pickware_erp_stock_ledger_entries`.created <= :salesIntervalTo
                GROUP BY articleDetailId
            ) AS sales_join_table
                ON sales_join_table.articleDetailId = s_articles_details.id
            LEFT JOIN (
                SELECT
                    `pickware_erp_warehouse_article_detail_configurations`.articleDetailId AS articleDetailId,
                    SUM(`pickware_erp_warehouse_article_detail_configurations`.targetStock) AS targetStock,
                    SUM(`pickware_erp_warehouse_article_detail_configurations`.minimumStock) AS minimumStock
                FROM `pickware_erp_warehouse_article_detail_configurations`
                LEFT JOIN `pickware_erp_warehouses`
                    ON `pickware_erp_warehouses`.id = `pickware_erp_warehouse_article_detail_configurations`.warehouseId
                WHERE `pickware_erp_warehouses`.stockAvailableForSale = 1
                GROUP BY `pickware_erp_warehouse_article_detail_configurations`.articleDetailId
            ) AS warehouse_stock_limits
                ON warehouse_stock_limits.articleDetailId = s_articles_details.id
            ' . $extraJoins . '
            LEFT JOIN (
                SELECT
                    SUM(items.cancelledQuantity) AS cancelledReturned,
                    SUM(items.returnedQuantity) AS returned,
                    items.orderDetailId AS orderDetailId
                FROM pickware_erp_return_shipment_items as items
                GROUP BY items.orderDetailId
            ) AS derivedQuantities 
                ON derivedQuantities.orderDetailId = s_order_details.id
            # Checks whether the article is initialized
            WHERE s_articles_attributes.pickware_stock_initialized = 1
            AND s_articles_attributes.pickware_stock_management_disabled = 0
            AND s_articles_esd.id is NULL
            '. $whereClause .'
            GROUP BY s_articles_details.id
            '. $havingClause .'
            '. $orderQuery .'
            LIMIT '.intval($start). ', '.intval($limit),
            [
                'stateAborted' => OrderStatus::ORDER_STATE_CANCELLED,
                'stockSaleType' => StockLedgerEntry::TYPE_SALE,
                'salesIntervalFrom' => $planningFrom->format(DateTime::ATOM),
                'salesIntervalTo' => $planningTo->format(DateTime::ATOM),
            ]
        );

        $foundRows = Shopware()->Db()->fetchOne('SELECT FOUND_ROWS()');

        // Fix additional texts
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($result, 'id'));
        foreach ($result as &$article) {
            $additionalText = $additionalTexts[$article['id']];
            $article['articleName'] .= $additionalText ? ' - ' . $additionalText : '';
        }

        return [
            'count' => $foundRows,
            'result' => $result,
        ];
    }
}
