<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Orx;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\CSRFWhitelistAware;
use Shopware\Components\DependencyInjection\Container;
use Shopware\Components\Model\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonCommon\Classes\CsvWriter;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;

class Shopware_Controllers_Backend_ViisonPickwareERPStockTakeExport extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    /**
     * @inheritdoc
     */
    public function getViewParams()
    {
        return [
            'showCurrentStockWhenStocktaking' => $this->get('pickware.erp.plugin_config_service')->getShowCurrentStockWhenStocktaking(),
        ];
    }

    /**
     * {@inheritdoc}
     *
     * Overridden to disable the renderer and output buffering for export stock action request to be able to download
     * CSVs as a response. This is relevant for the 'exportStockTakes' and 'exportPendingStockTakes' action.
     */
    public function init()
    {
        parent::init();
        if (in_array($this->Request()->getActionName(), ['exportStockTakes', 'exportPendingStockTakes'])) {
            Shopware()->Plugins()->Controller()->ViewRenderer()->setNoRender();
            $this->Front()->setParam('disableOutputBuffering', true);
        }
    }

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'exportStockTakes',
            'exportPendingStockTakes',
        ];
    }

    /**
     * Skips the pre dispatch for all 'export' requests to be able to display CSV files as response.
     */
    public function preDispatch()
    {
        $actionName = $this->Request()->getActionName();
        if ($actionName !== 'exportStockTakes' && $actionName !== 'exportPendingStockTakes') {
            parent::preDispatch();
        }
    }

    /**
     * Responds the IDs and names of all backend users plus an empty dummy user.
     */
    public function getUserListAction()
    {
        // Fetch the IDs and names of all backend users
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'u.id AS id',
            'u.name AS name'
        )->from(User::class, 'u')
            ->orderBy('u.name');
        $users = $builder->getQuery()->getArrayResult();

        // Add an empty dummy user to the beginning of the results
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_stock_take_export/main');
        $defaultUser = $namespace->get('filter/user/default');
        array_unshift($users, [
            'id' => null,
            'name' => $defaultUser,
        ]);

        $this->View()->assign([
            'success' => true,
            'data' => $users,
            'total' => count($users),
        ]);
    }

    /**
     * Responds a list of filtered, sorted and paginated stock-takes.
     */
    public function getStockTakeListAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $sort = $this->Request()->getParam('sort', []);

        // Create the main query builder
        $builder = $this->getStockTakesMainQueryBuilder($sort);

        // We are using an aggregated select and group by, which would be messed up by the paginator. Hence we need to
        // 'paginate' manually.
        $total = count($builder->getQuery()->getArrayResult());
        $builder->setFirstResult($start);
        $builder->setMaxResults($limit);
        $result = $builder->getQuery()->getArrayResult();

        $result = $this->addAdditionalTexts($result);
        $result = $this->calculateAveragePurchasePrice($result);
        $result = $this->calculateGroupedOldStockNewStock($result);
        $result = $this->addNetAndGrossValues($result);

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $total,
        ]);
    }

    /**
     * Responds a list of filtered, sorted and paginated pending (uncounted) stock-takes.
     */
    public function getPendingStockTakeListAction()
    {
        $start = $this->Request()->getParam('start', 0);
        $limit = $this->Request()->getParam('limit', 25);
        $sort = $this->Request()->getParam('sort', []);

        // Create the query builder
        $builder = $this->getPendingStockTakesMainQueryBuilder($sort);
        $builder
            ->setFirstResult($start)
            ->setMaxResults($limit);

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $paginator->setUseOutputWalkers(false);
        $total = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $result = $this->addAdditionalTexts($result);
        $result = $this->translateCodeOfNullBinLocation($result);

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $total,
        ]);
    }

    /**
     * Counts all given, pending article-binlocation combinations with the given values as a stock take entry.
     */
    public function recordStockTakesAction()
    {
        // Get the IDs of all changed bin location mappings
        $changedEntities = $this->Request()->getParam('countedEntities', []);
        $binLocationMappingIds = array_map(function (array $changedEntity) {
            return $changedEntity['articleDetailBinLocationMappingId'];
        }, $changedEntities);

        $builder = $this->get('models')->createQueryBuilder();
        $builder
            ->select('binLocationMapping')
            ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping', 'binLocationMapping.id')
            ->where('binLocationMapping.id IN (:binLocationMappingIds)')
            ->setParameter('binLocationMappingIds', $binLocationMappingIds);
        $binLocationMappings = $builder->getQuery()->getResult();

        /** @var StockLedgerService $stockLedgerService */
        $stockLedgerService = $this->get('pickware.erp.stock_ledger_service');
        foreach ($changedEntities as $changedEntity) {
            /** @var ArticleDetailBinLocationMapping $binLocationMapping */
            $binLocationMapping = $binLocationMappings[$changedEntity['articleDetailBinLocationMappingId']];
            if (!$binLocationMapping) {
                continue;
            }

            $stockLedgerService->recordStocktake(
                $binLocationMapping->getArticleDetail(),
                $binLocationMapping->getBinLocation(),
                $changedEntity['newStock'],
                $changedEntity['comment']
            );
        }

        $this->View()->assign('success', true);
    }

    /**
     * "Counts" a number of uncounted article-binlocation combinations by setting their stock to 0 as a stock take
     * entry. One request is one batch of article-binlocation combinations being counted.
     */
    public function recordStockTakesUsingZeroStockAction()
    {
        $batchSize = $this->Request()->getParam('batchSize');
        $builder = $this->getPendingStockTakesMainQueryBuilder();
        $builder->setMaxResults($batchSize);

        // Create the query and execute it to get the batch result
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $paginator->setUseOutputWalkers(false);
        $result = $paginator->getIterator()->getArrayCopy();

        $articleDetailBinLocationMappingIds = array_map(function ($pendingArticleDetailBinLocationMapping) {
            return $pendingArticleDetailBinLocationMapping['articleDetailBinLocationMappingId'];
        }, $result);

        /** @var ArticleDetailBinLocationMapping[] $articleDetailBinLocationMapping */
        $articleDetailBinLocationMappings = $this->get('models')->getRepository(ArticleDetailBinLocationMapping::class)->findBy([
            'id' => $articleDetailBinLocationMappingIds,
        ]);

        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_stock_take_export/main');
        $comment = $namespace->get('stocktake/defaultComment');

        /** @var StockLedgerService $stockLedgerService */
        $stockLedgerService = $this->get('pickware.erp.stock_ledger_service');
        foreach ($articleDetailBinLocationMappings as $articleDetailBinLocationMapping) {
            $stockLedgerService->recordStocktake(
                $articleDetailBinLocationMapping->getArticleDetail(),
                $articleDetailBinLocationMapping->getBinLocation(),
                0,
                $comment
            );
        }

        // Fetch the number of remaining (not counted) article detail bin location mappings after this batch was handled
        $pendingCount = $paginator->count();

        $this->View()->assign([
            'success' => true,
            'completed' => count($articleDetailBinLocationMappings),
            'pending' => $pendingCount,
        ]);
    }

    /**
     * Responds a CSV file containing the complete stock takes for the given filter.
     */
    public function exportStockTakesAction()
    {
        // Get the sorted and filtered stock-takes
        $builder = $this->getStockTakesMainQueryBuilder();
        $result = $builder->getQuery()->getArrayResult();

        $result = $this->addAdditionalTexts($result);
        $result = $this->calculateAveragePurchasePrice($result);
        $result = $this->calculateGroupedOldStockNewStock($result);
        $result = $this->addNetAndGrossValues($result);
        $result = $this->formatPrices($result);

        $filename = $this->getCsvFileNameBySnippetName('csv/complete/fileName');
        $csvWriter = new CsvWriter($this->Response(), $filename);
        try {
            // Add header row
            $csvWriter->write($this->getStockTakeCsvHeader());

            // Write the data rows
            foreach ($result as $row) {
                $csvWriter->write([
                    $row['articleNumber'],
                    $row['articleName'],
                    $row['oldStock'],
                    $row['newStock'],
                    $row['changeAmount'],
                    $row['purchasePrice'],
                    (int) $row['taxRate'] . '%',
                    $row['changeValueNet'],
                    $row['changeValueGross'],
                    $row['username'],
                    $row['stockCreationDate']->format('Y-m-d H:i:s'),
                    $row['comment'],
                ]);
            }
        } finally {
            $csvWriter->close();
        }
    }

    /**
     * Responds a CSV file containing the pending stock takes for the given filter.
     */
    public function exportPendingStockTakesAction()
    {
        // Get the sorted and filtered stock-takes
        $builder = $this->getPendingStockTakesMainQueryBuilder();
        $result = $builder->getQuery()->getArrayResult();

        $result = $this->addAdditionalTexts($result);
        $result = $this->translateCodeOfNullBinLocation($result);

        $filename = $this->getCsvFileNameBySnippetName('csv/pending/fileName');
        $csvWriter = new CsvWriter($this->Response(), $filename);

        $showCurrentStockWhenStocktaking = $this->get('pickware.erp.plugin_config_service')->getShowCurrentStockWhenStocktaking();
        try {
            // Add header row
            $csvWriter->write($this->getPendingStockTakeCsvHeader());

            // Create the data rows
            foreach ($result as $row) {
                $newCsvRow = [
                    $row['articleNumber'],
                    $row['articleName'],
                    $row['binLocationCode'],
                ];
                if ($showCurrentStockWhenStocktaking) {
                    $newCsvRow[] = $row['expectedStock'];
                }
                $newCsvRow[] = ''; // Leave last column 'newStock' empty
                $csvWriter->write($newCsvRow);
            }
        } finally {
            $csvWriter->close();
        }
    }

    /**
     * Returns a new query builder, which fetches the required data of stock-takes as well as associated fields and
     * applies both filter conditions and sorting.
     *
     * @param array $sort (optional)
     * @return QueryBuilder
     */
    private function getStockTakesMainQueryBuilder($sort = [])
    {
        // Create the query builder
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'stockLedgerEntry.created AS stockCreationDate',
            'MIN(stockLedgerEntry.oldStock) AS oldStockMin',
            'MIN(stockLedgerEntry.newStock) AS newStockMin',
            'MAX(stockLedgerEntry.oldStock) AS oldStockMax',
            'MAX(stockLedgerEntry.newStock) AS newStockMax',
            'SUM(stockLedgerEntry.changeAmount) AS changeAmount',
            'SUM(stockLedgerEntry.changeAmount * stockLedgerEntry.purchasePrice) AS changeValue',
            // Use the MIN() on the comment field to prevent SQL errors in case `only_full_group_by` is enabled. Since
            // all comments are equal for the same transaction by which we group, selecting any of them does not change
            // the result.
            'MIN(stockLedgerEntry.comment) AS comment',
            'articleDetail.id AS articleDetailId',
            'articleDetail.number AS articleNumber',
            'articleDetail.purchasePrice AS lastPurchasePrice',
            'article.id AS articleId',
            'article.name AS articleName',
            'tax.tax AS taxRate',
            'IFNULL(user.name, \'-\') AS username'
        )->from(StockLedgerEntry::class, 'stockLedgerEntry')
            ->join('stockLedgerEntry.articleDetail', 'articleDetail')
            ->join('articleDetail.article', 'article')
            ->join('article.tax', 'tax')
            ->leftJoin('stockLedgerEntry.user', 'user')
            // Group by article and request (which possibly sums up multiple/different purchase prices). Calculated
            // averages afterwards if necessary
            ->groupBy('articleDetail.id', 'stockLedgerEntry.transactionId');

        $builder = $this->addStockTakesFilterFromRequest($builder);

        if (count($sort) > 0) {
            // Map the sort fields correctly
            $sortFieldMappings = [
                'stockCreationDate' => 'stockLedgerEntry.created',
                'oldStock' => 'stockLedgerEntry.oldStock',
                'newStock' => 'stockLedgerEntry.newStock',
                'changeAmount' => 'stockLedgerEntry.changeAmount',
                'purchasePrice' => 'stockLedgerEntry.purchasePrice',
                'comment' => 'stockLedgerEntry.comment',
                'articleNumber' => 'articleDetail.number',
                'articleName' => 'article.name',
                'taxRate' => 'tax.tax',
                'username' => 'user.name',
            ];
            $sort = $this->applyMappingToSort($sortFieldMappings, $sort);
            $builder->addOrderBy($sort);
        } else {
            // Add default sort fields
            $builder
                ->addOrderBy('articleDetail.number', 'ASC')
                ->addOrderBy('stockLedgerEntry.id', 'ASC');
        }

        return $builder;
    }

    /**
     * Adds all filter conditions to the given QueryBuilder. The condition parameter are taken directly from the request
     * and are assigned if a parameter was found.
     *
     * @param QueryBuilder $builder
     * @return QueryBuilder
     */
    private function addStockTakesFilterFromRequest($builder)
    {
        $warehouseId = $this->Request()->getParam('warehouseId');
        $fromDate = $this->Request()->getParam('fromDate');
        if ($fromDate) {
            $fromDate = new DateTime($fromDate);
            $fromDate->setTime(0, 0, 0);
        }
        $toDate = $this->Request()->getParam('toDate');
        if ($toDate) {
            $toDate = new DateTime($toDate);
            $toDate->setTime(23, 59, 59);
        }
        $userId = $this->Request()->getParam('userId');
        $searchQuery = $this->Request()->getParam('query');

        $builder
            ->where('stockLedgerEntry.type = \'stocktake\'')
            ->andWhere('stockLedgerEntry.warehouseId = :warehouseId')
            ->setParameter('warehouseId', $warehouseId);

        // Add filters that are optional
        if ($fromDate) {
            $builder
                ->andWhere('stockLedgerEntry.created >= :fromDate')
                ->setParameter('fromDate', $fromDate->format('Y-m-d H:i:s'));
        }
        if ($toDate) {
            $builder
                ->andWhere('stockLedgerEntry.created <= :toDate')
                ->setParameter('toDate', $toDate->format('Y-m-d H:i:s'));
        }
        if ($userId) {
            $builder
                ->andWhere('user.id = :userId')
                ->setParameter('userId', $userId);
        }
        if ($searchQuery) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'article.name LIKE :searchQuery',
                    'stockLedgerEntry.comment LIKE :searchQuery',
                    'articleDetail.number LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        return $builder;
    }

    /**
     * Returns a new query builder, which fetches the required data of stock-takes as well as associated fields and
     * applies both filter conditions and sorting.
     *
     * @param array $sort (optional)
     * @return QueryBuilder
     */
    private function getPendingStockTakesMainQueryBuilder($sort = [])
    {
        $builder = $this->get('models')->createQueryBuilder();

        // Create the query builder
        $builder->select(
            'article.id AS articleId',
            'articleDetailBinLocationMapping.id AS articleDetailBinLocationMappingId',
            'articleDetail.id AS articleDetailId',
            'articleDetail.number AS articleNumber',
            'article.name AS articleName',
            'binLocation.code AS binLocationCode',
            'articleDetailBinLocationMapping.lastStocktake AS lastStockTakeDate',
            'articleDetailBinLocationMapping.stock AS expectedStock',
            'tax.tax AS taxRate'
        )->from(ArticleDetailBinLocationMapping::class, 'articleDetailBinLocationMapping')
            ->leftJoin('articleDetailBinLocationMapping.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'articleAttribute')
            ->leftJoin('articleDetailBinLocationMapping.binLocation', 'binLocation')
            ->leftJoin('article.tax', 'tax');

        $builder = $this->addPendingStockTakesFilterFromRequest($builder);

        if (count($sort) > 0) {
            // Map the sort fields correctly
            $sortFieldMappings = [
                'articleNumber' => 'articleDetail.number',
                'articleName' => 'article.name',
                'binLocationCode' => 'binLocation.code',
                'lastStockTakeDate' => 'articleDetailBinLocationMapping.lastStocktake',
                'expectedStock' => 'articleDetailBinLocationMapping.stock',
            ];
            $sort = $this->applyMappingToSort($sortFieldMappings, $sort);
            $builder->addOrderBy($sort);
        } else {
            $builder
                ->addOrderBy('articleDetail.number', 'ASC')
                ->addOrderBy('binLocation.code', 'ASC');
        }

        return $builder;
    }

    /**
     * Adds all filter conditions to the given QueryBuilder. The condition parameter are taken directly from the request
     * and are assigned if a parameter was found.
     *
     * @param QueryBuilder $builder
     * @return QueryBuilder
     */
    private function addPendingStockTakesFilterFromRequest(QueryBuilder $builder)
    {
        $warehouseId = $this->Request()->getParam('warehouseId');
        $fromDate = $this->Request()->getParam('fromDate');
        if ($fromDate) {
            $fromDate = new DateTime($fromDate);
            $fromDate->setTime(0, 0, 0);
        }
        $searchQuery = $this->Request()->getParam('query');
        $excludedArticleDetailBinLocationMappingIds = json_decode($this->Request()->getParam('excludedArticleDetailBinLocationMappingIds', ''));

        $builder
            ->where('articleAttribute.pickwareStockInitialized = 1')
            ->andWhere('articleAttribute.pickwareStockManagementDisabled = 0')
            ->andWhere('binLocation.warehouseId = :warehouseId')
            ->setParameter('warehouseId', $warehouseId);

        if ($excludedArticleDetailBinLocationMappingIds !== null
            && count($excludedArticleDetailBinLocationMappingIds) > 0
        ) {
            $builder
                ->andWhere('articleDetailBinLocationMapping.id NOT IN (:excludedMappingIds)')
                ->setParameter('excludedMappingIds', $excludedArticleDetailBinLocationMappingIds);
        }

        // Since we are handling pending stock takes which are entries where no stock take was saved, we need to invert
        // the date filter. I.e. "no stock take since 12.12.2012" is inverted to "last stock take NULL or before
        // (including) 12.12.2012"
        if ($fromDate) {
            $builder->setParameter('fromDate', $fromDate->format('Y-m-d H:i:s'));
            $builder->andWhere(
                $builder->expr()->orX(
                    'articleDetailBinLocationMapping.lastStocktake IS NULL',
                    new Orx('articleDetailBinLocationMapping.lastStocktake < :fromDate')
                )
            );
        } else {
            // If no date restriction was given, only show entries that habe no stock take date
            $builder->andWhere('articleDetailBinLocationMapping.lastStocktake IS NULL');
        }

        if ($searchQuery) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'article.name LIKE :searchQuery',
                    'articleDetail.number LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        return $builder;
    }

    /**
     * @return string[]
     */
    private function getStockTakeCsvHeader()
    {
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_stock_take_export/file');

        return [
            $namespace->get('csv/column/number/header'),
            $namespace->get('csv/column/name/header'),
            $namespace->get('csv/column/oldInstock/header'),
            $namespace->get('csv/column/newInstock/header'),
            $namespace->get('csv/column/changeAmount/header'),
            $this->getPurchasePriceSnippet(),
            $namespace->get('csv/column/tax/header'),
            $namespace->get('csv/column/changeValueNet/header'),
            $namespace->get('csv/column/changeValueGross/header'),
            $namespace->get('csv/column/user/header'),
            $namespace->get('csv/column/stock_created/header'),
            $namespace->get('csv/column/comment/header'),
        ];
    }

    /**
     * @return string[]
     */
    private function getPendingStockTakeCsvHeader()
    {
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_stock_take_export/file');

        $headerFields = [
            $namespace->get('csv/column/number/header'),
            $namespace->get('csv/column/name/header'),
            $namespace->get('csv/column/binLocationCode/header'),
        ];

        if ($this->get('pickware.erp.plugin_config_service')->getShowCurrentStockWhenStocktaking()) {
            $headerFields[] = $namespace->get('csv/column/expectedStock/header');
        }

        $headerFields[] = $namespace->get('csv/column/pendingNewInstock/header');

        return $headerFields;
    }

    /**
     * @return string
     */
    private function getPurchasePriceSnippet()
    {
        $purchasePriceNamespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_purchase_price_helper/main');
        $purchasePriceMode = $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->get('purchasePriceMode');

        return vsprintf(
            '%s (%s)',
            [
                $purchasePriceNamespace->get('purchase_price'),
                $purchasePriceNamespace->get($purchasePriceMode),
            ]
        );
    }

    /**
     * @param string $snippetName
     * @return string
     */
    private function getCsvFileNameBySnippetName($snippetName)
    {
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_stock_take_export/file');
        $now = new \DateTime();

        return sprintf(
            $namespace->get($snippetName),
            $now->format('Y-m-d_H-i-s')
        );
    }

    /**
     * Renames all 'property's of the given sort array to use them in a respective query builder accoding to the given
     * mapping (e.g. 'userId' => 'user.id').
     *
     * @param array $mappings
     * @param array $sort
     * @return array
     */
    private function applyMappingToSort($mappings, $sort)
    {
        foreach ($sort as &$field) {
            if (array_key_exists($field['property'], $mappings)) {
                $field['property'] = $mappings[$field['property']];
            }
        }

        return $sort;
    }

    /**
     * @param array $rows
     * @return array
     */
    private function addAdditionalTexts($rows)
    {
        // Fetch the variant additional texts of all article details contained in the result
        $articleDetailIds = array_column($rows, 'articleDetailId');
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts($articleDetailIds);

        foreach ($rows as &$row) {
            // Add Additional article name text
            if ($additionalTexts[$row['articleDetailId']]) {
                $row['articleName'] .= ' - ' . $additionalTexts[$row['articleDetailId']];
            }
        }

        return $rows;
    }

    /**
     * Calculates the purchase price for each stock take entry. If there were changes (changeAmount !== 0) we calculate
     * the purchase price by using the changeAmount and the changeValue. If no changes were logged (changeAmount = 0) we
     * use the last purchase price of the article detail.
     *
     * @param array $rows
     * @return array
     */
    private function calculateAveragePurchasePrice($rows)
    {
        foreach ($rows as &$row) {
            $row['changeAmount'] = $row['changeAmount'] ?: 0;
            if ($row['changeAmount'] !== 0) {
                $row['purchasePrice'] = $row['changeValue'] / $row['changeAmount'];
            } else {
                $row['purchasePrice'] = ($row['lastPurchasePrice'] ?: 0);
            }
            $row['purchasePrice'] = round($row['purchasePrice'], 2);
        }

        return $rows;
    }

    /**
     * @param array $rows
     * @return array
     */
    private function translateCodeOfNullBinLocation($rows)
    {
        $defaultBinLocationSnippet = $this->get('snippets')
            ->getNamespace('backend/viison_pickware_erp_stock_take_export/main')->get('grid/nullBinLocationCode');
        foreach ($rows as &$row) {
            if ($row['binLocationCode'] === 'pickware_null_bin_location') {
                $row['binLocationCode'] = $defaultBinLocationSnippet;
            }
        }

        return $rows;
    }

    /**
     * If the (completed) stocks are grouped to summarize stock entrys with the same purchase price, we need to select
     * the oldStock and newStock are not stored in a single row (because both come from different rows). Hence we need
     * to select the correct min and max values and return a "single row result" with values from different entries. The
     * new stock value is also needed to calculate the stock value.
     *
     * @param array $rows
     * @return array
     */
    private function calculateGroupedOldStockNewStock($rows)
    {
        foreach ($rows as &$row) {
            // Select the actual oldStock and newStock (which are messed up by the group by)
            if ($row['changeAmount'] > 0) {
                $row['oldStock'] = $row['oldStockMin'];
                $row['newStock'] = $row['newStockMax'];
            } else {
                $row['oldStock'] = $row['oldStockMax'];
                $row['newStock'] = $row['newStockMin'];
            }
        }

        return $rows;
    }

    /**
     * Since purchase price may be stored as a net or gross price, we need to calculate the changeValue depending on
     * this purchase price mode which lays in the Pickware ERP configuration. This method adds a changeValue in net and
     * gross.
     *
     * Remark rounding: We round at the end of the calculation "value = round(price * amount * tax)".
     *
     * @param array $rows
     * @return array
     */
    private function addNetAndGrossValues($rows)
    {
        $purchasePriceMode = $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->get('purchasePriceMode');

        foreach ($rows as &$row) {
            $taxFactor = 1.0 + $row['taxRate'] / 100.0;

            if ($purchasePriceMode === 'net') {
                $row['changeValueNet'] = round($row['changeValue'], 2);
                $row['changeValueGross'] = round($row['changeValue'] * $taxFactor, 2);
            } else {
                $row['changeValueNet'] = round($row['changeValue'] / $taxFactor, 2);
                $row['changeValueGross'] = round($row['changeValue'], 2);
            }
        }

        return $rows;
    }

    /**
     * @param array $rows
     * @return array
     */
    private function formatPrices($rows)
    {
        $decimalSeparator = '.';
        $swagImportExportBootstrap = ViisonCommonUtil::getPluginBootstrap('SwagImportExport');
        if ($swagImportExportBootstrap && $swagImportExportBootstrap->Config()->get('useCommaDecimal')) {
            $decimalSeparator = ',';
        }

        foreach ($rows as &$row) {
            if ($row['purchasePrice'] !== null) {
                $row['purchasePrice'] = number_format($row['purchasePrice'], 2, $decimalSeparator, '');
                $row['changeValueNet'] = number_format($row['changeValueNet'], 2, $decimalSeparator, '');
                $row['changeValueGross'] = number_format($row['changeValueGross'], 2, $decimalSeparator, '');
            }
            $row['stockValueNet'] = number_format($row['stockValueNet'], 2, $decimalSeparator, '');
            $row['stockValueGross'] = number_format($row['stockValueGross'], 2, $decimalSeparator, '');
        }

        return $rows;
    }
}
