<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\CSRFWhitelistAware;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\StockValuation\Report;
use Shopware\CustomModels\ViisonPickwareERP\StockValuation\ReportRow;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonCommon\Classes\CsvWriter;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockValuation\StockValuationService;

class Shopware_Controllers_Backend_ViisonPickwareERPStockValuation extends ViisonCommonBaseController implements CSRFWhitelistAware
{
    use BackendExceptionHandling;

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'exportReportAsCsv'
        ];
    }

    /**
     * Disables the renderer and output buffering for all 'exportReportAsCsv' requests
     * to be able to respond CSV files.
     */
    public function init()
    {
        parent::init();
        if ($this->Request()->getActionName() === 'exportReportAsCsv') {
            Shopware()->Plugins()->Controller()->ViewRenderer()->setNoRender();
            $this->Front()->setParam('disableOutputBuffering', true);
        }
    }

    /**
     * Don't use the JSON renderer for the 'exportReportAsCsv' action.
     */
    public function preDispatch()
    {
        if ($this->Request()->getActionName() !== 'exportReportAsCsv') {
            parent::preDispatch();
        }
    }

    public function getReportsAction()
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update prefixes of sort fields
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], '.') === false) {
                $sortField['property'] = 'report.' . $sortField['property'];
            }
        }
        unset($sortField);

        // Build the main query
        $builder = $entityManager->createQueryBuilder();
        $builder
            ->select('report', 'warehouse')
            ->from(Report::class, 'report')
            ->leftJoin('report.warehouse', 'warehouse')
            ->where('report.generated = 1')
            ->andWhere('report.preview = 0')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    public function getReportRowsAction()
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);
        $query = $this->Request()->getParam('query', '');
        $excludeWithoutStock = $this->Request()->getParam('excludeArticlesWithoutStock', 'false') === 'true';

        // Update prefixes of sort fields
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], '.') === false) {
                $sortField['property'] = 'reportRow.' . $sortField['property'];
            }
        }
        unset($sortField);

        // Update prefixes of sort fields
        foreach ($filter as &$filterField) {
            if (mb_strpos($filterField['property'], '.') === false) {
                $filterField['property'] = 'reportRow.' . $filterField['property'];
            }
        }
        unset($filterField);

        // Build the main query
        $builder = $entityManager->createQueryBuilder();
        $builder
            ->select(
                'reportRow',
                'articleDetail',
                'purchases',
                'article'
            )
            ->from(ReportRow::class, 'reportRow')
            ->join('reportRow.report', 'report')
            ->leftJoin('reportRow.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('reportRow.purchases', 'purchases')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        if (!empty($query)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'reportRow.articleDetailNumber LIKE :searchQuery',
                    'reportRow.articleDetailName LIKE :searchQuery'
                )
            )->setParameter('searchQuery', '%' . $query . '%');
        }

        if ($excludeWithoutStock) {
            $builder->andWhere('reportRow.stock > 0');
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $resultCount = $paginator->count();
        $reportRows = $paginator->getIterator()->getArrayCopy();

        // Surplus stock is shown in the UI as a purchase, so we create a "fake" purchase entry for the surplus stock
        foreach ($reportRows as &$reportRow) {
            if ($reportRow['surplusStock'] > 0) {
                array_unshift($reportRow['purchases'], [
                    'id' => 0,
                    'date' => null,
                    'purchasePriceNet' => $reportRow['surplusPurchasePriceNet'],
                    'quantity' => $reportRow['surplusStock'],
                    'quantityUsedForValuation' => $reportRow['surplusStock'],
                    'type' => 'surplus',
                ]);
            }
        }
        unset($reportRow);

        $this->View()->assign([
            'success' => true,
            'data' => $reportRows,
            'total' => $resultCount,
        ]);
    }

    public function createReportPreviewAction()
    {
        set_time_limit(600);
        try {
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');
            /** @var Warehouse $warehouse */
            $warehouse = $entityManager->find(Warehouse::class, $this->Request()->getParam('warehouseId'));
            $reportingDay = new DateTime($this->Request()->get('reportingDay'));
            $method = Shopware()->Plugins()->Core()->get('ViisonPickwareERP')->Config()->get('stockValuationMethod');
            $report = new Report($reportingDay, $method, $warehouse);
            $entityManager->persist($report);
            $entityManager->flush($report);

            $builder = $entityManager->createQueryBuilder();
            $builder
                ->select('report', 'warehouse')
                ->from(Report::class, 'report')
                ->join('report.warehouse', 'warehouse')
                ->where('report.id = :reportId')
                ->setParameter('reportId', $report->getId());
            $reportArray = $builder->getQuery()->getArrayResult()[0];

            $reportArray['nextGenerationStep'] = $report->getNextGenerationStep();
            $reportArray['progress'] = $report->getProgress();

            $this->View()->assign([
                'success' => true,
                'report' => $reportArray,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function performNextReportGenerationStepAction()
    {
        set_time_limit(600);
        try {
            /** @var ModelManager $entityManager */
            $entityManager = $this->get('models');

            $reportId = $this->Request()->getParam('reportId');
            ParameterValidator::assertIsInteger($reportId, 'reportId');
            /** @var Report $report */
            $report = $entityManager->find(Report::class, $reportId);
            ParameterValidator::assertEntityFound($report, Report::class, $reportId, 'reportId');

            /** @var StockValuationService $stockValuationService */
            $stockValuationService = $this->get('pickware_erp.stock_valuation_service');
            $stockValuationService->performNextReportGenerationStep($report);

            $builder = $entityManager->createQueryBuilder();
            $builder
                ->select('report', 'warehouse')
                ->from(Report::class, 'report')
                ->join('report.warehouse', 'warehouse')
                ->where('report.id = :reportId')
                ->setParameter('reportId', $report->getId());
            $reportArray = $builder->getQuery()->getArrayResult()[0];

            $entityManager->refresh($report);
            $reportArray['nextGenerationStep'] = $report->getNextGenerationStep();
            $reportArray['progress'] = $report->getProgress();

            $this->View()->assign([
                'success' => true,
                'report' => $reportArray,
            ]);
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    public function persistReportAction()
    {
        $reportId = $this->Request()->getParam('reportId');
        $comment = $this->Request()->getParam('comment', null);

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        /** @var Report $report */
        $report = $entityManager->find(Report::class, $reportId);
        $report->persist();
        $report->setComment($comment);
        $entityManager->flush($report);

        $this->View()->assign([
            'success' => true,
        ]);
    }

    public function getReportAggregationAction()
    {
        $reportId = $this->Request()->getParam('reportId');

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        // Build the main query
        $builder = $entityManager->createQueryBuilder();
        $builder
            ->select(
                'SUM(reportRow.valuationNet) AS totalValuationNet',
                'SUM(reportRow.valuationGross) AS totalValuationGross'
            )
            ->from(ReportRow::class, 'reportRow')
            ->join('reportRow.report', 'report')
            ->where('report.id = :reportId')
            ->setParameter('reportId', $reportId);

        $result = $builder->getQuery()->getArrayResult()[0];

        $result['totalValuationNet'] = (float) $result['totalValuationNet'];
        $result['totalValuationGross'] = (float) $result['totalValuationGross'];

        $this->View()->assign([
            'success' => true,
            'data' => $result,
        ]);
    }

    public function deleteReportAction()
    {
        $reportId = $this->Request()->getParam('reportId');

        /* @var $database \Enlight_Components_Db_Adapter_Pdo_Mysql */
        $database = $this->get('db');

        $database->query(
            'DELETE FROM pickware_erp_stock_valuation_reports
            WHERE id = ?',
            [$reportId]
        );

        $this->View()->assign([
            'success' => true,
        ]);
    }

    public function exportReportAsCsvAction()
    {
        $reportId = $this->Request()->getParam('reportId');
        $excludeWithoutStock = $this->Request()->getParam('excludeArticlesWithoutStock', 'false') === 'true';

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var Report $report */
        $report = $entityManager->find(Report::class, $reportId);

        $builder = $entityManager->createQueryBuilder();
        $builder
            ->select(
                'reportRow.articleDetailName AS articleDetailName',
                'reportRow.articleDetailNumber AS articleDetailNumber',
                'reportRow.stock AS stock',
                'reportRow.taxRate AS taxRate',
                'reportRow.valuationNet AS valuationNet',
                'reportRow.valuationGross AS valuationGross'
            )
            ->from(ReportRow::class, 'reportRow')
            ->join('reportRow.report', 'report')
            ->orderBy('reportRow.articleDetailNumber')
            ->where('report.id = :reportId')
            ->setParameter('reportId', $reportId);

        if ($excludeWithoutStock) {
            $builder->andWhere('reportRow.stock > 0');
        }

        $result = $builder->getQuery()->getArrayResult();

        $filename = sprintf(
            'stock_valuation_%s_%s_%s.csv',
            $report->getWarehouse()->getCode(),
            $report->getMethod(),
            $report->getReportingDay()->format('Y-m-d')
        );
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_stock_valuation/controller');
        $defaultCurrency = Currency::getDefaultCurrency();
        $csvWriter = new CsvWriter($this->Response(), $filename, true);
        $csvWriter->write([
            $namespace->get('csv_export/column/article_detail_number'),
            $namespace->get('csv_export/column/article_detail_name'),
            $namespace->get('csv_export/column/stock'),
            $namespace->get('csv_export/column/tax_rate'),
            sprintf($namespace->get('csv_export/column/valuation_net'), $defaultCurrency->getCurrency()),
            sprintf($namespace->get('csv_export/column/valuation_gross'), $defaultCurrency->getCurrency()),
        ]);

        $decimalSeparator = $namespace->get('csv_export/decimal_separator') ?: '.';
        if (ViisonCommonUtil::isPluginInstalledAndActive('', 'SwagImportExport')) {
            $useCommaDecimal = Shopware()->Plugins()->Backend()->get('SwagImportExport')->Config()->get('useCommaDecimal');
            $decimalSeparator = $useCommaDecimal ? ',' : '.';
        }
        $formatMoneyValue = function ($value) use ($decimalSeparator) {
            return number_format($value, 2, $decimalSeparator, '');
        };
        $formatTaxValue = function ($value) use ($decimalSeparator) {
            return number_format($value, 1, $decimalSeparator, '');
        };
        foreach ($result as $row) {
            $csvWriter->write([
                $row['articleDetailNumber'],
                $row['articleDetailName'],
                $row['stock'],
                $formatTaxValue($row['taxRate']),
                $formatMoneyValue($row['valuationNet']),
                $formatMoneyValue($row['valuationGross']),
            ]);
        }
        $csvWriter->close();
    }
}
