<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use DoctrineExtensions\Query\Mysql\Ceil;
use DoctrineExtensions\Query\Mysql\Greatest;
use Shopware\Components\Model\QueryBuilder;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\WarehouseArticleDetailStockCount;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Article\Supplier as Manufacturer;
use Shopware\Models\Shop\Currency;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonCommon\DoctrineExtensions\Query\Mysql\Ceil as ViisonCommonCeil;
use Shopware\Plugins\ViisonCommon\DoctrineExtensions\Query\Mysql\Greatest as ViisonCommonGreatest;

/**
 * Backend controller used for auto loading common Ext JS modules for supplier and supplier order management.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPSupplierCommon extends ViisonCommonBaseController
{
    const FILTER_NONE = 0;
    const FILTER_REORDER = 1;
    const FILTER_URGENT_REORDER = 2;

    /**
     * Responds a paginated, filtered and sorted list of article details.
     *
     * Remark: This method returns articles regardless of their current stock management
     * status, meaning that articles, whose stock is managed, are returned as well as those,
     * whose stock is not managed. This is based on the rational, that customers may sell
     * products, whose exact stock amount is not important (e.g. huge amount with very low
     * purchase price) and which are reordered "at sight".
     * Nevertheless managed articles are treated with higher priority: they are always
     * listed before the unmanaged ones regardless of any custom sorting.
     */
    public function getArticleDetailListAction()
    {
        $limit = $this->Request()->getParam('limit');
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);
        $considerOpenOrders = (bool) $this->Request()->getParam('considerOpenOrders', 1);
        $useTotalIncomingStock = (bool) $this->Request()->getParam('useTotalIncomingStock', 0);
        $filterReorder = (int) $this->Request()->getParam('filterReorder', self::FILTER_NONE);

        // Unroll filters
        $filter = self::unrollIdBlacklistFilter($filter, 'supplierIdBlacklist', 'pickware_erp_article_detail_supplier_mappings', 'articleDetailId', 'articleDetail.id');
        $filter = self::unrollIdBlacklistFilter($filter, 'orderIdBlacklist', 'pickware_erp_supplier_order_items', 'articleDetailId', 'articleDetail.id', 'supplierOrderId');
        $filter = self::unrollQueryBlacklistFilter(
            $filter,
            'preAssignedOrderArticlesBlacklist',
            'articleDetail.id',
            'SELECT articleDetailId
            FROM `pickware_erp_supplier_order_items`
            WHERE `supplierOrderId` IS NULL AND `articleDetailId` IS NOT NULL'
        );

        // Shopware 5.6 includes the required DQL functions, load them if they exist
        /** @var \Doctrine\ORM\Configuration $entityManagerConfig */
        $entityManagerConfig = $this->get('models')->getConfiguration();
        if (!$entityManagerConfig->getCustomNumericFunction('CEIL')) {
            $classToLoad = class_exists(Ceil::class) ? Ceil::class : ViisonCommonCeil::class;
            $entityManagerConfig->addCustomNumericFunction('CEIL', $classToLoad);
        }
        if (!$entityManagerConfig->getCustomStringFunction('GREATEST')) {
            $classToLoad = class_exists(Greatest::class) ? Greatest::class : ViisonCommonGreatest::class;
            $entityManagerConfig->addCustomStringFunction('GREATEST', $classToLoad);
        }

        // Build the main query
        /** @var QueryBuilder $builder */
        $builder = $this->get('models')->createQueryBuilder();
        $supplierId = $this->Request()->getParam('supplierId');
        if ($supplierId) {
            // Use the supplier article details as start
            $builder->from(ArticleDetailSupplierMapping::class, 'supplierMapping')
                    ->join('supplierMapping.articleDetail', 'articleDetail')
                    ->where('supplierMapping.supplierId = :supplierId')
                    ->setParameter('supplierId', $supplierId);
        } else {
            $builder->from(ArticleDetail::class, 'articleDetail');
        }

        // Join main article, its attributes and its 'supplier', that is fabricator
        $builder->join('articleDetail.article', 'article')
                ->join('articleDetail.attribute', 'attribute')
                ->join('article.supplier', 'manufacturer')
                ->join(
                    WarehouseArticleDetailStockCount::class,
                    'articleDetailStockCounts',
                    'WITH',
                    'articleDetail.id = articleDetailStockCounts.articleDetailId'
                )
                ->join(
                    WarehouseArticleDetailConfiguration::class,
                    'warehouseArticleDetailConfig',
                    'WITH',
                    (
                        // Perform a one-to-one like join with the article detail stock count to maintain the correct
                        // number or results
                        'articleDetailStockCounts.articleDetailId = warehouseArticleDetailConfig.articleDetailId'
                        . ' AND articleDetailStockCounts.warehouseId = warehouseArticleDetailConfig.warehouseId'
                    )
                );

        // Add filters
        $builder->addFilter($filter);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'article.name LIKE :searchQuery',
                    'article.description LIKE :searchQuery',
                    'articleDetail.number LIKE :searchQuery',
                    'manufacturer.name LIKE :searchQuery',
                    'articleDetail.supplierNumber LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Determine the total result count using a customised query. This is necessary,
        // because the default Doctrine paginator cannot handle a count on some specific
        // column, but always requires a 'default id' column. Use the FOUND_ROWS statement
        // since the GROUP BY command is not correctly considered in a SELECT COUNT( ).
        $builder->addGroupBy('articleDetail.id');
        $builder->select('articleDetail.id')->getQuery()->execute();
        $totalResult = $this->get('db')->fetchOne('SELECT FOUND_ROWS();');

        // The term "reserved stock" refers to the physical stock, which is already ordered by a customer,
        // but is not yet processed, meaning the physical stock is still stored in the warehouse. This
        // stock is no longer available for sale. Since we miss a reserved stock counter at the moment,
        // we "approximate" its value by using the difference of the sum of the physical stock of all warehouses,
        // whose stock is available in the online shop (pickwarePhysicalStockForSale), and Shopware's instock
        // counter, which gets decreased as soon as a new order has been placed and not just when, the related
        // physical stock has left the warehouse.
        $reservedStockStmt = '(MIN(attribute.pickwarePhysicalStockForSale) - MIN(articleDetail.inStock))';
        $physicalStockStmt = 'SUM(articleDetailStockCounts.stock)';
        if ($considerOpenOrders) {
            $availableStockStmt = '(' . $physicalStockStmt . ' - ' . $reservedStockStmt . ')';
        } else {
            $availableStockStmt = $physicalStockStmt;
        }
        $minimumStockStmt = 'SUM(warehouseArticleDetailConfig.minimumStock)';
        $targetStockStmt = 'SUM(warehouseArticleDetailConfig.targetStock)';
        if ($useTotalIncomingStock) {
            $incomingStockStmt = 'MIN(attribute.pickwareIncomingStock)';
        } else {
            $incomingStockStmt = 'SUM(articleDetailStockCounts.incomingStock)';
        }
        if ($supplierId) {
            $suggestedReorderQuantityStmt = sprintf(
                'CEIL(
                    GREATEST(
                        (%s - %s - %s),
                        (CASE WHEN supplierMapping.minimumOrderAmount IS NULL
                            THEN 0
                            ELSE supplierMapping.minimumOrderAmount
                        END)
                    ) / supplierMapping.packingUnit
                ) * supplierMapping.packingUnit',
                $targetStockStmt,
                $availableStockStmt,
                $incomingStockStmt
            );
        } else {
            $builder->leftJoin(
                ArticleDetailSupplierMapping::class,
                'defaultSupplierMapping',
                'WITH',
                'articleDetail.id = defaultSupplierMapping.articleDetailId AND defaultSupplierMapping.defaultSupplier = 1'
            );
            $suggestedReorderQuantityStmt = sprintf(
                'CEIL(
                    GREATEST(
                        (%s - %s - %s),
                        (CASE WHEN defaultSupplierMapping.minimumOrderAmount IS NULL
                            THEN 0
                            ELSE defaultSupplierMapping.minimumOrderAmount
                        END)
                    )
                    / (CASE WHEN defaultSupplierMapping.packingUnit IS NULL
                        THEN 1
                        ELSE defaultSupplierMapping.packingUnit
                    END)
                )
                * (CASE WHEN defaultSupplierMapping.packingUnit IS NULL
                    THEN 1
                    ELSE defaultSupplierMapping.packingUnit
                END)',
                $targetStockStmt,
                $availableStockStmt,
                $incomingStockStmt
            );
        }

        $builder->select(
            'articleDetail.id AS id',
            'article.id AS articleId',
            'article.name AS name',
            'articleDetail.number AS orderNumber',
            'manufacturer.name AS manufacturerName',
            'articleDetail.supplierNumber AS manufacturerArticleNumber',
            'IFNULL(articleDetail.stockMin, 0) AS stockMin',
            'articleDetail.purchasePrice AS purchasePrice',
            $minimumStockStmt . ' AS minimumStock',
            $targetStockStmt . ' AS targetStock',
            $availableStockStmt . ' AS availableStock',
            'articleDetail.inStock AS onlineAvailableStock',
            'attribute.pickwarePhysicalStockForSale AS onlinePhysicalStock',
            'CASE
                WHEN ' . $suggestedReorderQuantityStmt . ' > 0
                    THEN ' . $suggestedReorderQuantityStmt . '
                ELSE
                    0
            END AS suggestedReorderQuantity',
            $incomingStockStmt . ' AS incomingStock',
            'attribute.pickwareStockManagementDisabled'
        );

        $reorderCondition = '(' . $availableStockStmt . ' + ' . $incomingStockStmt .' < ' . $minimumStockStmt . ' AND attribute.pickwareStockManagementDisabled = 0)';
        $urgentReorderCondition = '(' . $availableStockStmt . ' + ' . $incomingStockStmt . ' < 0 AND attribute.pickwareStockManagementDisabled = 0)';

        if ($filterReorder === self::FILTER_REORDER) {
            $builder->andHaving($reorderCondition);
        } elseif ($filterReorder === self::FILTER_URGENT_REORDER) {
            $builder->andHaving($urgentReorderCondition);
        }

        // Articles shipped via drop shipping don't need to be reordered:
        $isDropShippingPluginInstalled = ViisonCommonUtil::isPluginInstalledAndActive('Backend', 'ViisonDropShipping');
        if ($filterReorder !== self::FILTER_NONE && $isDropShippingPluginInstalled) {
            $builder->andWhere('attribute.viisonDropShipperMail IS NULL OR TRIM(attribute.viisonDropShipperMail) = \'\'');
        }

        if ($isDropShippingPluginInstalled) {
            $builder->addSelect('CASE WHEN (attribute.viisonDropShipperMail IS NOT NULL AND TRIM(attribute.viisonDropShipperMail) != \'\') THEN 1 ELSE 0 END AS dropShippingArticle');
            $builder->addSelect(sprintf(
                'CASE
                    WHEN %1$s AND (attribute.viisonDropShipperMail IS NULL OR TRIM(attribute.viisonDropShipperMail) = \'\') THEN
                        CASE
                            WHEN %2$s
                                THEN 2
                            ELSE 1
                        END
                    ELSE 0
                END AS reorder',
                $reorderCondition,
                $urgentReorderCondition
            ));
        } else {
            $builder->addSelect(sprintf(
                'CASE
                    WHEN %1$s THEN
                        CASE
                            WHEN %2$s
                            THEN 2
                            ELSE 1
                        END
                    ELSE 0
                END AS reorder',
                $reorderCondition,
                $urgentReorderCondition
            ));
        }

        // Order the results
        if ($isDropShippingPluginInstalled) {
            // Use the drop shipping settings as the first sort criterion to push all
            // drop shipping articles to the end of the results
            array_unshift($sort, [
                'property' => 'dropShippingArticle',
                'direction' => 'ASC',
            ]);
        }
        array_unshift($sort, [
            'property' => 'attribute.pickwareStockManagementDisabled',
            'direction' => 'ASC',
        ]);

        $hasOrderNumberSorting = false;
        foreach ($sort as $sorting) {
            $builder->addOrderBy($sorting['property'], $sorting['direction']);
            if ($sorting['property'] === 'orderNumber') {
                $hasOrderNumberSorting = true;
            }
        }
        if (!$hasOrderNumberSorting) {
            // Ensure that sorted "groups" are internally sorted by the articles order number
            $builder->addOrderBy('orderNumber', 'ASC');
        }

        // Set pagination limits manually
        if ($offset) {
            $builder->setFirstResult($offset);
        }
        if ($limit) {
            $builder->setMaxResults($limit);
        }

        // Fetch the paginated results
        $result = $builder->getQuery()->getArrayResult();

        // Fetch additionalTexts
        $articleDetailIds = array_map(function ($articleDetail) {
            return $articleDetail['id'];
        }, $result);
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts($articleDetailIds);

        // Format the results
        foreach ($result as &$articleDetail) {
            // Create additional text
            $additionalText = $additionalTexts[$articleDetail['id']];
            $articleDetail['name'] .= ($additionalText) ? (' - ' . $additionalText) : '';

            // Add all suppliers that are mapped to the article
            $builder = $this->get('models')->createQueryBuilder();
            $builder->select('supplierMapping')
                    ->from(ArticleDetailSupplierMapping::class, 'supplierMapping')
                    ->where('supplierMapping.articleDetailId = :articleDetailId')
                    ->setParameter('articleDetailId', $articleDetail['id'])
                    ->orderBy('supplierMapping.defaultSupplier', 'DESC');
            $articleDetail['suppliers'] = $builder->getQuery()->getArrayResult();
        }

        $currencyId = $this->Request()->getParam('currencyId');
        if ($currencyId) {
            $result = $this->convertPurchasePricesIfNecessary($result, $currencyId);
        }

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Responds a paginated, filtered and sorted list of fabricators.
     */
    public function getFabricatorListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update sort fields
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], 'manufacturer.') === false) {
                $sortField['property'] = 'manufacturer.' . $sortField['property'];
            }
        }

        // Unroll filters
        $filter = self::unrollIdBlacklistFilter($filter, 'supplierIdBlacklist', 'pickware_erp_manufacturer_supplier_mappings', 'manufacturerId', 'manufacturer.id');

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'manufacturer'
        )->from(Manufacturer::class, 'manufacturer')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'manufacturer.name LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Checks the given filter array for the special ID blacklist filter and, if found,
     * uses the given table and field names to unroll the filter. That is, it fetches all assigned
     * supplier IDs and uses them to add a new 'id NOT IN' filter condition instead of the
     * original ID blacklist condition.
     *
     * @param array $filter
     * @param string $idBlacklistFilterName
     * @param string $tableName
     * @param string $tableFieldKey
     * @param string $modelFieldKey
     * @param string $filterTableFieldKey (optional, default 'supplierId')
     * @return array
     */
    public static function unrollIdBlacklistFilter($filter, $idBlacklistFilterName, $tableName, $tableFieldKey, $modelFieldKey, $filterTableFieldKey = 'supplierId')
    {
        // Check for special ID blacklist filter
        $idBlacklistFilterIndex = self::findFilter($filter, $idBlacklistFilterName);
        if ($idBlacklistFilterIndex === -1) {
            return $filter;
        }

        // Unroll the filter by loading the associated IDs
        $assignedIds = Shopware()->Container()->get('db')->fetchCol(
            'SELECT ' . $tableFieldKey . '
            FROM ' . $tableName . '
            WHERE ' . $filterTableFieldKey . ' = ? AND ' . $tableFieldKey . ' IS NOT NULL',
            [
                $filter[$idBlacklistFilterIndex]['value']
            ]
        );
        unset($filter[$idBlacklistFilterIndex]);

        return self::addNotInFilter($filter, $modelFieldKey, $assignedIds);
    }

    /**
     * Checks the given filter array for the special query blacklist filter and, if found,
     * uses the given query to unroll the filter. That is, it fetches a column of the query
     * results and uses them to add a new 'NOT IN' filter condition instead of the original
     * query blacklist condition.
     *
     * @param array $filter
     * @param string $blacklistFilterName
     * @param string $modelFieldKey
     * @param string $query
     * @return array
     */
    public static function unrollQueryBlacklistFilter($filter, $blacklistFilterName, $modelFieldKey, $query)
    {
        // Check for special blacklist filter
        $blacklistFilterIndex = self::findFilter($filter, $blacklistFilterName);
        if ($blacklistFilterIndex === -1) {
            return $filter;
        }

        // Unroll the filter by executing the query
        $assignedIds = Shopware()->Container()->get('db')->fetchCol($query);
        unset($filter[$blacklistFilterIndex]);

        return self::addNotInFilter($filter, $modelFieldKey, $assignedIds);
    }

    /**
     * Tries to find the filter item having the given property name and returns its index
     * or -1 if was not found.
     *
     * @param array $filter
     * @param string $property
     * @return int
     */
    private static function findFilter($filter, $property)
    {
        foreach ($filter as $index => $filterItem) {
            if ($filterItem['property'] === $property) {
                return $index;
            }
        }

        return -1;
    }

    /**
     * Adds a new filter element to the given filter array that represents a 'NOT IN'
     * condition on the given property.
     *
     * @param array $filter
     * @param string $property
     * @param array $values
     * @return array
     */
    private static function addNotInFilter($filter, $property, $values)
    {
        if (count($values) > 0) {
            $filter[] = [
                'property' => $property,
                'expression' => 'NOT IN',
                'value' => $values,
            ];
        }

        return $filter;
    }

    /**
     * Converts the purchase prices of the given article details (default currency) to the given currency.
     *
     * @param array $articleDetails
     * @param int $currencyId
     * @return array
     * @throws Exception
     */
    private function convertPurchasePricesIfNecessary($articleDetails, $currencyId)
    {
        /** @var Currency $defaultCurrency */
        $defaultCurrency = CurrencyUtil::getDefaultCurrency();
        if (!$defaultCurrency->getId() === $currencyId) {
            return $articleDetails;
        }

        /** @var Currency $supplierCurrency */
        $supplierCurrency = $this->get('models')->find(Currency::class, $currencyId);
        if (!$supplierCurrency) {
            return $articleDetails;
        }

        foreach ($articleDetails as &$articleDetail) {
            $articleDetail['purchasePrice'] = CurrencyUtil::convertAmountBetweenCurrencies(
                $articleDetail['purchasePrice'],
                $defaultCurrency,
                $supplierCurrency
            );
        }
        unset($articleDetail);

        return $articleDetails;
    }
}
