<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

require_once(__DIR__ . '/ViisonPickwareERPSupplierCommon.php');

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ManufacturerSupplierMapping;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier;
use Shopware\Models\Article\Article;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Article\Supplier as Manufacturer;
use Shopware\Models\Shop\Currency;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\Supplier\SupplierNumberGenerator;
use Shopware_Controllers_Backend_ViisonPickwareERPSupplierCommon as ViisonPickwareERPSupplierCommon;

/**
 * Backend controller for the supplier management module.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPSupplierManagement extends ViisonCommonBaseController
{

    /**
     * Responds a paginated, filtered and sorted list of suppliers.
     */
    public function getSupplierListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update prefixes of sort fields
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], 'supplier.') === false) {
                $sortField['property'] = 'supplier.' . $sortField['property'];
            }
        }

        // Unroll filters
        $filter = ViisonPickwareERPSupplierCommon::unrollIdBlacklistFilter($filter, 'articleDetailIdBlacklist', 'pickware_erp_article_detail_supplier_mappings', 'supplierId', 'supplier.id', 'articleDetailId');

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'supplier',
            'documentLocalizationSubShop',
            'currency'
        )->from(Supplier::class, 'supplier')
            ->leftJoin('supplier.documentLocalizationSubShop', 'documentLocalizationSubShop')
            ->leftJoin('supplier.currency', 'currency')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'supplier.name LIKE :searchQuery',
                    'supplier.contact LIKE :searchQuery',
                    'supplier.email LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Creates one or more suppliers using the POSTed data.
     */
    public function createSuppliersAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        $supplierNumberGenerator = $this->get('pickware.erp.supplier_number_generator_service');

        // Create a supplier for each given data array
        $newSuppliers = [];
        foreach ($data as $supplierData) {
            // Check documentLocalizationSubShop
            if ($supplierData['documentLocalizationSubShopId']) {
                $supplierData['documentLocalizationSubShop'] = $this->get('models')->find(Shop::class, $supplierData['documentLocalizationSubShopId']);
            } else {
                $supplierData['documentLocalizationSubShop'] = null;
            }
            unset($supplierData['documentLocalizationSubShopId']);

            $supplierData['currency'] = $this->get('models')->find(Currency::class, $supplierData['currencyId']);
            unset($supplierData['currencyId']);

            // Check supplier number
            if (isset($supplierData['number']) && !$supplierNumberGenerator->isSupplierNumberAvailable($supplierData['number'])) {
                $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_supplier_management/main');
                $nextFreeNumber = $supplierNumberGenerator->generateSupplierNumber();
                $this->View()->success = false;
                $this->View()->message = sprintf(
                    $namespace->get('notification/error/message/number_already_in_use'),
                    $supplierData['number'],
                    $nextFreeNumber
                );

                return;
            } elseif (!isset($supplierData['number'])) {
                $supplierData['number'] = $supplierNumberGenerator->generateSupplierNumber();
            }

            $supplier = new Supplier();
            $supplier->fromArray($supplierData);
            $this->get('models')->persist($supplier);
            $newSuppliers[] = $supplier;
        }

        // Save changes
        $this->get('models')->flush();

        // Set a filter parameter using the IDs of the newly created suppliers and add the data of the created suppliers
        // to the response
        $this->Request()->setParam('filter', [
            [
                'property' => 'supplier.id',
                'expression' => 'IN',
                'value' => array_map(
                    function ($supplier) {
                        return $supplier->getId();
                    },
                    $newSuppliers
                ),
            ],
        ]);
        $this->getSupplierListAction();
        $this->View()->success = true;
    }

    /**
     * Updates one or more existing suppliers using the POSTed data.
     */
    public function updateSuppliersAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        /** @var SupplierNumberGenerator $supplierNumberGenerator */
        $supplierNumberGenerator = $this->get('pickware.erp.supplier_number_generator_service');

        // Update all given suppliers with the respective data
        foreach ($data as $supplierData) {
            /** @var Supplier $supplier */
            $supplier = $this->get('models')->find(Supplier::class, $supplierData['id']);
            if (!$supplier) {
                continue;
            }

            $setNewSupplierNumber = isset($supplierData['number']) && $supplierData['number'] !== $supplier->getNumber();

            // Set or create supplier number. Return without success if a given number is already used.
            if ($setNewSupplierNumber && !$supplierNumberGenerator->isSupplierNumberAvailable($supplierData['number'])) {
                $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_supplier_management/main');
                $nextFreeNumber = $supplierNumberGenerator->generateSupplierNumber();
                $this->View()->success = false;
                $this->View()->message = sprintf(
                    $namespace->get('notification/error/message/number_already_in_use'),
                    $supplierData['number'],
                    $nextFreeNumber
                );

                return;
            } elseif ($setNewSupplierNumber) {
                $supplier->setNumber($supplierData['number']);
            }

            // Manually update the assigned article details
            if (isset($supplierData['articleDetails'])) {
                // Get the article detail IDs of the mappings
                $articleDetailData = [];
                foreach ($supplierData['articleDetails'] as $itemData) {
                    unset($itemData['articleDetail']);
                    $articleDetailData[$itemData['articleDetailId']] = $itemData;
                }
                unset($supplierData['articleDetails']);

                // Remove old mappings and update existing ones
                foreach ($supplier->getArticleDetailSupplierMappings() as $supplierArticleDetail) {
                    if (!$articleDetailData[$supplierArticleDetail->getArticleDetail()->getId()]) {
                        // Mapping was removed
                        $supplier->getArticleDetailSupplierMappings()->removeElement($supplierArticleDetail);
                        $this->get('models')->remove($supplierArticleDetail);
                    } else {
                        // Update mapping parameters
                        $supplierArticleDetail->fromArray($articleDetailData[$supplierArticleDetail->getArticleDetail()->getId()]);
                        unset($articleDetailData[$supplierArticleDetail->getArticleDetail()->getId()]);
                    }
                }

                // Add new mappings
                $articleDetails = $this->get('models')->getRepository(ArticleDetail::class)->findBy([
                    'id' => array_keys($articleDetailData),
                ]);
                foreach ($articleDetails as $articleDetail) {
                    $supplierArticleDetail = new ArticleDetailSupplierMapping($articleDetail, $supplier);
                    $this->get('models')->persist($supplierArticleDetail);
                    $itemData = $articleDetailData[$articleDetail->getId()];
                    unset($itemData['articleDetail']);
                    unset($itemData['supplier']);
                    $supplierArticleDetail->fromArray($itemData);
                }
            }

            // Manually update the assigned fabricators
            if (isset($supplierData['manufacturers'])) {
                // Get the manufacturer IDs of the mappings
                $manufacturerIds = array_map(
                    function ($item) {
                        return $item['manufacturerId'];
                    },
                    $supplierData['manufacturers']
                );
                unset($supplierData['manufacturers']);

                // Remove old mappings
                foreach ($supplier->getManufacturerSupplierMappings() as $fabricator) {
                    $idIndex = array_search($fabricator->getManufacturer()->getId(), $manufacturerIds);
                    if ($idIndex === false) {
                        // Mapping was removed
                        $supplier->getManufacturerSupplierMappings()->removeElement($fabricator);
                        $this->get('models')->remove($fabricator);
                    } else {
                        unset($manufacturerIds[$idIndex]);
                    }
                }

                // Add new mappings
                $newManufacturers = $this->get('models')->getRepository(Manufacturer::class)->findBy([
                    'id' => $manufacturerIds,
                ]);
                foreach ($newManufacturers as $manufacturer) {
                    $manufacturerMapping = new ManufacturerSupplierMapping($manufacturer, $supplier);
                    $this->get('models')->persist($manufacturerMapping);
                }
            }

            if ($supplierData['documentLocalizationSubShopId']) {
                $supplierData['documentLocalizationSubShop'] = $this->get('models')->find(Shop::class, $supplierData['documentLocalizationSubShopId']);
            } else {
                $supplierData['documentLocalizationSubShop'] = null;
            }
            unset($supplierData['documentLocalizationSubShopId']);

            $supplierData['currency'] = $this->get('models')->find(Currency::class, $supplierData['currencyId']);
            unset($supplierData['currencyId']);

            // Update all normal fields
            $supplier->fromArray($supplierData);
        }

        // Save changes
        $this->get('models')->flush();

        // Set a filter parameter using the IDs of the updated suppliers and add the updated supplier data to the response
        $this->Request()->setParam('filter', [
            [
                'property' => 'supplier.id',
                'expression' => 'IN',
                'value' => array_map(
                    function ($supplierData) {
                        return $supplierData['id'];
                    },
                    $data
                ),
            ],
        ]);
        $this->getSupplierListAction();
        $this->View()->success = true;
    }

    /**
     * Deletes the POSTed suppliers.
     */
    public function deleteSuppliersAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Remove all given suppliers
        foreach ($data as $supplierData) {
            $supplier = $this->get('models')->find(Supplier::class, $supplierData['id']);
            if ($supplier) {
                $this->get('models')->remove($supplier);
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Responds a paginated, filtered and sorted list of supplier article detail assignment.
     */
    public function getSupplierArticleDetailListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update sort fields
        foreach ($sort as &$sortField) {
            switch ($sortField['property']) {
                case 'defaultSupplier':
                case 'minimumOrderAmount':
                case 'packingUnit':
                case 'purchasePrice':
                case 'supplierArticleNumber':
                    $sortField['property'] = 'supplierMapping.' . $sortField['property'];
                    break;
                case 'orderNumber':
                    $sortField['property'] = 'articleDetail.number';
                    break;
                case 'name':
                    $sortField['property'] = 'article.' . $sortField['property'];
                    break;
                case 'manufacturerName':
                    $sortField['property'] = 'manufacturer.name';
                    break;
                case 'phone':
                case 'email':
                case 'contact':
                    $sortField['property'] = 'supplier.' . $sortField['property'];
                    break;
            }
        }

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'supplierMapping',
            'articleDetail',
            'article',
            'attribute',
            'supplier',
            'currency',
            'manufacturer'
        )->from(ArticleDetailSupplierMapping::class, 'supplierMapping')
            ->leftJoin('supplierMapping.supplier', 'supplier')
            ->leftJoin('supplier.currency', 'currency')
            ->leftJoin('supplierMapping.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'attribute')
            ->leftJoin('article.supplier', 'manufacturer')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'article.name LIKE :searchQuery',
                    'article.description LIKE :searchQuery',
                    'articleDetail.number LIKE :searchQuery',
                    'manufacturer.name LIKE :searchQuery',
                    'supplier.name LIKE :searchQuery',
                    'articleDetail.supplierNumber LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        // Fetch additionalTexts
        $articleDetailIds = array_map(function ($articleDetailSupplierMapping) {
            return $articleDetailSupplierMapping['articleDetailId'];
        }, $result);
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts($articleDetailIds);

        // Format the results
        foreach ($result as &$articleDetailSupplierMapping) {
            $additionalText = $additionalTexts[$articleDetailSupplierMapping['articleDetail']['id']];
            $additionalText = ($additionalText) ? (' - ' . $additionalText) : '';

            $articleDetail = $articleDetailSupplierMapping['articleDetail'];
            $articleDetailSupplierMapping['articleDetail'] = [
                'id' => $articleDetail['id'],
                'articleId' => $articleDetail['articleId'],
                'name' => $articleDetail['article']['name'] . $additionalText,
                'orderNumber' => $articleDetail['number'],
                'manufacturerName' => $articleDetail['article']['supplier']['name'],
                'manufacturerArticleNumber' => $articleDetail['supplierNumber'],
            ];
            $articleDetailSupplierMapping['currency'] = $articleDetailSupplierMapping['supplier']['currency'];
        }
        unset($articleDetailSupplierMapping);

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Creates one or more supplier article detail assignments using the POSTed data.
     */
    public function createSupplierArticleDetailsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Create a supplier article detail for each given data array
        foreach ($data as $articleDetailMappingData) {
            $articleDetail = $this->get('models')->find(
                ArticleDetail::class,
                $articleDetailMappingData['articleDetailId']
            );
            $supplier = $this->get('models')->find(Supplier::class, $articleDetailMappingData['supplierId']);
            if (!$articleDetail || !$supplier) {
                continue;
            }
            unset($articleDetailMappingData['articleDetail']);
            unset($articleDetailMappingData['supplier']);
            $articleDetailMapping = new ArticleDetailSupplierMapping($articleDetail, $supplier);
            $this->get('models')->persist($articleDetailMapping);
            $articleDetailMapping->fromArray($articleDetailMappingData);
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Updates one or more existing supplier article detail assignments using the POSTed data.
     */
    public function updateSupplierArticleDetailsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Update all given supplier article detail assignments with the respective data
        foreach ($data as $supplierArticleDetailData) {
            $supplierArticleDetail = $this->get('models')->getRepository(ArticleDetailSupplierMapping::class)->findOneBy([
                'supplierId' => $supplierArticleDetailData['supplierId'],
                'articleDetailId' => $supplierArticleDetailData['articleDetailId'],
            ]);
            if ($supplierArticleDetail) {
                unset($supplierArticleDetailData['id']);
                unset($supplierArticleDetailData['supplier']);
                unset($supplierArticleDetailData['articleDetail']);
                $supplierArticleDetail->fromArray($supplierArticleDetailData);
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Deletes the POSTed supplier article detail assignments.
     */
    public function deleteSupplierArticleDetailsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Remove all given supplier article detail assignments
        foreach ($data as $supplierArticleDetailData) {
            $supplierArticleDetail = $this->get('models')->getRepository(ArticleDetailSupplierMapping::class)->findOneBy([
                'supplierId' => $supplierArticleDetailData['supplierId'],
                'articleDetailId' => $supplierArticleDetailData['articleDetailId'],
            ]);
            if ($supplierArticleDetail) {
                $this->get('models')->remove($supplierArticleDetail);
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Responds a paginated, filtered and sorted list of supplier fabricator assignment.
     */
    public function getSupplierFabricatorListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update sort fields
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], 'manufacturer.') === false) {
                $sortField['property'] = 'manufacturer.' . $sortField['property'];
            }
        }

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'manufacturerMapping',
            'supplier',
            'manufacturer'
        )->from(ManufacturerSupplierMapping::class, 'manufacturerMapping')
            ->leftJoin('manufacturerMapping.supplier', 'supplier')
            ->leftJoin('manufacturerMapping.manufacturer', 'manufacturer')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'manufacturer.name LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Since a supplier needs both a locale and sub shop id to create supplier documents, this function fetches all
     * available sub shops (that also carry a locale reference) and returns their ids as well as a formatted displayName
     * for the backend user.
     */
    public function getSupplierDocumentLocalizationSubShopListAction()
    {
        $shops = $this->get('models')->createQueryBuilder()
            ->select([
                'shop',
                'locale',
            ])
            ->from(Shop::class, 'shop')
            ->leftJoin('shop.locale', 'locale')
            ->orderBy('locale.id', 'ASC')
            ->getQuery()
            ->getResult();

        // Add a default "empty" localization without a shop id
        $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_supplier_management/main');
        $documentLocalizationSubShops = [
            [
                'id' => 0,
                'displayName' => $namespace->get('supplier/field/languageAndTemplate/defaultEmpty'),
            ],
        ];
        foreach ($shops as $shop) {
            $locale = $shop->getLocale();
            // Show language name as Shopware does it as well as the sub shop name
            $displayName = sprintf('%s (%s), %s', $locale->getLanguage(), $locale->getTerritory(), $shop->getName());
            $documentLocalizationSubShops[] = [
                'id' => $shop->getId(),
                'displayName' => $displayName,
            ];
        }

        $this->View()->assign([
            'success' => true,
            'data' => $documentLocalizationSubShops,
        ]);
    }

    /**
     * Creates one or more supplier fabricator assignments using the POSTed data.
     */
    public function createSupplierFabricatorsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Create a supplier fabricator for each given data array
        foreach ($data as $manufacturerMappingData) {
            $manufacturer = $this->get('models')->find(Manufacturer::class, $manufacturerMappingData['manufacturerId']);
            $supplier = $this->get('models')->find(Supplier::class, $manufacturerMappingData['supplierId']);
            if (!$manufacturer || !$supplier) {
                continue;
            }
            unset($manufacturerMappingData['manufacturer']);
            unset($manufacturerMappingData['supplier']);
            $manufacturerMapping = new ManufacturerSupplierMapping($manufacturer, $supplier);
            $this->get('models')->persist($manufacturerMapping);
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Updates one or more existing supplier fabricator assignments using the POSTed data.
     */
    public function updateSupplierFabricatorsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Update all given supplier fabricator assignments with the respective data
        foreach ($data as $supplierFabricatorData) {
            $supplierFabricator = $this->get('models')->getRepository(ManufacturerSupplierMapping::class)->findOneBy([
                'supplierId' => $supplierFabricatorData['supplierId'],
                'manufacturerId' => $supplierFabricatorData['manufacturerId'],
            ]);
            if ($supplierFabricator) {
                unset($supplierFabricatorData['id']);
                unset($supplierFabricatorData['supplier']);
                unset($supplierFabricatorData['manufacturer']);
                $supplierFabricator->fromArray($supplierFabricatorData);
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Deletes the POSTed supplier fabricator assignments.
     */
    public function deleteSupplierFabricatorsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Remove all given supplier fabricator assignments
        foreach ($data as $supplierFabricatorData) {
            $supplierFabricator = $this->get('models')->getRepository(ManufacturerSupplierMapping::class)->findOneBy([
                'supplierId' => $supplierFabricatorData['supplierId'],
                'manufacturerId' => $supplierFabricatorData['manufacturerId'],
            ]);
            if ($supplierFabricator) {
                $this->get('models')->remove($supplierFabricator);
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Creates new supplier article details mappings for all articles, whose fabricator is
     * also mapped to the supplier with the given ID.
     */
    public function assignFabricatorArticlesAction()
    {
        // Try to find the supplier
        $supplierId = $this->Request()->getParam('id', 0);
        $supplier = $this->get('models')->find(Supplier::class, $supplierId);
        if (!$supplier) {
            $this->View()->success = false;

            return;
        }

        // Get already assigned articles
        $currentAssignments = $supplier->getArticleDetailSupplierMappings()->map(function ($articleDetailMapping) {
            return $articleDetailMapping->getArticleDetail()->getId();
        });

        // Assign all article details to the supplier, which are also assigned to one of the assigned fabricators
        foreach ($supplier->getManufacturerSupplierMappings() as $fabricatorMapping) {
            $articles = $this->get('models')->getRepository(Article::class)->findBy([
                'supplier' => $fabricatorMapping->getManufacturer()->getId(),
            ]);
            foreach ($articles as $article) {
                foreach ($article->getDetails() as $articleDetail) {
                    if (in_array($articleDetail->getId(), $currentAssignments->getValues())) {
                        continue;
                    }
                    $supplierArticleDetail = new ArticleDetailSupplierMapping($articleDetail, $supplier);
                    $this->get('models')->persist($supplierArticleDetail);
                    $supplierArticleDetail->setPurchasePrice($articleDetail->getPurchasePrice());
                }
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Checks for a currency change of the given supplier and currency.
     *
     * Checks if the given currency id is different from the currency id of the given supplier and returns the number
     * of existing numberOfArticleDetailMappings that may be affected by this currency change.
     */
    public function checkForCurrencyChangeAction()
    {
        $supplierId = (int) $this->Request()->getParam('id', 0);
        $newCurrencyId = (int) $this->Request()->getParam('currencyId', 0);

        /** @var Supplier $supplier */
        $supplier = $this->get('models')->find(Supplier::class, $supplierId);
        if (!$supplier) {
            throw new RuntimeException('Supplier not found.');
        }

        if ($supplier->getCurrency()->getId() === $newCurrencyId) {
            $this->View()->assign([
                'success' => true,
                'isCurrencyChange' => false,
                'numberOfArticleDetailMappings' => 0,
            ]);

            return;
        }

        $articleDetailMappings = $supplier->getArticleDetailSupplierMappings()->toArray();
        $this->View()->assign([
            'success' => true,
            'isCurrencyChange' => true,
            'numberOfArticleDetailMappings' => count($articleDetailMappings),
        ]);
    }

    /**
     * Converts (updates) all purchase prices from article detail mappings from the given supplier to the new currency.
     */
    public function convertPurchasePricesAction()
    {
        $supplierId = $this->Request()->getParam('id', 0);
        $newCurrencyId = $this->Request()->getParam('currencyId', 0);

        /** @var Supplier $supplier */
        $supplier = $this->get('models')->find(Supplier::class, $supplierId);
        if (!$supplier) {
            throw new RuntimeException('Supplier not found.');
        }

        if ($supplier->getCurrency()->getId() === $newCurrencyId) {
            $this->View()->success = true;

            return;
        }

        /** @var Currency $newCurrency */
        $newCurrency = $this->get('models')->find(Currency::class, $newCurrencyId);
        if (!$newCurrency) {
            throw new RuntimeException('Currency not found.');
        }

        // This refresh is necessary because the currency proxy of the supplier model would not contain a factor.
        $this->get('models')->refresh($supplier->getCurrency());
        $changedEntities = [];
        /** @var ArticleDetailSupplierMapping $articleDetailMapping */
        foreach ($supplier->getArticleDetailSupplierMappings()->toArray() as &$articleDetailMapping) {
            if ($articleDetailMapping->getPurchasePrice()) {
                $articleDetailMapping->setPurchasePrice(
                    CurrencyUtil::convertAmountBetweenCurrencies(
                        $articleDetailMapping->getPurchasePrice(),
                        $supplier->getCurrency(),
                        $newCurrency
                    )
                );
                $changedEntities[] = $articleDetailMapping;
            }
        }
        unset($articleDetailMapping);
        $this->get('models')->flush($changedEntities);

        $this->View()->success = true;
    }

    /**
     * Assigns data for a new supplier to the view. This information is used when creating a new supplier.
     */
    public function getDefaultSupplierDataAction()
    {
        /** @var SupplierNumberGenerator $supplierNumberGenerator */
        $supplierNumberGenerator = $this->get('pickware.erp.supplier_number_generator_service');
        $defaultCurrency = CurrencyUtil::getDefaultCurrency();

        $this->View()->assign([
            'success' => true,
            'unusedSupplierNumber' => $supplierNumberGenerator->generateSupplierNumber(),
            'defaultCurrencyId' => $defaultCurrency->getId(),
        ]);
    }
}
