<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

require_once(__DIR__ . '/ViisonPickwareERPSupplierCommon.php');

use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\CSRFWhitelistAware;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\BackendExceptionHandling;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\ArticleDetailSupplierMapping;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrder;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderAttachment;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItemStatus;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderStatus;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Media\Album as MediaAlbum;
use Shopware\Models\Media\Media;
use Shopware\Models\Shop\Currency;
use Shopware\Models\Shop\Shop;
use Shopware\Plugins\ViisonCommon\Classes\Document\PaperLayout;
use Shopware\Plugins\ViisonCommon\Classes\Document\RenderedDocument\RenderedDocumentWithDocumentType;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\CurrencyException;
use Shopware\Plugins\ViisonCommon\Classes\TranslationServiceFactory;
use Shopware\Plugins\ViisonCommon\Classes\Util\Currency as CurrencyUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\DocumentRenderingContextFactoryService;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\PluginConfig\PluginConfig;
use Shopware_Components_Translation as Translator;

/**
 * Backend controller for the supplier orders module.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPSupplierOrders extends Shopware_Controllers_Backend_ExtJs implements CSRFWhitelistAware
{
    use BackendExceptionHandling;

    /**
     * @var Translator $translator
     */
    protected $translator;

    /**
     * Disables the renderer and output buffering for all 'downloadOrderPDF' requests
     * to be able to display PDFs as response.
     */
    public function init()
    {
        parent::init();
        $this->translator = TranslationServiceFactory::createTranslationService();
        if (in_array($this->Request()->getActionName(), ['downloadOrderPDF', 'downloadOrderCSV'])) {
            Shopware()->Plugins()->Controller()->ViewRenderer()->setNoRender();
            $this->Front()->setParam('disableOutputBuffering', true);
        }
    }

    /**
     * @inheritdoc
     */
    public function getWhitelistedCSRFActions()
    {
        return [
            'downloadOrderPDF',
            'downloadOrderCSV',
        ];
    }

    /**
     * Don't use the JSON renderer for the 'exportInventory' action.
     */
    public function preDispatch()
    {
        if (!in_array($this->Request()->getActionName(), ['index', 'load', 'skeleton', 'extends', 'downloadOrderPDF', 'downloadOrderCSV'])) {
            $this->Front()->Plugins()->Json()->setRenderer();
        }
    }

    /**
     * Passes the ID of the supplier order attachments media album to the view.
     */
    public function loadAction()
    {
        // Find the supplier order attachments album
        $album = $this->get('models')->getRepository(MediaAlbum::class)->findOneBy([
            'name' => 'Pickware Lieferantenbestellungen',
        ]);
        if (!$album) {
            return;
        }

        $this->View()->assign([
            'viisonSupplierOrderAttachmentsAlbumId' => $album->getId(),
        ]);
    }

    /**
     * Responds with a list of all available warehouses including a
     * "all online available warehouses" group option.
     */
    public function getSourceWarehouseListAction()
    {
        $snippets = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_supplier_orders/detail/configuration');

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select('warehouse')->from(Warehouse::class, 'warehouse');
        $warehouses = $builder->getQuery()->getArrayResult();

        $warehousesWhoseStockIsAvailableOnline = [];
        foreach ($warehouses as &$warehouse) {
            if ($warehouse['stockAvailableForSale']) {
                $warehousesWhoseStockIsAvailableOnline[] = $warehouse;
            }

            $warehouse['displayName'] = sprintf(
                '%s (%s) %s',
                $warehouse['name'],
                $warehouse['code'],
                $warehouse['stockAvailableForSale'] ? ' - ' . $snippets->get('source_warehouses/online_warehouse/postfix') : ''
            );
        }
        unset($warehouse);

        // Add option "all online available warehouses" (if any available)
        if (count($warehousesWhoseStockIsAvailableOnline) > 0) {
            $warehouseCodes = implode(', ', array_map(function (array $warehouse) {
                return $warehouse['code'];
            }, $warehousesWhoseStockIsAvailableOnline));

            // Add option to the beginning of the result
            array_unshift($warehouses, [
                'id' => 0,
                'displayName' => sprintf(
                    '%s (%s)',
                    $snippets->get('source_warehouses/all_online_warehouses/item_name'),
                    $warehouseCodes
                ),
            ]);
        }

        $this->View()->assign([
            'success' => true,
            'data' => $warehouses,
            'total' => count($warehouses),
        ]);
    }

    /**
     * Responds a paginated, filtered and sorted list of supplier orders.
     */
    public function getOrderListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $order = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update prefixes of sort fields
        foreach ($order as &$sortField) {
            if (mb_strpos($sortField['property'], '.') === false) {
                $sortField['property'] = 'supplierOrder.' . $sortField['property'];
            }
        }

        // Build the main query
        /** @var \Shopware\Components\Model\QueryBuilder $builder */
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'supplierOrder',
            'supplier',
            'attachments',
            'items',
            'articleDetail',
            'article',
            'attribute',
            'manufacturer',
            'warehouse',
            'currency'
        )->from(SupplierOrder::class, 'supplierOrder')
            ->leftJoin('supplierOrder.supplier', 'supplier')
            ->leftJoin('supplierOrder.warehouse', 'warehouse')
            ->leftJoin('supplierOrder.attachments', 'attachments')
            ->leftJoin('supplierOrder.items', 'items')
            ->leftJoin('supplierOrder.currency', 'currency')
            ->leftJoin('items.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'attribute')
            ->leftJoin('article.supplier', 'manufacturer')
            ->addFilter($filter)
            ->addOrderBy($order)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'supplierOrder.orderNumber LIKE :searchQuery',
                    'supplierOrder.comment LIKE :searchQuery',
                    'supplier.name LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);

        $paginator = new Paginator($query);
        $total = $paginator->count();
        $data = $paginator->getIterator()->getArrayCopy();

        // Convert the total of each supplier order to the default currency
        $defaultCurrency = CurrencyUtil::getDefaultCurrency();
        foreach ($data as &$supplierOrder) {
            // Use a temporary currency entity to make use of the CurrencyUtil function.
            $tempSupplierCurrency = new Currency();
            $tempSupplierCurrency->fromArray($supplierOrder['currency']);
            $supplierOrder['totalInDefaultCurrency'] = CurrencyUtil::convertAmountBetweenCurrencies(
                $supplierOrder['total'],
                $tempSupplierCurrency,
                $defaultCurrency
            );
        }
        unset($supplierOrder);

        $this->View()->assign([
            'success' => true,
            'data' => $data,
            'total' => $total,
        ]);
    }

    /**
     * Creates one or more orders using the POSTed data.
     */
    public function createOrdersAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Create an order for each given data array
        $newOrders = [];
        $changedEntities = [];
        foreach ($data as $orderData) {
            // Remove supplier, warehouse and article data since it cannot be mapped automatically
            $supplier = $orderData['supplier'];
            unset($orderData['supplier']);
            $warehouseId = $orderData['warehouseId'];
            unset($orderData['warehouseId']);
            unset($orderData['warehouse']);
            $articles = $orderData['items'];
            unset($orderData['items']);
            unset($orderData['created']);

            // Create the main order
            $order = new SupplierOrder();
            $order->fromArray($orderData);
            $order->setStatus($this->findOrderStatusByName(SupplierOrderStatus::OPEN));
            $order->setOrderNumber($this->getNextOrderNumber());
            $order->setUser(ViisonCommonUtil::getCurrentUser());

            // Sets supplier
            if ($supplier && $supplier['id']) {
                /** @var Supplier $supplier */
                $supplier = $this->get('models')->find(Supplier::class, $supplier['id']);
                $order->setSupplier($supplier);
                $order->setCurrency($supplier->getCurrency());
            }

            // Set warehouse
            if ($warehouseId) {
                $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
                $order->setWarehouse($warehouse);
            }

            // Set the (expected) delivery date
            if ($order->getSupplier() !== null && $order->getSupplier()->getDeliveryTime() > 0) {
                $now = new DateTime();
                $deliveryDate = $now->add(new DateInterval('P' . $order->getSupplier()->getDeliveryTime() . 'D'));
                $order->setDeliveryDate($deliveryDate);
            }

            // Add the order items
            foreach ($articles as $articleData) {
                $articleData['status'] = $this->get('models')->find(
                    SupplierOrderItemStatus::class,
                    $articleData['statusId']
                );

                // Try to find the pre-assigned order article
                $orderArticle = $this->get('models')->getRepository(SupplierOrderItem::class)->findOneBy([
                    'supplierOrderId' => null,
                    'articleDetailId' => $articleData['articleDetailId'],
                ]);
                if ($orderArticle) {
                    // Add pre-assigned article
                    $orderArticle->setSupplierOrder($order);
                } else {
                    // Create and add new article
                    $orderArticle = $this->addItemToOrder($articleData, $order);
                }
                $changedEntities[] = $orderArticle;
            }

            // Recompute the cached order total
            $order->recomputeTotal();

            $this->get('models')->persist($order);
            $changedEntities[] = $order;
            $newOrders[] = $order;
        }

        // Save changes
        $this->get('models')->flush($changedEntities);

        // Set a filter parameter using the IDs of the newly created orders and add the data of the created orders
        // to the response
        $this->Request()->setParam('filter', [
            [
                'property' => 'supplierOrder.id',
                'expression' => 'IN',
                'value' => array_map(
                    function ($order) {
                        return $order->getId();
                    },
                    $newOrders
                ),
            ],
        ]);
        $this->getOrderListAction();
        $this->View()->success = true;
    }

    /**
     * Updates one or more existing orders using the POSTed data. That is, updates the base order
     * date and adds/updates/removes its order items and attachments.
     */
    public function updateOrdersAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }
        $defaultCurrency = CurrencyUtil::getDefaultCurrency();

        // Update all given orders with the respective data
        $changedEntities = [];
        foreach ($data as $orderData) {
            /** @var SupplierOrder $order */
            $order = $this->get('models')->find(SupplierOrder::class, $orderData['id']);
            if (!$order) {
                continue;
            }

            // Remove supplier, currency, warehouse, article and attachment data since it cannot be mapped automatically
            unset($orderData['supplier']);
            unset($orderData['currency']);
            $warehouseId = $orderData['warehouseId'];
            unset($orderData['warehouseId'], $orderData['warehouse']);
            $articles = $orderData['items'];
            unset($orderData['items']);
            $attachments = $orderData['attachments'];
            unset($orderData['attachments']);

            // Update the order
            $orderData['status'] = $this->get('models')->find(SupplierOrderStatus::class, $orderData['statusId']);
            $order->fromArray($orderData);
            $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
            $order->setWarehouse($warehouse);

            // Check for order items that are not contained in the POSTed data anymore and remove them from the order
            $remainingItemIds = array_column($articles, 'id');
            foreach ($order->getItems() as $item) {
                if (!in_array($item->getId(), $remainingItemIds)) {
                    // Remove item from order and its associated stock entries
                    $order->getItems()->removeElement($item);
                }
            }

            // Create new and update existing items
            foreach ($articles as $articleData) {
                $articleData['status'] = $this->get('models')->find(
                    SupplierOrderItemStatus::class,
                    $articleData['statusId']
                );

                // Try to find the order item by article detail id (not by supplier order item id). This way we make
                // sure that no article detail is added to the same order multiple times.
                $orderItem = $this->get('models')->getRepository(SupplierOrderItem::class)->findOneBy([
                    'supplierOrderId' => $orderData['id'],
                    'articleDetailId' => $articleData['articleDetailId'],
                ]);
                if ($orderItem) {
                    $oldDeliveredQuantity = $orderItem->getDeliveredQuantity();
                    $orderItem->fromArray($articleData);
                } else {
                    // Item does not exist yet, hence create it
                    $orderItem = $this->addItemToOrder($articleData, $order);
                    $oldDeliveredQuantity = 0;

                    // We need to persist the new order item so that stock entries can be generated below that reference it
                    $this->get('models')->flush($orderItem);
                }

                if ($orderItem->getDeliveryTime() === null && $articleData['status']->getName() === SupplierOrderItemStatus::COMPLETELY_RECEIVED) {
                    // Calculate and set the delivery time
                    $deliveryTime = self::getDaysSince($order->getCreated());
                    $orderItem->setDeliveryTime($deliveryTime);
                }
                $changedEntities[] = $orderItem;

                $deliveredQuantityChange = $articleData['deliveredQuantity'] - $oldDeliveredQuantity;
                if ($deliveredQuantityChange !== 0 && $orderItem->getArticleDetail()) {
                    // Log stock changes according to the changed delivered quantity
                    $stockChanges = $this->get('pickware.erp.stock_change_list_factory_service')->createStockChangeList(
                        $order->getWarehouse(),
                        $orderItem->getArticleDetail(),
                        $deliveredQuantityChange
                    );

                    if ($deliveredQuantityChange > 0) {
                        if ($order->getCurrency()->getId() !== $defaultCurrency->getId()) {
                            // Convert the purchase price of the order item, so that all stock entries are in the
                            // default currency and add comment to describe this conversion.
                            $price = CurrencyUtil::convertAmountBetweenCurrencies(
                                $orderItem->getPrice(),
                                $order->getCurrency(),
                                $defaultCurrency
                            );
                            $namespace = $this->get('snippets')->getNamespace('backend/viison_pickware_erp_supplier_orders/main');
                            $message = sprintf(
                                $namespace->get('currency/conversion/stock_entry_comment'),
                                CurrencyUtil::getFormattedPriceStringByCurrency($orderItem->getPrice(), $order->getCurrency()),
                                CurrencyUtil::getFormattedPriceStringByCurrency($price, $defaultCurrency)
                            );
                            $this->get('pickware.erp.stock_ledger_service')->recordPurchasedStock(
                                $orderItem->getArticleDetail(),
                                $stockChanges,
                                $price,
                                $message,
                                $orderItem
                            );
                        } else {
                            // No currency conversion or comment necessary
                            $this->get('pickware.erp.stock_ledger_service')->recordPurchasedStock(
                                $orderItem->getArticleDetail(),
                                $stockChanges,
                                $orderItem->getPrice(),
                                null,
                                $orderItem
                            );
                        }
                    } else {
                        // Outgoing stock, hence log an 'outgoing' stock entry
                        $this->get('pickware.erp.stock_ledger_service')->recordOutgoingStock(
                            $orderItem->getArticleDetail(),
                            $stockChanges
                        );
                    }
                }
            }

            // Check for attachments that are not contained in the POSTed data anymore and remove them from the order
            $remainingAttachmentIds = $attachments !== null ? array_column($attachments, 'id') : [];
            foreach ($order->getAttachments() as $attachment) {
                if (!in_array($attachment->getId(), $remainingAttachmentIds)) {
                    $order->getAttachments()->removeElement($attachment);
                }
            }

            // Create new attachments
            foreach ($attachments as $attachmentData) {
                // Try to find the order attachment
                foreach ($order->getAttachments() as $attachment) {
                    if ($attachment->getMedia()->getId() === $attachmentData['mediaId']) {
                        break 2;
                    }
                }

                $media = $this->get('models')->find(Media::class, $attachmentData['mediaId']);
                if (!$media) {
                    continue;
                }

                // Create and add the new attachment
                $attachment = new SupplierOrderAttachment($order, $media);
                $this->get('models')->persist($attachment);
                $attachment->fromArray($attachmentData);
                $changedEntities[] = $attachment;
            }

            // Recompute the cached order total
            $order->recomputeTotal();
            $changedEntities[] = $order;
        }

        // Save changes
        $this->get('models')->flush($changedEntities);

        $this->View()->success = true;
    }

    /**
     * Deletes the POSTed orders.
     */
    public function deleteOrdersAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Remove all given orders
        $changedEntities = [];
        foreach ($data as $orderData) {
            $order = $this->get('models')->find(SupplierOrder::class, $orderData['id']);
            if ($order) {
                $this->get('models')->remove($order);
                $changedEntities[] = $order;
            }
        }

        // Save changes
        $this->get('models')->flush($changedEntities);

        $this->View()->success = true;
    }

    /**
     * Responds a filtered and sorted list of supplier order articles.
     * Note that this is list not paginated i.e. it uses no limit on its results.
     */
    public function getOrderArticleListAction()
    {
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'orderArticle',
            'articleDetail',
            'attribute'
        )->from(SupplierOrderItem::class, 'orderArticle')
            ->leftJoin('orderArticle.supplierOrder', 'supplierOrder')
            ->leftJoin('orderArticle.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'attribute')
            ->leftJoin('article.supplier', 'manufacturer')
            ->addFilter($filter);

        // Check for the sort parameter. We allow sorting by supplier (supplierId) view-wise, but not in this controller
        // (i.e. when refreshing the store when using the search bar), since the supplier is not part of this query.
        $sortContainsSupplier = false;
        foreach ($sort as $sortCondition) {
            if ($sortCondition['property'] === 'supplierId') {
                $sortContainsSupplier = true;
                break;
            }
        }
        if (!$sortContainsSupplier) {
            $builder->addOrderBy($sort);
        }

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere(
                $builder->expr()->orX(
                    'article.name LIKE :searchQuery',
                    'article.description LIKE :searchQuery',
                    'articleDetail.number LIKE :searchQuery',
                    'manufacturer.name LIKE :searchQuery',
                    'articleDetail.supplierNumber LIKE :searchQuery'
                )
            )->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $result = $builder->getQuery()->getArrayResult();

        // Format the results
        foreach ($result as &$orderArticle) {
            $orderArticle['orderNumber'] = $orderArticle['articleDetail']['number'];
            // Add supplier information
            $orderArticle['suppliers'] = [];
            $builder = $this->get('models')->createQueryBuilder();
            if ($orderArticle['supplierOrderId']) {
                // Add the supplier Id of the order
                $builder->select('supplier')
                        ->from(Supplier::class, 'supplier')
                        ->leftJoin('supplier.supplierOrders', 'supplierOrders')
                        ->where('supplierOrders.id = :supplierOrderId')
                        ->setParameter('supplierOrderId', $orderArticle['supplierOrderId']);
                $supplier = $builder->getQuery()->getOneOrNullResult();
                $orderArticle['supplierId'] = ($supplier) ? $supplier->getId() : null;
            } else {
                // Add all suppliers that are mapped to the article
                $builder->select('supplierMapping')
                        ->from(ArticleDetailSupplierMapping::class, 'supplierMapping')
                        ->where('supplierMapping.articleDetailId = :articleDetailId')
                        ->setParameter('articleDetailId', $orderArticle['articleDetailId'])
                        ->orderBy('supplierMapping.defaultSupplier', 'DESC');
                $suppliers = $builder->getQuery()->getArrayResult();

                if (isset($suppliers[0]['id'])) {
                    // Auto-selects the first supplier that is mapped to this article detail,
                    // which is the default supplier, since they are already ordered so
                    // that the default supplier is first in the list.
                    $orderArticle['supplierId'] = $suppliers[0]['supplierId'];
                }
                $orderArticle['suppliers'] = $suppliers;
            }

            // Add 'instock' as 'onlineAvailableStock' for better understanding
            $orderArticle['onlineAvailableStock'] = $orderArticle['articleDetail']['inStock'];
            // Add 'article.pickware_physical_stock_for_sale' as 'onlinePhysicalStock' for better understanding
            $orderArticle['onlinePhysicalStock'] = $orderArticle['articleDetail']['attribute']['pickwarePhysicalStockForSale'];

            // Remove unnecessary attribute information from the result
            unset($orderArticle['articleDetail']['attribute']);
        }

        $this->View()->assign([
            'success' => true,
            'data' => $result,
        ]);
    }

    /**
     * Creates one or more supplier order articles using the POSTed data.
     */
    public function createOrderArticlesAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Create an order article for each given data array
        $changedEntities = [];
        foreach ($data as $orderArticleData) {
            // Remove some data fields
            unset($orderArticleData['id']);

            $articleDetail = $this->get('models')->find(ArticleDetail::class, $orderArticleData['articleDetailId']);
            if (!$articleDetail) {
                continue;
            }
            if ($orderArticleData['statusId'] !== null) {
                $orderArticleData['status'] = $this->get('models')->find(
                    SupplierOrderItemStatus::class,
                    $orderArticleData['statusId']
                );
            }

            // Create the order article
            $orderArticle = new SupplierOrderItem(null, $articleDetail);
            $orderArticle->fromArray($orderArticleData);
            $this->get('models')->persist($orderArticle);
            $changedEntities[] = $orderArticle;
        }

        // Save changes
        $this->get('models')->flush($changedEntities);

        $this->View()->success = true;
    }

    /**
     * Updates one or more existing supplier order articles using the POSTed data.
     */
    public function updateOrderArticlesAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Update all given order articles with the respective data
        $changedEntities = [];
        foreach ($data as $orderArticleData) {
            $orderArticle = $this->get('models')->getRepository(SupplierOrderItem::class)->findOneBy([
                'supplierOrderId' => $orderArticleData['supplierOrderId'],
                'articleDetailId' => $orderArticleData['articleDetailId'],
            ]);
            if (!$orderArticle) {
                continue;
            }

            // Remove some data fields
            unset($orderArticleData['id']);

            // Update the order article
            if ($orderArticleData['statusId'] !== null) {
                $orderArticleData['status'] = $this->get('models')->find(
                    SupplierOrderItemStatus::class,
                    $orderArticleData['statusId']
                );
            }
            $orderArticle->fromArray($orderArticleData);
            $changedEntities[] = $orderArticle;
        }

        // Save changes
        $this->get('models')->flush($changedEntities);

        $this->View()->success = true;
    }

    /**
     * Checks whether the given article details have ArticleDetailSupplierMapping that do not exist in the database.
     */
    public function checkArticleDetailsForChangedArticleDetailSupplierMappingsAction()
    {
        $articleDetailsData = $this->Request()->getPost();
        try {
            ParameterValidator::assertIsArray($articleDetailsData, 'data');

            $hasNewArticleDetailSupplierMappings = false;

            foreach ($articleDetailsData as $articleDetailData) {
                $articleDetailSupplierMapping = $this->get('models')->getRepository(ArticleDetailSupplierMapping::class)->findOneBy([
                    'supplierId' => $articleDetailData['supplierId'],
                    'articleDetailId' => $articleDetailData['id'],
                ]);

                $articleDetailData['price'] = sprintf('%.2f', $articleDetailData['price']);
                if (!$articleDetailSupplierMapping || $articleDetailSupplierMapping->getPurchasePrice() !== $articleDetailData['price']) {
                    $hasNewArticleDetailSupplierMappings = true;

                    break;
                }
            }

            $this->View()->assign(
                [
                    'hasNewArticleDetailSupplierMappings' => $hasNewArticleDetailSupplierMappings,
                    'success' => true,
                ]
            );
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * Creates or updates the ArticleDetailSupplierMapping for given supplier article details.
     */
    public function saveArticleDetailSupplierMappingsForArticleDetailsAction()
    {
        $articleDetailsData = $this->Request()->getPost();
        try {
            ParameterValidator::assertIsArray($articleDetailsData, 'data');

            foreach ($articleDetailsData as $articleDetailData) {
                // Update/Create supplier->article mapping
                $articleDetailSupplierMapping = $this->get('models')->getRepository(ArticleDetailSupplierMapping::class)->findOneBy([
                    'supplierId' => $articleDetailData['supplierId'],
                    'articleDetailId' => $articleDetailData['id'],
                ]);
                if (!$articleDetailSupplierMapping) {
                    $supplier = $this->get('models')->getRepository(Supplier::class)->findOneBy([
                        'id' => $articleDetailData['supplierId'],
                    ]);
                    // Create the article mapping
                    $articleDetail = $this->get('models')->find(ArticleDetail::class, $articleDetailData['id']);
                    $articleDetailSupplierMapping = new ArticleDetailSupplierMapping($articleDetail, $supplier);
                    $articleDetailSupplierMapping->fromArray($articleDetailData);
                    $this->get('models')->persist($articleDetailSupplierMapping);
                }
                $articleDetailSupplierMapping->setPurchasePrice($articleDetailData['price']);
                $this->get('models')->flush($articleDetailSupplierMapping);
            }
            $this->View()->success = true;
        } catch (\Exception $e) {
            $this->handleException($e);
        }
    }

    /**
     * Deletes the POSTed supplier order articles.
     */
    public function deleteOrderArticlesAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Remove all given order articles
        $changedEntities = [];
        foreach ($data as $orderArticleData) {
            $orderArticle = $this->get('models')->getRepository(SupplierOrderItem::class)->findOneBy([
                'supplierOrderId' => $orderArticleData['supplierOrderId'],
                'articleDetailId' => $orderArticleData['articleDetailId'],
            ]);
            if ($orderArticle) {
                $this->get('models')->remove($orderArticle);
                $changedEntities[] = $orderArticle;
            }
        }

        // Save changes
        $this->get('models')->flush($changedEntities);

        $this->View()->success = true;
    }

    /**
     * Responds with a list containing all available supplier order statuses.
     */
    public function getOrderStatusListAction()
    {
        // Fetch all supplier order statuses from the database
        $statuses = $this->get('models')->createQueryBuilder()
            ->select('status')
            ->from(SupplierOrderStatus::class, 'status')
            ->getQuery()
            ->getArrayResult();

        $this->View()->assign([
            'success' => true,
            'data' => $statuses,
            'total' => count($statuses),
        ]);
    }

    /**
     * Responds with a list containing all available supplier order article statuses.
     */
    public function getOrderArticleStatusListAction()
    {
        // Fetch all supplier order article statuses from the database
        $statuses = $this->get('models')->createQueryBuilder()
            ->select('status')
            ->from(SupplierOrderItemStatus::class, 'status')
            ->getQuery()
            ->getArrayResult();

        $this->View()->assign([
            'success' => true,
            'data' => $statuses,
            'total' => count($statuses),
        ]);
    }

    /**
     * Creates an email containing information about the order with the given ID
     * and responds its receiver address, subject, content and attachment name.
     */
    public function getOrderMailAction()
    {
        // Try to find the order
        $orderId = $this->Request()->getParam('orderId', 0);
        $order = $this->get('models')->find(SupplierOrder::class, $orderId);
        if (!$order) {
            $this->View()->success = false;

            return;
        }

        // Create the email
        $mail = $this->createOrderMail($order);

        $this->View()->success = true;
        $this->View()->data = [
            'toAddress' => ($order->getSupplier()) ? $order->getSupplier()->getEmail() : '',
            'subject' => $mail->getSubject(),
            'content' => $mail->getPlainBodyText(), // returns plain text
            'contentHtml' => $mail->getPlainBody(), // returns html text
            'isHtml' => !empty($mail->getPlainBody()),
            'attachment' => $order->getOrderNumber() . '.pdf',
        ];
    }

    /**
     * Creates an email containing information about the order with the given ID
     * and sends it to the respective supplier using the POSTed email address,
     * subject and email body.
     */
    public function sendOrderMailAction()
    {
        // Try to find the order
        $orderId = $this->Request()->getParam('orderId', 0);
        $order = $this->get('models')->find(SupplierOrder::class, $orderId);
        if (!$order) {
            $this->View()->success = false;

            return;
        }

        // Get mail data
        $toAddress = $this->Request()->getParam('toAddress', '');
        $subject = $this->Request()->getParam('subject', '');
        $isHtmlMail = (bool) $this->Request()->getParam('isHtml', false);
        $contentHtml = $this->Request()->getParam('contentHtml', '');
        $content = $this->Request()->getParam('content', '');

        // Create the email and update it with the POSTed mail data
        $mail = $this->createOrderMail($order);
        $mail->clearRecipients();
        $mail->addTo($toAddress);
        $mail->clearSubject();
        $mail->setSubject($subject);
        if ($isHtmlMail) {
            $mail->setBodyHtml($contentHtml);
        } else {
            $mail->setBodyText($content);
        }

        // Attach the order document to the email
        $document = $this->createOrderDocument($order);
        $mail->createAttachment(
            $document->getPdf(),
            'application/pdf',
            Zend_Mime::DISPOSITION_ATTACHMENT,
            Zend_Mime::ENCODING_BASE64,
            $order->getOrderNumber() . '.pdf'
        );

        // Send mail
        $mail->send();

        // Mark the order as 'sent to supplier'
        $newOrderStatus = $this->get('models')->getRepository(SupplierOrderStatus::class)->findOneBy([
            'name' => SupplierOrderStatus::SENT_TO_SUPPLIER,
        ]);
        $order->setStatus($newOrderStatus);

        // Save changes
        $this->get('models')->flush($order);

        $this->View()->success = true;
        $this->View()->data = [
            'statusId' => $order->getStatus()->getId(),
        ];
    }

    /**
     * Creates a new supplier order document for the order with the given ID and
     * adds it to the response.
     */
    public function downloadOrderPDFAction()
    {
        // Try to find the order
        $orderId = $this->Request()->getParam('orderId', 0);
        $order = $this->get('models')->find(SupplierOrder::class, $orderId);
        if (!$order) {
            echo(sprintf('Order by ID %d not found.', $orderId));

            return;
        }

        // Create the document and add it to the response
        $snippetManager = $this->get('snippets');
        $namespace = $snippetManager->getNamespace('backend/viison_pickware_erp_supplier_orders/main');
        $document = $this->createOrderDocument($order);
        $supplierName = $this->getCleanedSupplierNameForFileName($order);
        $filename = sprintf(
            '%s_%s_%s_%s.pdf',
            $namespace->get('reorder'),
            $supplierName,
            $order->getOrderNumber(),
            $order->getCreated()->format('Y-m-d')
        );
        $document->sendPdfAsHttpResponse($this->Response(), $filename);
    }

    /**
     * Creates a new CSV files containing all articles of the order with the given ID and
     * adds it to the response.
     */
    public function downloadOrderCSVAction()
    {
        // Try to find the order
        $orderId = $this->Request()->getParam('orderId', 0);
        $order = $this->get('models')->find(SupplierOrder::class, $orderId);
        if (!$order) {
            echo(sprintf('Order by ID %d not found.', $orderId));

            return;
        }

        // Fetch correct localization sub shop
        $localizationSubShop = ($order->getSupplier()->getDocumentLocalizationSubShop()) ?: $this->get('models')->getRepository(Shop::class)->getActiveDefault();

        // Create the localised CSV header
        $csvRows = [
            $this->getCSVHeader($order)
        ];

        // Create the data rows
        foreach ($order->getItems() as $article) {
            $csvRows[] = $this->getCSVArticleData($article, $localizationSubShop);
        }

        ViisonCommonUtil::respondWithCSV($this->Response(), $csvRows, $this->getCsvAttachmentName($order));
    }

    /**
     * Returns the file name of the CSV attachment file of a supplier order.
     *
     * @param SupplierOrder $order
     * @return string
     */
    private function getCsvAttachmentName(SupplierOrder $order)
    {
        $snippetManager = $this->get('snippets');
        $namespace = $snippetManager->getNamespace('backend/viison_pickware_erp_supplier_orders/main');
        $supplierName = $this->getCleanedSupplierNameForFileName($order);

        return sprintf(
            '%s_%s_%s_%s.csv',
            $namespace->get('reorder'),
            $supplierName,
            $order->getOrderNumber(),
            $order->getCreated()->format('Y-m-d')
        );
    }

    /**
     * Returns an array containing the CSV header names localized based on the supplier's language.
     *
     * @param SupplierOrder $order
     * @return array
     */
    private function getCSVHeader(SupplierOrder $order)
    {
        // Make sure to use the suppliers locale, if available
        $snippetManager = clone $this->get('snippets');
        if ($order->getSupplier() && $order->getSupplier()->getDocumentLocalizationSubShop()) {
            $snippetManager->setShop($order->getSupplier()->getDocumentLocalizationSubShop());
        }
        $namespace = $snippetManager->getNamespace('backend/viison_pickware_erp_supplier_orders/document');

        // Get header snippets
        $fields = $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->toArray()['supplierOrderCSVFields'];
        $header = array_map(
            function ($field) use ($namespace) {
                return $namespace->get($field);
            },
            $fields
        );

        return $header;
    }

    /**
     * Returns an array with CSV data for a given article. Values may depend on shop configuration,
     * but should always match fields in CSV header.
     *
     * @param SupplierOrderItem $article
     * @param Shop $localizationSubShop
     * @return array
     */
    private function getCSVArticleData(SupplierOrderItem $article, Shop $localizationSubShop)
    {
        $articleDetail = $article->getArticleDetail();
        $units = $this->getArticleDetailUnitsIfExists($localizationSubShop, $articleDetail);
        $unitString = $units['purchaseUnit'] . $units['unit'];
        if (mb_strlen($units['packUnit']) > 0) {
            $unitString .= ' ' . $units['packUnit'];
        }

        // Prepare all article data
        $data = [
            'fabricatorNumber' => $article->getManufacturerArticleNumber(),
            'fabricator' => $article->getManufacturerName(),
            'articleName' => $article->getName(),
            'articleNumber' => $article->getOrderNumber(),
            'supplierArticleNumber' => $article->getSupplierArticleNumber(),
            'ean' => ($articleDetail) ? $articleDetail->getEan() : '',
            'unit' => $unitString,
            'quantity' => $article->getOrderedQuantity(),
            'purchasePrice' => $article->getPrice(),
            'price' => ($articleDetail) ? $articleDetail->getPrices()->first()->getPrice() : '',
            'orderNumber' => $article->getSupplierOrder()->getOrderNumber(),
            'warehouseAddress' => str_replace("\n", ', ', $article->getSupplierOrder()->getWarehouse()->getAddress()),
        ];

        // Keep only the configured data fields and sort the data in the same order as the fields
        $fields = $this->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config()->toArray()['supplierOrderCSVFields'];
        $finalData = [];
        foreach ($data as $key => $value) {
            $fieldIndex = array_search($key, $fields);
            if ($fieldIndex === false) {
                continue;
            }
            $finalData[$fieldIndex] = $value;
        }
        ksort($finalData);

        return array_values($finalData);
    }

    /**
     * Creates a new supplier order article using the given 'itemData'
     * and adds it to the given order.
     *
     * @param array $itemData The data of the new item.
     * @param SupplierOrder $order The order to which the item shall be added.
     * @return SupplierOrderItem The newly created order item.
     */
    private function addItemToOrder($itemData, SupplierOrder $order)
    {
        // Create a new item and add it to the order
        $articleDetail = $this->get('models')->find(ArticleDetail::class, $itemData['articleDetailId']);
        $supplerOrderItem = new SupplierOrderItem($order, $articleDetail);
        $this->get('models')->persist($supplerOrderItem);
        $supplerOrderItem->fromArray($itemData);

        return $supplerOrderItem;
    }

    /**
     * Creates a new supplier order email by rendering the template using the data of the given order and its supplier.
     * Uses the suppliers DocumentLocalizationSubShop or active default shop as a fallback.
     *
     * @param SupplierOrder $order
     * @return \Enlight_Components_Mail
     */
    private function createOrderMail($order)
    {
        // Get supplier data
        $supplierId = ($order->getSupplier()) ? $order->getSupplier()->getId() : 0;
        $supplierData = $this->get('models')->createQueryBuilder()
            ->select('supplier')
            ->from(Supplier::class, 'supplier')
            ->where('supplier.id = :supplierId')
            ->setParameter('supplierId', $supplierId)
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult(Query::HYDRATE_ARRAY);

        // Use supplier document localization shop, or fall back to active default shop if no such shop is set.
        $localizationShop = $order->getSupplier()->getDocumentLocalizationSubShop() ?: $this->get('models')->getRepository(Shop::class)->getActiveDefault();

        // Create the mail template
        $mail = $this->get('templatemail')->createMail(
            'viisonSupplierOrder',
            [
                'ordernumber' => $order->getOrderNumber(),
                'supplier' => $supplierData,
            ],
            $localizationShop
        );

        return $mail;
    }

    /**
     * Translates the name of the given SupplierOrderItem according to the given Shop. Shopwares "translations" are
     * handled per sub shop, not per language/locale, that's why we are using a Shop as parameter. If the
     * SupplierOrderItem is a variant article with options, these options are also translated and concatenated to the
     * name. Default (german) names are used if no translation was found for the given shop.
     *
     * @param Shop $localizationSubShop
     * @param SupplierOrderItem $orderArticle
     * @return string
     */
    private function getTranslatedArticleNameFromSupplierOrderItem(
        Shop $localizationSubShop,
        SupplierOrderItem $orderArticle
    ) {
        // Safeguard if shopware article was deleted
        if (!$orderArticle->getArticleDetail()) {
            return $orderArticle->getName();
        }
        $mainArticle = $orderArticle->getArticleDetail()->getArticle();
        $this->get('models')->refresh($mainArticle);

        $translation = $this->translator->read(
            $localizationSubShop->getId(),
            'article',
            $mainArticle->getId()
        );

        /**
         * If no translation was found, still use the article name from shopware main article, since the orderArticle
         * already has a (untranslated) variant suffix
         */
        $articleName = (!empty($translation['name'])) ? $translation['name'] : $mainArticle->getName();

        // Check if article has no variant (is single main article), or configurator options are empty
        if (!$mainArticle->getConfiguratorSet() || $orderArticle->getArticleDetail()->getConfiguratorOptions()->isEmpty()) {
            return $articleName;
        }

        // Merge translated configurator option names
        $translator = $this->translator;
        $optionStrings = $orderArticle->getArticleDetail()->getConfiguratorOptions()->map(
            function ($option) use ($localizationSubShop, $translator) {
                $translation = $translator->read(
                    $localizationSubShop->getId(),
                    'configuratoroption',
                    $option->getId()
                );

                return (!empty($translation['name'])) ? $translation['name'] : $option->getName();
            }
        );

        // Append option names to article name
        return $articleName . ' - ' . implode(' ', $optionStrings->toArray());
    }

    /**
     * Creates a new supplier order document (supplier_order.tpl) using the data
     * of the given order.
     *
     * @param SupplierOrder $order
     * @return RenderedDocumentWithDocumentType
     */
    protected function createOrderDocument(SupplierOrder $order)
    {
        /** @var PluginConfig $pluginConfig */
        $pluginConfig = $this->get('pickware.erp.plugin_config_service');
        $documentTemplate = $pluginConfig->getSupplierOrderDocumentTye();
        $localizationSubShop = ($order->getSupplier()->getDocumentLocalizationSubShop()) ?: $this->get('models')->getRepository(Shop::class)->getActiveDefault();

        // Gather all article information
        $articles = array_map(function ($article) use ($localizationSubShop) {
            $articleInformation = [
                'fabricatorNumber' => $article->getManufacturerArticleNumber(),
                'fabricator' => $article->getManufacturerName(),
                'articlenumber' => $article->getOrderNumber(),
                'supplierArticleNumber' => $article->getSupplierArticleNumber(),
                'name' => $this->getTranslatedArticleNameFromSupplierOrderItem($localizationSubShop, $article),
                'orderAmount' => $article->getOrderedQuantity(),
                // Article is an instace of SupplierOrderItem
                'article' => $article,
            ];
            // Add the three relevant unit variables to the position
            $articleInformation = array_merge(
                $articleInformation,
                $this->getArticleDetailUnitsIfExists($localizationSubShop, $article->getArticleDetail())
            );

            return $articleInformation;
        }, $order->getItems()->toArray());

        // Split the articles into multiple pages
        $pages = [];
        $pageSize = $documentTemplate->getPageBreak();
        foreach ($articles as $articleIndex => $article) {
            if (($articleIndex % $pageSize) === 0) {
                $pages[] = [];
            }
            $pages[count($pages) - 1][] = $article;
        }

        $shopConfig = $this->get('config');
        $supplier = $order->getSupplier();
        $warehouse = $order->getWarehouse();
        $this->get('models')->refresh($warehouse); // Refresh entity to avoid getting an empty proxy

        $vars = [
            /**
             * Remark: Sender information (shop info) is handled via Document/Element Header_Box_Right
             * We have to override variables even if we do not use them, since the document is initialized with default values.
             */
            'Pages' => $pages,
            'Supplier' => $supplier, // \ViisonSupplier\Supplier\Supplier
            'Warehouse' => $warehouse, // \ViisonPickwareERP\Warehouse\Warehouse
            'hasWarehouseAddress' => $warehouse->getAddress(),
            'total' => $order->getTotal(),
            'User' => [
                'billing' => [
                    'company' => '<strong>' . $supplier->getName() . '</strong>',
                    'salutation' => $supplier->getSalutation(),
                    'firstname' => '',
                    'lastname' => $supplier->getContact(), // First- and lastname of contact
                    // Pack whole address in street variable with line separator. Maybe improve supplier address-handling in the future
                    'street' => str_replace("\n", '<br>', $supplier->getAddress()),
                    'additional_address_line1' => '',
                    'additional_address_line2' => '',
                    'zipcode' => '',
                    'city' => '',
                    'customernumber' => $supplier->getCustomerNumber(),
                    'ustid' => $shopConfig->get('vatcheckadvancednumber'),
                    'state' => [
                        'shortcode' => '',
                    ],
                    'country' => [
                        'countryen' => '',
                    ],
                ],
            ],
            'Document' => [
                'id' => $order->getOrderNumber(),
                'date' => $order->getCreated()->format('d.m.Y'),
                'comment' => $order->getDocumentComment(),
            ],
            'Order' => [
                '_order' => [
                    'ordernumber' => $order->getOrderNumber(),
                    'deliverydate' => $order->getDeliveryDate() ? $order->getDeliveryDate()->format('d.m.Y') : '',
                ],
            ],
            'currency' => [
                'name' => $order->getCurrency()->getName(),
                'currency' => $order->getCurrency()->getCurrency(),
                'symbol' => $order->getCurrency()->getSymbol(),
                'symbolOnLeft' => CurrencyUtil::isCurrencySymbolOnLeft($order->getCurrency()),
            ],
            /**
             * Supplier Order custom variables
             */
            'hasContactInformation' => $supplier->getEmail()|| $supplier->getPhone() || $supplier->getFax(),
            'email' => $supplier->getEmail(),
            'phone' => $supplier->getPhone(),
            'fax' => $supplier->getFax(),
        ];

        /** @var DocumentRenderingContextFactoryService $factory */
        $factory = $this->get('viison_common.document_rendering_context_factory');
        $renderingContext = $factory->createDocumentRenderingContext(
            $documentTemplate,
            PaperLayout::createDefaultPaperLayout('A4'),
            $localizationSubShop->getLocale(),
            $localizationSubShop,
            DocumentRenderingContextFactoryService::RENDERER_MPDF
        );
        $renderingContext->assignTemplateVar($vars);

        return $renderingContext->renderDocument();
    }

    /**
     * Returns the unit information of a given article detail in a preformatted associative array. It contains the
     * fields purchaseUnit, unit and packUnit. (e.g. "700", "ml" and "Flasche(n)")
     *
     * @param Shop $localizationSubShop
     * @param ArticleDetail|null $articleDetail
     * @return array
     */
    private function getArticleDetailUnitsIfExists(Shop $localizationSubShop, ArticleDetail $articleDetail = null)
    {
        $units = [
            'purchaseUnit' => '',
            'unit' => '',
            'packUnit' => '',
        ];

        // In case the respective article detail of the order position was deleted
        if (!$articleDetail) {
            return $units;
        }

        $language = $localizationSubShop->getLocale(); // Remark: refresh to avoid empty proxy
        $this->get('models')->refresh($language);
        $locale = $language->getLocale();
        $purchaseUnit = ViisonCommonUtil::localizedNumberFormat(floatval($articleDetail->getPurchaseUnit()), $locale); // e.g. "0.3500", floatval() removed unneeded zeros
        $unit = $articleDetail->getUnit(); // e.g. "L"

        if ($purchaseUnit !== '0') {
            $units['purchaseUnit'] = $purchaseUnit;
            if ($unit) {
                $unitTranslation = $this->translator->read(
                    $localizationSubShop->getId(),
                    'config_units'
                );
                $units['unit'] = $unit->getUnit();
                if (!empty($unitTranslation[$unit->getId()]['unit'])) {
                    $units['unit'] = $unitTranslation[$unit->getId()]['unit'];
                }
            }
        }
        if ($articleDetail->getPackUnit() !== '') {
            $translation = $this->translator->read(
                $localizationSubShop->getId(),
                'article',
                $articleDetail->getArticle()->getId()
            );
            $units['packUnit']  = $articleDetail->getPackUnit();
            if (!empty($translation['packUnit'])) {
                $units['packUnit'] = $translation['packUnit'];
            }
        }

        return $units;
    }

    /**
     * Tries to find the supplier order status with the given name.
     *
     * @param string orderStatusName
     * @return SupplierOrderStatus
     */
    private function findOrderStatusByName($orderStatusName)
    {
        return $this->get('models')->getRepository(SupplierOrderStatus::class)->findOneBy([
            'name' => $orderStatusName,
        ]);
    }

    /**
     * Tries to find the supplier order article status with the given name.
     *
     * @param string orderArticleStatusName
     * @return SupplierOrderItemStatus
     */
    private function findSupplierOrderItemStatusByName($orderArticleStatusName)
    {
        return $this->get('models')->getRepository(SupplierOrderItemStatus::class)->findOneBy([
            'name' => $orderArticleStatusName,
        ]);
    }

    /**
     * Determines the next order number by reading the lastly created number
     * from the database and increasing it by one. The new number is also written
     * back to the database.
     *
     * @return string
     */
    private function getNextOrderNumber()
    {
        // Get next order number
        $lastNumber = $this->get('db')->fetchOne(
            'SELECT MAX(number)
            FROM s_order_number
            WHERE name = \'viison_supplier_order\''
        );
        $number = $lastNumber + 1;

        // Update the last used order number
        $this->get('db')->query(
            'UPDATE s_order_number
            SET number = ?
            WHERE name = \'viison_supplier_order\'',
            [
                $number
            ]
        );

        return $number;
    }

    /**
     * Determines the number of days since the given date.
     *
     * @param \DateTime $date
     * @return int
     */
    private static function getDaysSince(DateTime $date)
    {
        $date = new DateTime($date->format('Y-m-d'));
        $now = new DateTime(date('Y-m-d'));

        return $now->diff($date)->format('%a');
    }

    /**
     * Returns a "cleaned" supplier name of the given supplier order that can be used for the file name of a download.
     * (i.e. removes commas that would cause a response header error in chrome)
     *
     * @param SupplierOrder $order
     * @return string
     */
    private function getCleanedSupplierNameForFileName(SupplierOrder $order)
    {
        if (!$order->getSupplier() || !$order->getSupplier()->getName()) {
            return '';
        }

        return str_replace(',', '', $order->getSupplier()->getName());
    }
}
